package moos.operations.portal.auvctd;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Properties;
import java.util.ResourceBundle;
import java.util.StringTokenizer;

import moos.operations.portal.ssds.IXMLFactory;
import moos.ssds.model.DataFile;
import moos.ssds.model.Deployment;
import moos.ssds.model.Device;
import moos.ssds.model.HeaderDescription;
import moos.ssds.model.Person;
import moos.ssds.model.RecordDescription;
import moos.ssds.model.RecordVariable;
import moos.ssds.model.Resource;
import moos.ssds.model.XmlBuilder;

/**
 * <h2><u>Description</u></h2>
 * <p><!--Insert summary here--></p>
 *
 * <h2><u>UML</u></h2>
 * <pre>
 *
 *   [AUVXMLFactory]
 *        |
 *        |
 *        V 1
 *   [XmlBuilder]
 *
 * </pre>
 *
 * <h2><u>License</u></h2>
 * <p><font size="-1" color="#336699"><a href="http://www.mbari.org">
 * The Monterey Bay Aquarium Research Institute (MBARI)</a> provides this
 * documentation and code &quot;as is&quot;, with no warranty, express or
 * implied, of its quality or consistency. It is provided without support and
 * without obligation on the part of MBARI to assist in its use, correction,
 * modification, or enhancement. This information should not be published or
 * distributed to third parties without specific written permission from
 * MBARI.</font></p>
 *
 * <p><font size="-1" color="#336699">Copyright 2003 MBARI.
 * MBARI Proprietary Information. All rights reserved.</font></p>
 *
 *@author     <a href="mailto:brian@mbari.org">Brian Schlining</a>
 *@version    $Id: AUVXMLFactory.java,v 1.11 2003/08/05 21:00:01 brian Exp $
 *@since      Jun 5, 2003 11:00:22 AM
 */
public class AUVXMLFactory implements IXMLFactory {

    /**
     *@param  localDir The local mission directory (ex. ./AUVCTD2003191/2003.191.03)
     *@param  remoteUrl The base URL of the mission archive 
     *        (http://prey.shore.mbari.org/auvctd/AUVCTD2003191/2003.191.03)
     */
    public AUVXMLFactory(File localDir, URL remoteUrl) {
        this.localDir = localDir;
        this.remoteUrl = remoteUrl;
        this.devicesCfg = new File(localDir, "devices.cfg");

        try {

            // Load the vehicle.cfg file as properties
            properties.load(new FileInputStream(new File(localDir, "vehicle.cfg")));

            // Make the AUV "Platform" deployment. Everything else hangs off of this.
            Deployment dp = makePlatformDeployment();
            xmlBuilder.addDeployment(dp);

            // Get all the log (aka Data) files
            File[] logs = getLogs();

            // Get all the resources into a collection. (We need this later)
            File[] resrc = getResources();
            for (int i = 0; i < resrc.length; i++) {
                resources.add(resrc[i]);
            }

            // Generate an instrument deployment description for each log file
            for (int i = 0; i < logs.length; i++) {
                File log = logs[i];
                Deployment di = makeInstrumentDeployment(log);
                Device dvi = di.getDevice();
                if (dvi != null) {
                    if (dvi.getId() != null) {
                        dp.addChildDeployment(di);
                    }

                }

            }

            // add any resources that are not associate with an instrument to 
            // the platform
            for (Iterator i = resources.iterator(); i.hasNext();) {
                File f = (File) i.next();
                Resource r = new Resource();
                r.setName(f.getName());
                r.setDescription("Undescribed resource");
                r.setUrl(makeURL(f));
                dp.addResource(r);
            }

            xmlBuilder.marshal();
            System.out.println("Marshalled the AUV mission in " + localDir.getAbsolutePath() + " into XML");
        }
        catch (Exception e) {
            System.out.println(e.getClass().getName() + ": " + e.getMessage());
        }
    }

    /**
     * Prints the XML to the Console
     * 
     * @since Jul 10, 2003
     */
    public void print() {
        try {
            xmlBuilder.print();
        }
        catch (UnsupportedEncodingException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
    }

    /**
     *  Gets the xML attribute of the AUVXMLFactory object
     *
     *@return    The xML value
     */
    public String getXML() {
        return xmlBuilder.toXML();
    }

    /**
     *  Gets the xMLAsByteArray attribute of the AUVXMLFactory object
     *
     *@return    The xMLAsByteArray value
     */
    public byte[] getXMLAsByteArray() {
        ByteArrayOutputStream buf = new ByteArrayOutputStream();
        try {
            xmlBuilder.print(buf);
        }
        catch (UnsupportedEncodingException e) {
            // This should never be throw. But if its we'd better catch it
            System.err.println(e.getClass().getName() + ": Encoding is not supported");
        }
        catch (IOException e) {
            // This should never be throw. But if its we'd better catch it
            System.err.println(e.getClass().getName() + ": Unable to write to non-file system byte array");
        }
        return buf.toByteArray();
    }

    /**
     * USed by XMLPublisher to assign a device ID to the SIAM packet sent to 
     * SSDS. For AUVs this will be the AUVs ID.
     * 
     */
    public long getDeviceId() {
        long id = -1;
        try {
            id = getPlatformId().longValue();
        }
        catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        return id;
    }

    /**
     * Retrieves device ids from devices.cfg. It seraches for a line with the
     * format # deviceID:[logfilename]=id1, id2, id3, etc.
     * @param log
     * @return Collection where the first item is the instrument ID and the 
     *         subsequent items are sensorIDs
     * @since Jul 10, 2003
     */
    public Collection getDeviceIds(File log) {
        Collection ids = new ArrayList();
        try {
            String line = findString(devicesCfg, "deviceID: " + log.getName());
            //System.out.println(line);
            if (line != null) {

                int i = line.indexOf('=');
                StringTokenizer st = new StringTokenizer(line.substring(i + 1).trim(), ",");
                while (st.hasMoreTokens()) {
                    ids.add(Long.valueOf(st.nextToken()));
                }
            }
        }
        catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        return ids;
    }

    /**
     *  Gets the deviceId attribute of the AUVXMLFactory object
     *
     *@return    The deviceId value
     */
    private Long getInstrumentId(File log) {
        Collection ids = getDeviceIds(log);
        Iterator iterator = ids.iterator();
        return (Long) iterator.next();
    }

    private Long getPlatformId() throws IOException {
        // Locate the ilne that containe platformID (case insensitive search)
        // It's form is "# platformID:102
        if (platformId == null) {
            String line = findString(devicesCfg, "platformid:");
            if (line != null) {
                int i = line.indexOf(':');
                platformId = Long.valueOf(line.substring(i + 1).trim());
            }
        }
        return platformId;
    }

    /**
     * Search through a file of locate the first occurence of a String
     * @param file
     * @param searchString
     * @return The line first line in a file containing an occurence of the search string.
     *         <b>null</b> is returned if the serach string is not found.
     * @throws IOException
     * @since Jul 10, 2003
     */
    private String findString(File file, String searchString) throws IOException {
        //System.out.println("Searching " + file + " for " + searchString);
        BufferedReader in = new BufferedReader(new FileReader(file));
        String line = null;
        String lcLine = null;
        String out = null;
        int i = -1;
        searchString = searchString.toLowerCase();
        while ((line = in.readLine()) != null) {
            lcLine = line.toLowerCase();
            i = lcLine.indexOf(searchString);
            if (i > -1) {
                out = line;
                break;
            }
        }
        in.close();
        return out;
    }

    /**
     *@return    All the log files in a given directory
     */
    private File[] getLogs() {

        FilenameFilter filter = new FilenameFilter() {
            public boolean accept(File someDir, String someName) {
                return someName.endsWith(".log");
            }
        };

        return localDir.listFiles(filter);
    }

    /**
     *@return    All the log files in a given directory
     */
    private File[] getResources() {

        FilenameFilter filter = new FilenameFilter() {
            public boolean accept(File someDir, String someName) {
                return (!someName.endsWith(".log") && !someName.endsWith(".m"));
            }
        };

        return localDir.listFiles(filter);
    }

    /**
     * Generates a deployment description (object model) for a log file in
     * an AUV deployment
     *
     *@param  log                     The log (i.e auv data) file to create a description for
     *@return                         A 'fleshed-out' deployment object
     *@throws  MalformedURLException
     *@throws  FileNotFoundException
     *@throws  IOException
     *@since                          Apr 15, 2003
     */
    private Deployment makeInstrumentDeployment(File log)
        throws MalformedURLException, FileNotFoundException, IOException {

        // If a file has the same name but different extension as a log file 
        // add it as a resource to the deployment
        String logName = log.getName();
        int ext = log.getName().indexOf('.');
        String logShortName = logName.substring(0, ext);

        // Describe the instrument deployment
        Deployment d = new Deployment();
        d.setRole(Deployment.INSTRUMENT_ROLE);
        d.setName(logShortName); 
        d.setDescription("Deployment information for " + log.getName());

        for (Iterator i = resources.iterator(); i.hasNext();) {
            File f = (File) i.next();
            String fName = f.getName();
            if (fName.startsWith(logShortName)) {
                Resource r = new Resource();
                r.setName(f.getName());
                r.setDescription("Undescribed resource");
                r.setUrl(makeURL(f));
                d.addResource(r);
                i.remove();
            }
        }

        // Get instrument and sensor IDs
        Collection ids = getDeviceIds(log);
        Iterator iterator = ids.iterator();

        try {

            // Get device id from devices.cfg
            Device inst = new Device((Long) iterator.next());
            inst.setName(logShortName);
            d.setDevice(inst);

            Deployment sensorDeployment = null;
            Device sensor = null;
            while (iterator.hasNext()) {
                sensor = new Device((Long) iterator.next());

                sensorDeployment = new Deployment(sensor);
                sensorDeployment.setRole(Deployment.SENSOR_ROLE);
                d.addChildDeployment(sensorDeployment);
            }
        }
        catch (Exception e) {
            System.out.println("WARNING!! Device information is missing from " + log);
            return d;
        }

        // Log files are DataFiles. They contain URL's. Here the URL is local
        // but we shoudl really use web URL's (http://blah.com). Need to map ploarbear onto
        // an internal server to do this.
        DataFile out = new DataFile();
        out.setName(log.getName());
        out.setDescription("AUV data");
        out.setUrl(makeURL(log));
        out.setOriginal(true);
        d.addOutput(out);

        // Tell SSDS how to ignore headers on the log files
        // AUV logs always used "#" as a comment tag
        HeaderDescription hd = new HeaderDescription();
        hd.addCommentTag("#");
        out.setHeaderDescription(hd);

        // All log files have the same record description. Don't need
        // to change this
        RecordDescription rd = new RecordDescription();
        rd.setBufferStyle(RecordDescription.BINARY_BUFFERSTYLE);
        rd.setBufferLengthType(RecordDescription.FIXED_BUFFERLENGTH);
        rd.setEndian(RecordDescription.LITTLE_ENDIAN);
        rd.setParseable(true);
        out.setRecordDescription(rd);

        // Read the header of the log and use the info to describe the RecordVariables
        BufferedReader in = new BufferedReader(new FileReader(log));
        String line = null;
        line = in.readLine();
        int i = 0;
        RecordVariable v = null;
        StringTokenizer st = null;
        String format = null;
        String units = null;
        String longName = null;
        String name = null;
        String cFmt = null;
        line = in.readLine();
        while ((line != null) && (!line.startsWith("# begin"))) {
            i++;
            //System.out.println(getClass().getName() + ": line = " + line);
            st = new StringTokenizer(line);

            // Ignore the #
            st.nextToken();

            // All files shoudl hav the following
            format = st.nextToken().trim();
            name = st.nextToken().trim();
            cFmt = st.nextToken(",").trim();

            // Newer fiels have longNmes and units but we need to
            // keep backward compatibility with older files.
            try {
                longName = st.nextToken(",").trim();
            }
            catch (Exception e) {
                longName = name;
            }
            try {
                units = st.nextToken().trim();
            }
            catch (Exception e1) {
                units = "UNKNOWN";
            }

            // Convert weird formats to known formats. 
            if (format.equals("timeTag")) {
                format = "double";
                //              TODO 20030606 brian - When the AUV guys include the corecto metadata get rid of these 2 lines
                units = "seconds since 1970-01-01 00:00:00";

                longName = "Time (GMT)";
            }
            else if (format.equals("angle")) {
                format = "double";
            }

            v = new RecordVariable();
            v.setFormat(format);
            v.setName(name);
            v.setLongName(longName);
            v.setUnits(units);
            v.setColumnIndex(i);
            rd.addRecordVariable(v);
            line = in.readLine();

        }
        in.close();
        return d;
    }

    /**
     * Each mission should have a contact to send a notification to. Ideally, this would be the
     * auv mailing list so that everyone knows when a mission is completed.
     *
     *@return
     *@since     Apr 15, 2003
     */
    private Person makeMissionContact() {
        // TODO 20030717 brian: The contact info should be parsed out of the
        //      auvportal.properties file
        ResourceBundle rb = ResourceBundle.getBundle("auvportal");
        Person person = new Person();
        person.setFirstname(rb.getString("contact.firstname"));
        person.setSurname(rb.getString("contact.surname"));
        person.setEmail(rb.getString("contact.email"));
        return person;
    }

    /**
     * Make an empty AUV deployment. It still needs to have the log info added
     *
     *@return                         A mock-up of the AUV deployment
     *@throws  MalformedURLException
     *@since                          Apr 15, 2003
     */
    private Deployment makePlatformDeployment() throws IOException {

        // Create a Ploatform deployment for the AUV
        Deployment d = new Deployment();
        d.setRole(Deployment.PLATFORM_ROLE);
        d.setName(localDir.getName());
        d.setDescription(
            "AUV mission generated from information in the directory "
                + localDir.getName()
                + ". WARNING!!! Metadata in missing from he mission directory.");
        d.setStartDate(new Date(0));
        d.setEndDate(new Date());
        d.setContact(makeMissionContact());

        // Describe the AUV as a Device
        Long auvId = getPlatformId();
        if (auvId == null) {
            throw new IOException(
                devicesCfg.getAbsolutePath()
                    + " does not contain a platform ID"
                    + " (Searched for a line starting with '# platformID:')");
        }
        Device auv = new Device(getPlatformId());
        auv.setDescription("Dorado AUV");
        auv.setName(properties.getProperty("vehicleName"));
        auv.setMfgName("MBARI");
        d.setDevice(auv);

        return d;
    }

    /**
     * Generate the correct URL
     *
     *@param  file
     *@return
     *@throws  MalformedURLException
     *@since                          May 6, 2003
     */
    private URL makeURL(File file) throws MalformedURLException {
        String path = remoteUrl.getFile();
        if (!path.endsWith("/")) {
            path = path + "/";
        }
        path = path + file.getName();
        return new URL(remoteUrl.getProtocol(), remoteUrl.getHost(), path);
    }

    public static void main(String args[]) throws Exception {
        AUVXMLFactory f = new AUVXMLFactory(new File(args[0]), new URL(args[1]));
        f.print();
    }

    /**
     * The directory containing the ALL the auv logs and cfg files
     */
    private File localDir;

    /**
     *  Properties that are obtained by loading the vehicle.cfg file of an AUV mission
     */
    private Properties properties = new Properties();

    /**
     * The web URL that corresponds to the local directory. For Example, if the
     * data is stored on Tornado:/AUVCTD/mymission then the web URL to access that directory
     * is http://prey.shore.mbari.org/auvctd/mymission
     */
    private URL remoteUrl;

    /**
     *  The class that converts the object graph to XML
     */
    private XmlBuilder xmlBuilder = new XmlBuilder();

    private File devicesCfg;

    private Long platformId;

    /**
     * key = a File pointing to a resource, 
     * value = Boolean (false = not added to deployment, true = added to deployment)
     */
    private Collection resources = new HashSet();

}
