package moos.operations.portal;

import com.enterprisedt.net.ftp.FTPClient;
import com.enterprisedt.net.ftp.FTPException;
import com.enterprisedt.net.ftp.FTPTransferType;
import com.ice.tar.TarArchive;
import com.ice.tar.TarEntry;
import com.ice.tar.TarInputStream;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.URL;
import java.util.ResourceBundle;
import java.util.StringTokenizer;
import java.util.zip.GZIPInputStream;

import moos.operations.portal.auvctd.AUVXMLFactory;
import moos.operations.portal.ssds.XMLPublisher;

/**
 * <h2><u>Description</u></h2>
 * <p>AUVPortalService provides a "bridge" between shore/shipboard networks
 * and low-bandwidth links to AUVs. </p>
 *
 * <h2><u>UML</u></h2>
 * <pre>
 *                       1
 *    [AUVPortalServer]-->[ServerSocket]
 *          |
 *          |
 *          V 1       1
 *    [XMLPublisher]-->[AUVXMLFactory]
 *
 * </pre>
 *
 * <h2><u>License</u></h2>
 * <p><font size="-1" color="#336699"><a href="http://www.mbari.org">
 * The Monterey Bay Aquarium Research Institute (MBARI)</a> provides this
 * documentation and code &quot;as is&quot;, with no warranty, express or
 * implied, of its quality or consistency. It is provided without support and
 * without obligation on the part of MBARI to assist in its use, correction,
 * modification, or enhancement. This information should not be published or
 * distributed to third parties without specific written permission from
 * MBARI.</font></p>
 *
 * <p><font size="-1" color="#336699">Copyright 2003 MBARI.
 * MBARI Proprietary Information. All rights reserved.</font></p>
 *
 * @author  Tom O'Reilly, Brian Schlining
 * @version  $Id: AUVPortalService.java,v 1.6 2003/08/05 21:34:43 brian Exp $
 * @since  Jun 5, 2003 10:59:14 AM
 */
public class AUVPortalService {

    /**
     * @param  port
     * @exception  IOException So many reasons to toss an IOException.
     * @throws  IOException
     */
    public AUVPortalService(int port) throws IOException {
        _serverSocket = new ServerSocket(port);
        initPortal();
    }

    /**
     * This consturctor si used to skip attempting to contact the AUV.
     * It just copies the data from the localhost to the
     * repository location and notifies SSDS.
     *
     * @param  survey Description of the Parameter
     * @param  file Description of the Parameter
     * @exception  IOException Description of the Exception
     * @throws  IOException
     */
    public AUVPortalService(String survey, String file) throws IOException {
        initPortal();
        WorkerThread workerThread = new WorkerThread2(survey + ";" + file);
        workerThread.start();
    }

    /**
     * Wait for and process new client connections.
     */
    public void run() {
        Socket clientSocket = null;
        if (_serverSocket != null) {

            while (true) {
                try {
                    System.out.println("Wait for client...");
                    clientSocket = _serverSocket.accept();
                    System.out.println("accepted connection");
                    processConnection(clientSocket);
                }
                catch (IOException e) {
                    System.err.println("IOException: " + e.getMessage());
                }
            }
        }

    }

    /**
     * Read and parse message from AUV, then retrieve files from AUV
     * via ftp.
     *
     * @param  clientSocket The socket connection to the AUV
     * @exception  IOException Lots of IO occurs here. Catch any excpetions
     */
    protected void processConnection(Socket clientSocket) throws IOException {

        byte buf[] = new byte[1024];

        InputStream in = clientSocket.getInputStream();
        int nBytes = 0;
        // Read message from client
        while (true) {
            int c = in.read();
            if (c == -1) {
                System.err.println("processConnection() - end-of-stream");
                break;
            }
            buf[nBytes++] = (byte) c;
        }

        String message = new String(buf);

        WorkerThread workerThread = new WorkerThread(message);
        workerThread.start();

    }

    /**
     *  Reads initialization parameters for a properties file.
     *
     * @exception  IOException Problem reading properties file
     */
    private void initPortal() throws IOException {

        // load the properties from auvportal.properties
        ResourceBundle rb = ResourceBundle.getBundle("auvportal");

        // The localDir is the directory on the portal machine to copy data into.
        // It's alwasy the directory that the JVM isstarted from. THe FTP library
        // dosn't provide any methdos to switch the local directory.
        localDir = new File(System.getProperty("user.dir"));

        // The remote directory is the location of the AUV archive. It's only the
        // base, not the full path (i.e it should be \\Tornado\AUVCTD on windows)
        remoteDir = new File(rb.getString("remote.path"));

        // The remote URL is the web server URL that points to 'remoteDir'
        remoteUrl = new URL(rb.getString("remote.url"));

        // Of course we need a JMS topic to contact SSDS
        jmsTopic = rb.getString("jms.topic");

        xmlPublisher = new XMLPublisher(jmsTopic);
    }

    /**
     *  The main program for the AUVPortalService class
     *
     * @param  args The command line arguments
     */
    public static void main(String[] args) {
        try {
            if (args.length == 0) {
                AUVPortalService service = new AUVPortalService(4444);
                service.run();
            }
            else if (args.length == 1) {
                AUVPortalService service = new AUVPortalService(Integer.parseInt(args[0]));
                service.run();
            }
            else if (args.length == 2) {
                AUVPortalService service = new AUVPortalService(args[0], args[1]);
            }

        }
        catch (IOException e) {
            System.err.println("Caught IOException: " + e.getMessage());
        }
    }

    /**
     * Subclass of workerThread. This thread does not contact the AUV, it proceses
     * tar fiels that have already been download.
     *
     * @author  brian
     * @version
     * @created  July 14, 2003
     */
    public class WorkerThread2 extends WorkerThread {
        /**
         * @param  message
         */
        public WorkerThread2(String message) {
            super(message);
        }

        /**
         * Process message received from AUV.
         */
        public void run() {

            System.out.println("processConnection() got message:\n" + _message);
            // Parse message
            StringTokenizer tokenizer = new StringTokenizer(_message, ";");
            if (tokenizer.countTokens() < 2) {
                System.err.println("Not enough tokens in message:\n" + _message);
                return;
            }
            // Survey is used for grouping
            String survey = null;
            String file = null;
            int nToken = 0;
            while (tokenizer.hasMoreTokens()) {
                String token = tokenizer.nextToken();
                switch (nToken++) {
                    case 0 :
                        survey = token;
                        break;
                    default :
                        file = token;
                        try {
                            unpackTarball(survey, file);
                            // Peeks into the tarball to get mission name
                            String missionName = getMissionName(file);
                            copyToArchive(survey, missionName);
                            notifySSDS(survey, missionName);
                        }
                        catch (Exception e) {
                            System.out.println(e.getClass().getName() + ": " + e.getMessage());
                            e.printStackTrace();
                        }
                }
            }

        }
    }

    /**
     * Responsible for processing contact from AUV. Retrieves data
     * from AUV, processes metadata, and notifies SSDS.
     *
     * @author  brian
     * @version
     * @created  July 14, 2003
     */
    class WorkerThread extends Thread {

        /**
         *Constructor for the WorkerThread object
         *
         * @param  message Description of the Parameter
         */
        public WorkerThread(String message) {
            _message = message;
        }

        /**
         * Process message received from AUV.
         */
        public void run() {

            System.out.println("processConnection() got message:\n" + _message);

            // Parse message
            StringTokenizer tokenizer = new StringTokenizer(_message, ";");
            if (tokenizer.countTokens() < 5) {
                System.err.println("Not enough tokens in message:\n" + _message);
                return;
            }

            String auvHost = null;
            String login = null;
            String passwd = null;
            // Survey is used for grouping
            String survey = null;
            // This is the full path of the file on the AUV
            String file = null;

            int nToken = 0;
            while (tokenizer.hasMoreTokens()) {
                String token = tokenizer.nextToken();

                switch (nToken++) {
                    case 0 :
                        auvHost = token;
                        break;
                    case 1 :
                        login = token;
                        break;
                    case 2 :
                        passwd = token;
                        break;
                    case 3 :
                        survey = token;
                        break;
                    default :
                        // We had probleems with extra ine feeds. trim() fixed it.
                        file = token.trim();
                        try {
                            retrieveFromAUV(auvHost, login, passwd, file);

                            // Get the name (no path) of the file
                            File remoteFile = new File(file);
                            String fileName = remoteFile.getName();

                            unpackTarball(survey, fileName);
                            // Peeks into the tarball to get mission name
                            String missionName = getMissionName(fileName);
                            copyToArchive(survey, missionName);
                            notifySSDS(survey, missionName);
                        }
                        catch (Exception e) {
                            System.out.println(e.getClass().getName() + ": " + e.getMessage());
                        }
                }
            }

        }

        /**
         *  Copy a file to a new location
         *
         * @param  src The source location
         * @param  dst The dstination location
         * @exception  IOException Need I explain?
         */
        void copy(File src, File dst) throws IOException {
            BufferedInputStream in = new BufferedInputStream(new FileInputStream(src));
            BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(dst));
            int b = 0;
            while ((b = in.read()) > -1) {
                out.write(b);
            }
            in.close();
            out.close();
        }

        /**
         *  Copy a mission to the archive repository (i.e. tornado:/AUVCTD)
         *
         * @param  survey The name of the survey (ex. AUVCTD2003191). A
         *         survey is a logical grouping of missions
         * @param  mission The name of the mission (ex. 2003.191.03)
         * @exception  IOException IO happens. Catch the bad stuff.
         */
        void copyToArchive(String survey, String mission) throws IOException {

            File localMissionDir = new File(new File(localDir, survey), mission);
            File remoteMissionDir = new File(new File(remoteDir, survey), mission);
            if (!remoteMissionDir.exists()) {

                if (!remoteMissionDir.mkdirs()) {
                    throw new IOException(
                        "Unable to create the archive directory, " + remoteMissionDir.getAbsolutePath());
                }
            }

            File[] files = localMissionDir.listFiles();
            File dst = null;
            File src = null;
            for (int i = 0; i < files.length; i++) {
                src = files[i];
                dst = new File(remoteMissionDir, src.getName());
                System.out.print("Copying " + src + " to " + dst);
                if (dst.exists()) {
                    System.out.println(" [WARNING!! Overwriting existing file]");
                }
                else {
                    System.out.println();
                }

                copy(src, dst);
            }
        }

        /**
         * Open the tarball and find the mission name. The mission name should
         * be the only directory in the tarball.
         *
         * @param  tarballName The name of the tarball
         * @return  The name of the mission (i.e 2003.191.03). null if none was found.
         * @throws  IOException
         */
        String getMissionName(String tarballName) throws IOException {

            File tarFile = new File(localDir, tarballName);
            InputStream in = new FileInputStream(tarFile);
            //          Check for compression
            if (tarballName.endsWith(".gz") || tarballName.endsWith(".tgz")) {
                in = new GZIPInputStream(in);
            }
            TarInputStream tis = new TarInputStream(in);
            TarEntry te = tis.getNextEntry();
            while (te != null && !te.isDirectory()) {
                te = tis.getNextEntry();
            }
            tis.close();

            String missionName = null;
            if (te != null) {
                missionName = te.getName();
            }
            else {
                throw new IOException(
                    "The file, " + tarFile.getAbsolutePath() + "does not contain a mission directory.");
            }

            return missionName;
        }

        /**
         *  parse the metadata from the AUV mission files and submit them to
         * SSDS
         *
         * @param  survey Description of the Parameter
         * @param  mission Description of the Parameter
         * @exception  Exception Thrown if unable to establish a JMS connection
         *             to SSDS.
         */
        void notifySSDS(String survey, String mission) throws Exception {
            if (xmlPublisher.isConnected()) {
                File missionDir = new File(new File(remoteDir, survey), mission);
                URL missionUrl =
                    new URL(
                        remoteUrl.getProtocol(),
                        remoteUrl.getHost(),
                        remoteUrl.getPort(),
                        remoteUrl.getPath() + "/" + survey + "/" + mission);
                System.out.println(missionUrl);
                AUVXMLFactory xmlFactory = new AUVXMLFactory(missionDir, missionUrl);
                synchronized (xmlPublisher) {
                    xmlPublisher.setXmlFactory(xmlFactory);
                    System.out.println("Attempting to publish " + missionDir.getAbsolutePath() + " to SSDS.");
                    xmlPublisher.publish();
                    System.out.println(" SUCCESS!! published " + missionDir.getAbsolutePath() + " to SSDS");
                    //xmlFactory.print();
                }

            }

            else {
                throw new Exception("FAILED!! Unable to contact SSDS.");
            }
        }

        /**
         *  FTP the files from the AUV to the localhost.
         *
         * @param  auvHost The host address
         * @param  login FTP login
         * @param  passwd FTP password
         * @param  fileName name of the tarball to copy from the AUV
         * @exception  IOException IO happens in this method. Catch bad stuff.
         * @exception  FTPException Problem with FTP connection
         */
        void retrieveFromAUV(String auvHost, String login, String passwd, String fileName)
            throws IOException, FTPException {
            // Retrieve file via ftp...
            System.out.println("Retrieve mission files from AUV host:");
            System.out.println(auvHost + ": " + fileName);
            // Use the directory specified by auvportal.properties
            File remoteFile = new File(fileName);
            File file = new File(localDir, remoteFile.getName());
            FTPClient ftpClient = new FTPClient(auvHost);
            ftpClient.user(login);
            ftpClient.password(passwd);
            ftpClient.setType(FTPTransferType.BINARY);
            if (file.exists()) {
                System.out.println("Retrieving " + fileName + " [WARNING!! Overwriting existing file]");
            }
            else {
                System.out.println("Retrieving " + fileName);
            }
            ftpClient.get(file.getName(), fileName.trim());
            // All done
            System.out.println("All done");
            ftpClient.quit();
        }

        /**
         * The tarball will have a <u>single</u> mission directory that will need to
         * be unpacked, such as:
         *     2003.191.00
         *     |~~ vehicle.cfg
         *     |~~ ctdDriver.log
         *     `~~ (etc)
         *
         * The tarball is dowloaded into the localDir (see auvportal.properties).
         * Then extracted into localDir/surveyname.
         *
         * @param  survey The name of the AUV sruvey (AUVCTD2003191)
         * @param  tarballName The name of the tarfile (somedata.tar.gz)
         * @exception  IOException Thrown if problem unpakcing the tarball.
         */
        void unpackTarball(String survey, String tarballName) throws IOException {
            // The ultimate destination of the unpacked archvie
            File dstDir = new File(localDir, survey);
            if (!dstDir.exists()) {
                if (!dstDir.mkdirs()) {
                    throw new IOException("Unable to create the directory " + dstDir.getAbsolutePath());
                }
            }
            // The location of the current tarball
            File tarFile = new File(localDir, tarballName);
            InputStream in = new FileInputStream(tarFile);
            // Check for compression
            if (tarballName.endsWith(".gz") || tarballName.endsWith(".tgz")) {
                in = new GZIPInputStream(in);
            }

            System.out.println("Unpacking " + tarFile.getAbsolutePath());
            // Extract tar, overwriting existing files.
            TarArchive archive = new TarArchive(in);
            archive.setKeepOldFiles(false);
            archive.extractContents(dstDir);
            archive.closeArchive();
        }

        /**
         * Message received from AUV.
         */
        protected String _message;

    }

    /**
     *  Conncetion to computer connection tot he portal
     */
    protected ServerSocket _serverSocket;
    /**
     *  The JMS topic used to send metadata to SSDS
     */
    private String jmsTopic;
    /**
     *  The directory on the localhost to download AUV data into
     */
    private File localDir;
    /**
     *  The directory on the archive host to download AUV data into
     */
    private File remoteDir;
    /**
     *  The web URL thats mapped to remoteDir
     */
    private URL remoteUrl;

    private XMLPublisher xmlPublisher;

}
