package example;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FilenameFilter;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Date;
import java.util.Properties;
import java.util.StringTokenizer;

import moos.ssds.model.Deployment;
import moos.ssds.model.Device;
import moos.ssds.model.DataFile;
import moos.ssds.model.HeaderDescription;
import moos.ssds.model.Person;
import moos.ssds.model.RecordDescription;
import moos.ssds.model.Resource;
import moos.ssds.model.RecordVariable;
import moos.ssds.model.XmlBuilder;

/**
 * <p>Generates a mock-up xml description for an AUV deployment</p>
 * Use as:
 * <pre>
 *  XMLMockup xmlMockup = new XMLMockup(dir);
 *  XmlBuilder xmlBuilder = xmlMockup.getXmlBuilder();
 *  xmlBuilder.marshall() // creates the XML;
 *  xmlBuilder.print()    // prints to screen
 *  // or xmlBuilder.print(someOutputStream) // writes XML to a Stream
 * </pre>
 * <p><font color="FFAAAA">Note: The pattern "****" indicates that XML data is missing from
 * the AUV metadata.</font></p>
 * <hr>
 * @author  : $Author: brian $
 * @version : $Revision: 1.3 $
 *
 * <hr><p><font size="-1" color="#336699"><a href="http://www.mbari.org">
 * The Monterey Bay Aquarium Research Institute (MBARI)</a> provides this
 * documentation and code &quot;as is&quot;, with no warranty, express or
 * implied, of its quality or consistency. It is provided without support and
 * without obligation on the part of MBARI to assist in its use, correction,
 * modification, or enhancement. This information should not be published or
 * distributed to third parties without specific written permission from MBARI.</font></p><br>
 * <font size="-1" color="#336699">Copyright 2002 MBARI.<br>
 * MBARI Proprietary Information. All rights reserved.</font><br><hr><br>
 */
public class XMLMockup {

	/**
	 *
	 * @param dir The mission directory containing an AUV mission
	 */
	public XMLMockup(File dir) throws MalformedURLException {
		this(dir, dir.toURL());
	}

	/**
	 *
	 * @param dir The source directory
	 * @param url The Web URL to access this directory
	 */
	public XMLMockup(File dir, URL url) throws MalformedURLException {


		this.dir = dir;
        baseURL = url;

        id = (int) Math.round(Math.random() * 100000);

		try {
			// Load the vehicle.cfg file as properties
			auvProps.load(new FileInputStream(new File(dir, "vehicle.cfg")));
			// Make the AUV "Platform" deployment
			Deployment dp = makeDeployment();
			xmlBuilder.addDeployment(dp);
			// Get all the log (aka Data) files
			File[] logs = getLogs();
			File log = null;
			Deployment di = null;
			for (int i = 0; i < logs.length; i++) {
				log = logs[i];
				// make an instrument deployment for each log file.
				di = makeDeployment(log);
				dp.addChildDeployment(di);
			}
		}
		catch (Exception e) {
			e.printStackTrace();
		}
	}

	/**
	 * XmlBuilder does all the XML manipulations.
	 * @return XML generation object for creating SSDS metadata
	 * @since Apr 15, 2003
	 */
	public XmlBuilder getXmlBuilder() {
		return xmlBuilder;
	}

	/**
	 * Generates a deployment description (object model) for a log file in
	 * an AUV deployment
	 *
	 * @param log The log (i.e auv data) file to create a description for
	 * @return A 'fleshed-out' deployment object
	 * @throws MalformedURLException
	 * @throws FileNotFoundException
	 * @throws IOException
	 * @since Apr 15, 2003
	 */
	private Deployment makeDeployment(File log) throws MalformedURLException, FileNotFoundException, IOException {

		// Describe the instrument deployment
		Deployment d = new Deployment();
		d.setRole(Deployment.INSTRUMENT_ROLE);
		d.setDescription("**** Mockup deployment information for " + log.getName() + " ****");

		// This is a total mockup..We're making up device IDs and descritptions.
		// We're creating a name form the log file
		Device inst = new Device(getDeviceId());
		int ext = log.getName().indexOf('.');
		inst.setName(log.getName().substring(0, ext));
		inst.setDescription("**** Mockup instrument ****");
		d.setDevice(inst);

		// Log files are DataFiles. They contain URL's. Here the URL is local
		// but we shoudl really use web URL's (http://blah.com). Need to map ploarbear onto
		// an internal server to do this.
		DataFile out = new DataFile();
		out.setName(log.getName());
		out.setDescription("**** Mockup instrument information for " + log.getName() + " ****");
		out.setUrl(makeURL(log));
		out.setOriginal(true);
		d.addOutput(out);

		// Tell SSDS how to ignore headers on the log files
		// AUV logs always used "#" as a comment tag
		HeaderDescription hd = new HeaderDescription();
		hd.addCommentTag("#");
		out.setHeaderDescription(hd);

		// All log files have the same record description. Don't need
		// to change this
		RecordDescription rd = new RecordDescription();
		rd.setBufferStyle(RecordDescription.BINARY_BUFFERSTYLE);
		rd.setBufferLengthType(RecordDescription.FIXED_BUFFERLENGTH);
		rd.setEndian(RecordDescription.LITTLE_ENDIAN);
		rd.setParseable(true);
		out.setRecordDescription(rd);

		// Read the header of the log and use the info to describe the RecordVariables
		BufferedReader in = new BufferedReader(new FileReader(log));
		String line = null;
		line = in.readLine();
		int i = 0;
		RecordVariable v = null;
		StringTokenizer st = null;
		String format = null;
		String units = null;
		String longName = null;
		String name = null;
		line = in.readLine();
		while ((line != null) && (!line.startsWith("# begin"))) {
			i++;

			//System.out.println(getClass().getName() + ": line = " + line);
			st = new StringTokenizer(line);
			st.nextToken(); // Ignore the #
			format = st.nextToken();
			name = st.nextToken();
			if (format.equals("timeTag")) {
				format = "double";
				units = "seconds since 1970-01-01 00:00:00";
				longName = "Time (GMT)";
			}
			else if (format.equals("angle")) {
				format = "double";
				units = "**** degrees ****";
				longName = name;
			}
			else {
				units = "**** UNKNOWN ****";
				longName = name;
			}

			v = new RecordVariable();
			v.setFormat(format);
			v.setName(name);
			v.setLongName(longName);
			v.setUnits(units);
			v.setColumnIndex(i);
            rd.addRecordVariable(v);
			line = in.readLine();

		}
		in.close();
		return d;

	}

    /**
     * Generate the correct URL
     * @param file
     * @return
     * @throws MalformedURLException
     * @since May 6, 2003
     */
    private URL makeURL(File file) throws MalformedURLException {
        String path = baseURL.getFile();
        if (!path.endsWith("/")) {
            path = path + "/";
        }
        path = path + file.getName();
        return new URL(baseURL.getProtocol(), baseURL.getHost(), path);
    }

	/**
	 * Make an empty AUV deployment. It still needs to have the log info added
	 * @return A mock-up of the AUV deployment
	 * @throws MalformedURLException
	 * @since Apr 15, 2003
	 */
	private Deployment makeDeployment() throws MalformedURLException {

		// Create a Ploatform deployment for the AUV
		Deployment d = new Deployment();
		d.setRole(Deployment.PLATFORM_ROLE);
		d.setName(dir.getName());
		d.setDescription("**** Mockup of an AUV mission generated from " + dir.getName() + " ****");
		d.setStartDate(new Date(0));
		d.setEndDate(new Date());
		d.setContact(makeMissionContact());

		// Describe the AUV as a Device
		Device auv = new Device(getDeviceId());
		auv.setDescription("**** Mockup AUV ****");
		auv.setName(auvProps.getProperty("vehicleName"));
		auv.setMfgName("**** MBARI ****");
		d.setDevice(auv);

		File[] resrcs = getResources();
		Resource r = null;
		for (int i = 0; i < resrcs.length; i++) {

			r = new Resource();
			r.setName(resrcs[i].getName());
			r.setDescription("**** A resource in search of a description ****");
			r.setUrl(makeURL(resrcs[i]));
			d.addResource(r);
		}

		return d;
	}

	/**
	 * Each mission should have a contact to send a notification to. Ideally, this would be the
	 * auv mailing list so that everyone knows when a mission is completed.
	 * @return
	 * @since Apr 15, 2003
	 */
	private Person makeMissionContact() {
		Person person = new Person();
		person.setFirstname("Brian");
		person.setSurname("Schlining");
		person.setEmail("brian@mbari.org");
		return person;
	}

	/**
	     *
	     * @return All the log files in a given directory
	     */
	private File[] getResources() {

		FilenameFilter filter = new FilenameFilter() {
			public boolean accept(File someDir, String someName) {
				return (!someName.endsWith(".log") && !someName.endsWith(".m"));
			}
		};

		return dir.listFiles(filter);

	}

	/**
	 *
	 * @return All the log files in a given directory
	 */
	private File[] getLogs() {

		FilenameFilter filter = new FilenameFilter() {
			public boolean accept(File someDir, String someName) {
				return someName.endsWith(".log");
			}
		};

		return dir.listFiles(filter);

	}

	private Long getDeviceId() {
		return new Long((long)id++);
	}

	public static void main(String[] args) {
		try {
            XMLMockup mockup;
            if (args.length == 1) {
                mockup = new XMLMockup(new File(args[0]));
            }
            else {
                mockup = new XMLMockup(new File(args[0]), new URL(args[1]));
            }

			XmlBuilder xb = mockup.getXmlBuilder();
			xb.marshal();
			xb.print();
		}
		catch (Exception e) {
			System.out.println("Usage: java example.XMLMockup someAuvDataDir");
		}

	}

	int id;
	File dir;
    URL baseURL;
	XmlBuilder xmlBuilder = new XmlBuilder();
	Properties auvProps = new Properties();

}
