package example;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.TimeZone;
import moos.ssds.model.*;

/**
 * <p>Example for marshalling XML from the SSDS object model</p><hr>
 * @author  : $Author: brian $
 * @version : $Revision: 1.2 $
 *
 * <hr><p><font size="-1" color="#336699"><a href="http://www.mbari.org">
 * The Monterey Bay Aquarium Research Institute (MBARI)</a> provides this
 * documentation and code &quot;as is&quot;, with no warranty, express or
 * implied, of its quality or consistency. It is provided without support and
 * without obligation on the part of MBARI to assist in its use, correction,
 * modification, or enhancement. This information should not be published or
 * distributed to third parties without specific written permission from MBARI.</font></p><br>
 * <font size="-1" color="#336699">Copyright 2002 MBARI.<br>
 * MBARI Proprietary Information. All rights reserved.</font><br><hr><br>
 */

public class Example1 {
    public Example1(String filename) {

        // Need a Calendar for getting and setting time
        Calendar calendar = new GregorianCalendar(TimeZone.getTimeZone("GMT"));

        /*
            Create an AUV deployment. The AUV deployment MUST have a unique
            name. The AUV has the role of a PLATFORM.
            Every deployment needs to have a Device assigned to it. The Device
            number is the ISI ID number.
            Any mission, configuration, or documentation files (i.e everything
            else downloaded from an auv after a mission that is NOT a datafile)
            can be added to the Deployment as a Resource. Calibration files are
            usually added as resources to a device.
         */
        Deployment auvDeploy = new Deployment();
        Device auv = new Device(9873); // All deployments require a Device
        auvDeploy.setDevice(auv); // Associate the device with a deployment
        auvDeploy.setRole(Deployment.PLATFORM_ROLE); // Set the ROLE!!
        auvDeploy.setName("2003.30.01"); // Assign the deployment a UNIQUE name
        calendar.set(2001, 12, 17, 9, 30, 47);
        auvDeploy.setStartDate(calendar.getTime()); // Date deployment started
        calendar.set(2001, 12, 17, 10, 30, 47);
        auvDeploy.setEndDate(calendar.getTime()); // Date deployment ended

        /*
           Child instrument deployments get added to the AUV Deployment. For
           AUVs, there should be one instrument deployment for each *.log data
           file.
         */
        Deployment instDeploy = new Deployment();
        auvDeploy.addChildDeployment(instDeploy); // Add an instument deployment to the AUV deployment
        Device inst = new Device(101);
        instDeploy.setDevice(inst);
        instDeploy.setRole(Deployment.INSTRUMENT_ROLE); // Set the ROLE!!

        /*
           Some of the log files are compilations of data from several sensors.
           For example, a cd may have serveral sensors conneced to it but they
           are all stored in the ctdDriver.log. These are handled as SENSOR
           deployments. Sensor deployments get added to the parent Instruments
           deployment
         */
        Deployment sensDeploy1 = new Deployment();
        instDeploy.addChildDeployment(sensDeploy1); // Add to instrument deployment
        Device sens1 = new Device(102); // Don't forget Deployments must have Devices
        sensDeploy1.setDevice(sens1);

        /*
           This SENSOR delployment has a configuration file associated with it
           as an example of how to add a resource.
         */
        Deployment sensDeploy2 = new Deployment();
        instDeploy.addChildDeployment(sensDeploy2);
        Device sens2 = new Device(103);
        sensDeploy2.setDevice(sens2);
        Resource res2 = new Resource();
        try {
            res2.setUrl(new URL(
                "http://dods.mbari.org/brian/pub/auv/conf/humid_20010231.cfg"));
        }
        catch (MalformedURLException ex1) {
        }
        sensDeploy2.addResource(res2);

        /*
           Create a reference to the log file (a.k.a the binary data files. The
           log file MUST be placed somewhere that SSDS can reference it. We
           reference the data file using URLs. The URL can be a file URL, for
           example, file:/hosts/tornado/vol/vol0/auvdata/data.log, but the file
           path needs to be one that the machines predator and prey can access.
           Here we use a web URL.
         */
        DataFile log = new DataFile();
        instDeploy.addOutput(log); // Add the file as output to an Instrument deployment
        try {
            log.setUrl(new URL(
                "http://dods.mbari.org/brian/pub/auv/data/2003.030.01/ctdDriver.log"));
        }
        catch (MalformedURLException ex2) {
        }

        /*
           Describe the contents of the log file and how to parse it. From what
           I've seen the HeaderDescription and RecordDescription is exactly the
             same for ALL AUV binary log files; The code snippet below should work
           for all of them.
         */
        HeaderDescription header = new HeaderDescription();
        log.setHeaderDescription(header);
        header.addCommentTag("#");
        RecordDescription rd = new RecordDescription();
        log.setRecordDescription(rd);
        rd.setBufferStyle(RecordDescription.BINARY_BUFFERSTYLE);
        rd.setBufferLengthType(RecordDescription.FIXED_BUFFERLENGTH);
        rd.setParseable(true);
        rd.setEndian(RecordDescription.LITTLE_ENDIAN);

        /*
           Variables describe each field in a data record.
           For brevity in this example, I'm only adding one variable...
         */
        RecordVariable v = new RecordVariable();
        rd.addRecordVariable(v); // Add the variable to the RecordDescription
        v.setName("time"); // Please DO NOT put spaces in the name
        v.setColumnIndex(1); // The order that this variable appears in the record (start at 1 and count up)
        v.setFormat("double"); // most AUV data is double precision (including time). Can use integer, float, etc too.
        v.setLongName("Time (GMT)"); // Give us a nice descriptive name that makes sense on a graph. Don't put units in it
        v.setUnits("milliseconds since Jan 01, 1970"); // Use this for time.

        /*
             Generate the XML. Once its marshalled it can written to a file or an
            OutputStream
         */
        XmlBuilder xb = new XmlBuilder();
        xb.addDeployment(auvDeploy); // All the objects created above are
        // chained under auvDeply. So we only need
        // to add auvDeploy to the XmlBuilder
        xb.marshal(); // <- Don't forget to marshal!!!

        try {
            // write XML to a file
            xb.toFile(new File(filename));
            // write XML to the console
            xb.print();
            // Write XML to a Stream
            //xb.print(someOutputStream, "ISO-8859-1");
        }
        catch (IOException ex) {
        }

    }

    public static void main(String[] args) {

        Example1 example11 = new Example1(args[0]);

    }
}
