/****************************************************************************/
/* Copyright 1991 MBARI                                                     */
/****************************************************************************/
/* $Header: /home/cvs/oasis3/src/operations/src/atlas.c,v 1.1 2003/08/20 19:39:43 headley Exp $			    */
/* Summary  : ATLAS decode routines for decode.c, extract.c		    */
/* Filename : atlas.c							    */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 1.1 $							    */
/* Created  : 01/24/92							    */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* 24jan91 rah - created						    */
/* $Log: atlas.c,v $
/* Revision 1.1  2003/08/20 19:39:43  headley
/* no message
/*
 * Revision 3.1  2001/06/19  13:13:39  13:13:39  oasisa (Oasis users)
 * Periodic Update 6/19/2001 (klh)
 * 
 * Revision 3.0  99/05/12  10:11:25  10:11:25  bobh (Bob Herlien)
 * Added tstring, misc changes
 * 
 * Revision 2.9  98/08/24  13:45:53  13:45:53  bobh (Bob Herlien)
 * Archiving sources after M2/M3 & Eqpac deployments of 1998
 * 
 * Revision 2.8  98/03/17  11:11:39  11:11:39  bobh (Bob Herlien)
 * Archiving sources prior to porting to DOS/Windows
 * 
 * Revision 2.7  97/09/09  09:52:45  09:52:45  bobh (Bob Herlien)
 * Archiving various changes
 * 
 * Revision 2.4  94/01/21  14:36:17  14:36:17  hebo (Bob Herlien)
 * Added support for date ranges in cfg file
 * 
 * Revision 1.3  94/01/17  11:07:26  11:07:26  hebo (Bob Herlien)
 * Misc changes
 * 
 * Revision 1.2  92/09/24  10:20:30  10:20:30  hebo (Bob Herlien)
 * Added check for broken sensors, write 0.0 result.  This due to M2 broken 500m
 * 
 * Revision 1.1  92/09/12  10:10:14  10:10:14  hebo (Bob Herlien)
 * Corrected problem with wind direction due to checking bounds before atan2()
 * 
 * Revision 1.0  92/02/25  10:46:57  10:46:57  hebo (Bob Herlien)
 * Initial revision
 * 
*/
/****************************************************************************/

#include <stdio.h>			/* Standard I/O			    */
#include <mbariTypes.h>			/* MBARI type definitions	    */
#include <mbariConst.h>			/* MBARI constants		    */
#include <decode.h>			/* OASIS controller definitions	    */
#include <time.h>			/* Time				    */
#include <memory.h>			/* for memcpy()			    */
#include <ctype.h>			/* for toupper()		    */
#include <math.h>

#define CAL_FILE	"cable01a.coef"	/* ATLAS calibration file	    */
#define RAWSPEED_CNT	0.192		/* .192 m/s for each count of raw spd*/


/********************************/
/*	External Functions	*/
/********************************/

Extern Nat16	getIntelword( Byte *p ); /* Get word in Intel format	    */


/************************************************************************/
/* Function    : init_pod_cal						*/
/* Purpose     : Initialize one ATLAS pod calibration			*/
/* Inputs      : Ptr to pod to init					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
init_pod_cal( ACal *p )
{
    p->a = (double)0.0;
    p->b = (double)1.0;
    p->c = (double)0.0;
    p->d = (double)0.0;
    p->e = (double)1.0;

} /* init_pod_cal() */


/************************************************************************/
/* Function    : init_atlas_cal						*/
/* Purpose     : Initialize ATLAS calibration				*/
/* Inputs      : Cal struct ptr						*/
/* Outputs     : none							*/
/************************************************************************/
	Void
init_atlas_cal( AtlasCal *cp )
{
    Int32	i;

    init_pod_cal( &cp->ac_air );
    init_pod_cal( &cp->ac_sst );
    for ( i = 0; i < 10; i++ )
	init_pod_cal( &cp->ac_temp[i] );
    init_pod_cal( &cp->ac_rh );
    init_pod_cal( &cp->ac_wind );
    cp->ac_wind.a = (double)1.0;
    cp->ac_wind.b = (double)0.0;
    cp->ac_wind.c = (double)1.0;

} /* init_atlas_cal() */


/************************************************************************/
/* Function    : read_atlas_cal						*/
/* Purpose     : Read ATLAS calibration file				*/
/* Inputs      : Name of file, ptr to calibr struct			*/
/* Outputs     : OK or ERROR						*/
/* Comments    : From ATLAS documention, sensor type field is:		*/
/*	1	Original temperature sensor (old cable, RP/LP air and SST)*/
/*	2	Pressure						*/
/*	3	Wind							*/
/*	4	Conductivity						*/
/*	5	New-style temperature sensor calibration		*/
/*	6	Air (for HP and Standard ATLAS)				*/
/*	7	SST (for HP tube)					*/
/*	8	Relative humidity sensor				*/
/************************************************************************/
	Status
read_atlas_cal( char *name, AtlasCal *cp )
{
    FILE	*fp;
    Int32	t, p;
    ACal	fc;
    char	buff[256];
    char	id[16];

    init_atlas_cal( cp );

    if ( name == NULL )
	fp = fopen( CAL_FILE, "rb" );
    else
	fp = fopen( name, "rb" );

    if ( fp == (FILE *)NULL )
	return( ERROR );

    t = p = 0;

    while ( fgets(buff, sizeof(buff), fp) != NULL )
    {
	init_pod_cal( &fc );
	if ( sscanf(buff, "%1d %d %4c %lg %lg %lg %lg %lg", &fc.pod_type, 
		 &fc.ident, id, &fc.a, &fc.b, &fc.c, &fc.d, &fc.e) >= 6 )
	{
	    switch( fc.pod_type )
	    {
	      case 1:
	      case 5:
		if ( t < TEMPS )
		    memcpy( (void *)&(cp->ac_temp[t++]), 
			    (void *)&fc, sizeof(fc) );
		break;

	      case 2:
		if ( p < PRESS )
		    memcpy( (void *)&(cp->ac_press[p++]), 
			    (void *)&fc, sizeof(fc) );
		break;

	      case 3:
		memcpy( (void *)&(cp->ac_wind), (void *)&fc, sizeof(fc) );
		break;

	      case 6:
		memcpy( (void *)&(cp->ac_air), (void *)&fc, sizeof(fc) );
		break;

	      case 7:
		memcpy( (void *)&(cp->ac_sst), (void *)&fc, sizeof(fc) );
		break;
	    }
	}
    }

    fclose( fp );
    return( OK );

} /* read_atlas_cal() */


/************************************************************************/
/* Function    : decode_temp						*/
/* Purpose     : Decode ATLAS temperature				*/
/* Inputs      : Count from temp pod, ptr to ACal struct		*/
/* Outputs     : Double-precision temperature, or 0.0 if error		*/
/************************************************************************/
	double
decode_temp( Nat16 count, ACal *cp )
{
    double	r, logr, temp;

    if ( count < 0x1000 )		/* Check for broken sensor*/
	return( (double)0.0 );

    switch( cp->pod_type )
    {
      case 1:				/* Original pod, incl. orig SST	*/
	r = 7.68e08 / ((2.525 * (double)count) - 7680.0);
	break;

      case 5:				/* Self-calibrating temp pod	*/
	r = 3.072e08 / (double)count;
	break;

      case 7:				/* HP SST			*/
	r = ((4.0e08/(double)count) * cp->e) + cp->d;
	break;

      default:
	return( (double)0.0 );

    }

    logr = log10(r);
    temp = 1.0 / (cp->a + (cp->b * logr) + (cp->c * logr * logr * logr))
	    - 273.15;

    if ( temp < 0.0 )			/* More error checking	  */
	return( (double)0.0 );

    return( temp );

} /* decode_temp() */


/************************************************************************/
/* Function    : decode_atlas						*/
/* Purpose     : Decode ATLAS information				*/
/* Inputs      : Pointer to ATLAS data, length, ptr to return struct	*/
/* Outputs     : OK, SIZE_ERR or CHKSUM_ERR				*/
/************************************************************************/
	Status
decode_atlas( Byte *atdata, Int32 atlen, AtlasDecode *adp, AtlasCal *acp )
{
    double	r, u, v;
    Int32	i;
    Int32	at_words;
    Nat16	chksum;
    signed char	cu, cv;
    union
    {
	Atlas		ata;
	AtlasLong	atl;
	Nat16		atn[ATLAS_LONG];
    } atlas;

    at_words = atlen / sizeof(Nat16);
    if ( (at_words != ATLAS_WORDS) && (at_words != ATLAS_LONG) )
	return( SIZE_ERR );			/* Check # words, rtn if bad */

    chksum = 0;					/* Init checksum	     */
    for ( i = 0; i < at_words - 1; i++ )
    {						/* Translate Intel word type */
	atlas.atn[i] = getIntelword(atdata + i * sizeof(Nat16));
	chksum += atlas.atn[i];			/* Compute checksum	     */
    }

    if ( chksum != getIntelword(atdata + i * sizeof(Nat16)) )
	return( CHKSUM_ERR );			/* Return error if bad chksum*/

		/* Get time of ATLAS data, convert to ddd.fff	*/
    adp->atd_time = (double)atlas.ata.at_time / 128.0;
    adp->atd_time += fmod(adp->atd_time, 1.0) / 3.0;

    r = (5.0 * (double)atlas.ata.at_air / 1024.0 * acp->ac_air.e)
	+ acp->ac_air.d;			/* Calc voltage of air temp  */
    adp->atd_air = (r * acp->ac_air.b) + acp->ac_air.a;
						/* Calc air temp from voltage*/

    adp->atd_sst = decode_temp( atlas.ata.at_sst, &acp->ac_sst );

    for ( i = 0; i < TEMPS; i++ )	/* Calculate temp of each temp pod  */
	adp->atd_temp[i] = decode_temp(atlas.ata.at_temp[i], &acp->ac_temp[i]);


    for ( i = 0; i < PRESS; i++ )	/* Calculate pressure of each pod  */
    {
	if ( atlas.ata.at_press[i] == 0 )	/* Check for broken sensor*/
	    adp->atd_press[i] = 0.0;
	else
	    adp->atd_press[i] = 0.6897 * (((76800.0 / atlas.ata.at_press[i])
				   * acp->ac_press[i].b) + acp->ac_press[i].a);
    }
				/* Calculate relative humidity		     */
    adp->atd_rh = (((double)atlas.ata.at_rh / 2.56) * acp->ac_rh.b) 
			+ acp->ac_rh.a;

				/* Calculate wind speed and direction	     */
    cu = (signed char)(atlas.ata.at_wind >> 8);
    cv = (signed char)(atlas.ata.at_wind & 0xff);
    u = acp->ac_wind.a * (double)cu;
    v = acp->ac_wind.a * (double)cv;
    adp->atd_windspd = (sqrt(u*u + v*v) * acp->ac_wind.e) + acp->ac_wind.d;

    if ( u == 0 )
	adp->atd_winddir = 0.0;
    else
	adp->atd_winddir = 180.0/PI * atan2(u,v);
    adp->atd_winddir += acp->ac_wind.b;
    while ( adp->atd_winddir < 0.0 )
	adp->atd_winddir += 360.0;

    adp->atd_windu = adp->atd_windspd * sin(adp->atd_winddir * PI / 180.0);
    adp->atd_windv = adp->atd_windspd * cos(adp->atd_winddir * PI / 180.0);

    adp->atd_compass = (double)(atlas.atl.at_covane >> 8) * 360./256.;
    adp->atd_vane  = (double)(atlas.atl.at_covane & 0xff) * 360./256.;
    adp->atd_rawdir = adp->atd_compass + adp->atd_vane + acp->ac_wind.b + 180.;
    while ( adp->atd_rawdir >= 360.0 )
	adp->atd_rawdir -= 360.0;

    adp->atd_rawspeed = (double)(atlas.atl.at_windspeed >> 8) * RAWSPEED_CNT;

    return( OK );

} /* decode_atlas() */
