/*--- formatted by Jindent 2.1, (www.c-lab.de/~jindent) ---*/

// Title:        NetCDF library
// Version:
// Copyright:    Copyright (c) 1999
// Author:       Brian Schlining
// Company:      MBARI
// Description:
package org.mbari.util;

import java.lang.reflect.*;
import java.text.*;
import java.util.*;

/**
 * Utilities for converting between diferent types of commonly used
 * scientific date formats
 *
 * @author Brian Schlining
 * @revision 1.0.0, 19 Jul 1999
 */
public class TimeUtil {

	// ///////////////////////////////////////
	// Public Static Methods

	/**
	 * Convert the Day of a Year to a Date object. This method can handle days of
	 * the year greater than 365 (or 366 for leap years). It simple wraps the day
	 * around to the next year. For example: doyToDate(367, 1999) will return a
	 * Date object for Jan 2, 2000. All times must be in GMT.
	 *
	 * @param dayOfYear  The decimal day of the year. Jan 01, 00:00:00 = 1
	 * @param year       The year
	 * @return A Date object for the dayOfYear specified
	 */
	public static final Date dayOfYearToDate(double dayOfYear, double year) {

		// Determine if the year is a leap year. Specify the number of days in Feb
		double	feb = 28.0d;
		double	max = 365d;

		if (TimeUtil.isLeapYear(year)) {
			feb = 29.0d;
			max = 366d;
		}

		double[] daysPerMonth = {
			0d, 31d, feb, 31d, 30d, 31d, 30d, 31d, 31d, 30d, 31d, 30d, 31d
		};
		double[] days = MathUtil.cumSum(daysPerMonth);

		// Handle days outside a normal year
		while (Math.floor(dayOfYear) > max) {
			dayOfYear -= max;
			year++;

			if (DateConverter.isLeap(year)) {
				max = 366d;
			}
			else {
				max = 365d;
			}
		}

		// Get the month
		int	month;

		loop:
		{
			for (month = 0; month < days.length; month++) {
				if (dayOfYear < days[month]) {
					break loop;
				}
			}
		}

		// Get the day
		double		day = dayOfYear - days[month - 1];

		// Get the hour
		double		hour = (day - Math.floor(day)) * 24.0d;

		// Get the minute
		double		minute = (hour - Math.floor(hour)) * 60.0d;

		// Get the second
		double		second = (minute - Math.floor(minute)) * 60.0d;
		GmtCalendar gc =
			new GmtCalendar((int) year,
								 (month - 1),		// In Java months are incremented 0 to 11
		   (int) Math.floor(day), (int) Math.floor(hour),
		   (int) Math.floor(minute), (int) Math.rint(second));

		/*
		 * int tz = gc.getTimeZone().getRawOffset();
		 * gc.add(Calendar.MILLISECOND, tz);
		 *
		 * gc.setTimeZone(TimeZone.getTimeZone("GMT"));
		 */
		return gc.getTime();
	}

	/**
	 * Convert the Day of a Year to a Date object
	 *
	 * @param dayOfYear  The decimal day of the year. Jan 01, 00:00:00 = 1
	 * @param year       The year
	 * @return A Date object for the dayOfYear specified
	 */
	public static final Date dayOfYearToDate(int dayOfYear, int year) {
		return TimeUtil.dayOfYearToDate((double) dayOfYear, (double) year);
	}

	/**
	 * Convert the Day of a Year to a Date object
	 *
	 * @param dayOfYear  The decimal day of the year. Jan 01, 00:00:00 = 1
	 * @param year       The year
	 * @return A Date object for the dayOfYear specified
	 */
	public static final Date dayOfYearToDate(float dayOfYear, int year) {
		return TimeUtil.dayOfYearToDate((double) dayOfYear, (double) year);
	}

	/**
	 * Convert the Day of a Year to a Date object
	 *
	 * @param dayOfYear  The decimal day of the year. Jan 01, 00:00:00 = 1
	 * @param year       The year
	 * @return A Date object for the dayOfYear specified
	 */
	public static final Date dayOfYearToDate(double dayOfYear, int year) {
		return TimeUtil.dayOfYearToDate(dayOfYear, (double) year);
	}

	/**
	 * Convert from UTC (milliseconds since 01 Jan 1970 00:00:00) to serial days,
	 * the format used by matlab (Serial days. 01 jan 0000 00:00:00 = Day 1).
	 *
	 * @param utc time in UTC
	 * @return Serial day format utilized by matlab
	 */
	public static final double utcToSerialDays(long utc) {
		return TimeUtil.utcToSerialDays((double) utc);
	}

	/**
	 * Convert from UTC (milliseconds since 01 Jan 1970 00:00:00) to serial days,
	 * the format used by matlab (Serial days. 01 jan 0000 00:00:00 = Day 1).
	 *
	 * @param utc time in UTC
	 * @return Serial day format utilized by matlab
	 */
	public static final double utcToSerialDays(double utc) {
		return utc / 1000D / 60D / 60D / 24D + 719529D;
	}

	/**
	 * Convert from serial days, the format used by matlab,
	 * (Serial days.01 jan 0000 00:00:00 = Day 1) to UTC
	 * (milliseconds since 01 Jan 1970 00:00:00).
	 *
	 * @param serialDay The date value to be conerted
	 * @return UTC time
	 */
	public static final long serialDaysToUtc(double serialDay) {
		return (long) ((serialDay - 719529D) * 1000D * 60D * 60D * 24D);
	}

	/**
	 * Check to see if a year is a leap year
	 *
	 * @param year  The year in question
	 * @return True if the year is a leap year, false otherwise
	 */
	public static final boolean isLeapYear(double year) {
		GregorianCalendar gc = new GregorianCalendar();
		return gc.isLeapYear((int) year);
	}

	/**
	 * Check to see if a year is a leap year
	 *
	 * @param year  The year in question
	 * @return True if the year is a leap year, false otherwise
	 */
	public static final boolean isLeapYear(int year) {
		return DateConverter.isLeap((double) year);
	}

	/**
	 * Convert a date object ot serial days
	 *
	 * @param d The data oject to be converted
	 * @return Serial date corresponding to the <i>d</i>
	 */
	public static double toSerialDays(Date d) {
		return utcToSerialDays(d.getTime());
	}

   public static int dateToDayOfYear(Date date) {
      GmtCalendar gmt = new GmtCalendar(date);
      return gmt.get(Calendar.DAY_OF_YEAR);
   }

   public static int dateToDayOfYear(long millis) {
      GmtCalendar gmt = new GmtCalendar(millis);
      return gmt.get(Calendar.DAY_OF_YEAR);
   }

   public static double getJulianDate(Date date) {
      return TimeUtil.getJulianDate(date.getTime());
   }

   public static double[] getJulianDate(long[] millis) {
      double[] julianDate = new double[millis.length];
      for (int i = 0; i < millis.length; i++ ) {
         julianDate[i] = TimeUtil.getJulianDate(millis[i]);
      }
      return julianDate;
   }

   /**
    * Julian Date (not Day of Year). Check values against matlabs juldate_.m on
    * 14 Nov 2000. Found some problems so I checked it again on 16 Nov 2000. It
    * appears to match exactly now.
    */
   public static double getJulianDate(long millis) {
      double startGregorian = 588829; // Julian date when Gregorian calendar was adopted
      GmtCalendar gmt = new GmtCalendar(millis);
      int year  = gmt.get(Calendar.YEAR);
      int month = gmt.get(Calendar.MONTH) + 1;
      int day   = gmt.get(Calendar.DAY_OF_MONTH);

      double JA,JY,JM,JD,JJ;

      if (gmt.get(Calendar.ERA) == 0)
         year = -year + 1;

      JY = year - 1;
      JM = month + 13;

      if (month > 2) {
        JY = year;
        JM = month + 1;
      }

      JD = Math.floor(365.25 * JY) + Math.floor(30.6001 * JM) + day + 1720995 - 0.5;
      JJ = day + 31 * (month + 12 * year);

      if (JJ >= startGregorian) {
         JA = Math.floor(0.01*JY);
         JD = JD + 2 - JA + Math.floor(0.25*JA);
      }

      //System.out.println("JD = " + JD + "\nJY = " + JY + "\nJM = " +  JM + "\nYear = " +
      // year + "\nmonth = " + month + "\nday = " + day);

      return JD;

   }


}



/*--- formatting done in "Brian Schlining" style on 03-06-2000 ---*/

