
/**
 * Title:        <p>
 * Description:  <p>
 * Copyright:    Copyright (c) <p>
 * Company:      <p>
 * @author
 * @version 1.0
 */
package org.mbari.util;

import java.io.*;
import org.mbari.io.*;
import org.mbari.model.*;


public class SolarUtil {

   /**
    * @return the wavelength of Neckles and Labs Extra-terrestrial Irradiance in
    * nanometers (cf. getNeckelLabIrradiance())
    */
   public static final double[] getNeckelLabWavelengths() {
      double[] lambda = {
       329.80,  331.80,  333.80,  335.90,  337.90,  339.80,  341.90,  343.80,  345.80,  347.90,
       349.80,  351.90,  353.90,  355.90,  357.90,  360.00,  362.00,  364.10,  366.10,  367.80,
       369.80,  371.90,  373.70,  375.50,  377.50,  379.40,  381.40,  383.40,  385.40,  387.40,
       389.40,  391.20,  393.20,  395.00,  396.90,  398.90,  400.10,  402.00,  404.00,  406.00,
       407.90,  409.90,  411.70,  413.70,  415.60,  417.60,  419.10,  421.10,  423.10,  424.80,
       426.80,  428.60,  430.60,  432.60,  434.00,  435.90,  437.20,  439.20,  440.90,  442.80,
       444.70,  446.40,  448.30,  450.30,  452.40,  454.40,  456.40,  458.40,  460.40,  462.40,
       464.40,  466.40,  468.40,  470.40,  472.30,  474.30,  476.30,  478.30,  480.30,  482.30,
       484.30,  486.30,  488.30,  489.30,  491.30,  493.30,  495.10,  497.10,  498.90,  500.90,
       502.90,  504.90,  506.80,  508.80,  510.80,  512.80,  514.80,  516.80,  518.80,  520.00,
       521.90,  523.90,  525.80,  527.80,  529.80,  531.60,  533.50,  535.40,  537.20,  539.20,
       541.00,  543.00,  545.00,  547.00,  549.00,  550.80,  552.70,  554.70,  556.70,  558.70,
       560.70,  562.70,  564.60,  566.60,  568.60,  570.60,  572.60,  574.60,  576.60,  578.50,
       580.50,  582.50,  584.50,  586.50,  588.40,  590.40,  592.30,  594.30,  596.30,  598.30,
       600.30,  602.30,  604.30,  606.30,  608.00,  610.00,  612.00,  614.00,  616.00,  618.00,
       620.00,  622.00,  623.90,  625.90,  627.90,  629.90,  631.90,  633.90,  635.90,  637.90,
       639.90,  641.90,  643.90,  645.90,  647.90,  649.90,  651.90,  653.90,  655.90,  662.10,
       666.30,  679.00,  709.00
      };
      return lambda;
   }


   public static final double[] getNeckelLabIrradiance(double[] lambda) {
      return MathUtil.interpLinear(SolarUtil.getNeckelLabWavelengths(),
       SolarUtil.getNeckelLabIrradiance(), lambda);
   }

   /**
    * @returns Neckels and Labs extra-terrestrial irradiance (uW cm^-2 nm^-2)
    */
   public static final double[] getNeckelLabIrradiance() {
      double[] E0 = {
       107.65,   98.70,   95.00,   91.00,   88.95,  100.85,   95.75,   87.85,   94.40,   95.40,
        98.40,   97.30,  115.55,  108.00,   76.30,  108.85,  106.90,  101.60,  128.80,  117.85,
       123.05,  119.00,   98.45,  104.70,  139.00,  122.95,  114.80,   74.90,  103.05,   99.90,
       117.30,  129.95,   76.55,  130.10,   88.65,  160.60,  170.15,  178.55,  162.20,  164.55,
       169.15,  166.00,  181.35,  173.30,  178.85,  169.25,  165.60,  181.85,  165.00,  175.10,
       162.00,  164.90,  123.00,  179.35,  173.65,  185.55,  184.35,  167.35,  179.35,  195.55,
       194.80,  185.15,  202.00,  213.00,  199.65,  201.30,  205.90,  201.60,  202.15,  209.45,
       199.45,  195.65,  201.10,  194.30,  201.85,  203.20,  197.20,  205.05,  206.40,  204.15,
       201.20,  171.50,  187.75,  193.95,  188.40,  193.50,  198.90,  199.35,  192.10,  182.35,
       192.20,  191.55,  196.70,  192.75,  193.20,  189.50,  185.25,  169.20,  173.30,  181.80,
       188.10,  191.35,  183.15,  180.80,  194.10,  193.40,  184.35,  193.75,  184.95,  184.25,
       180.50,  185.45,  188.75,  184.35,  186.75,  185.80,  185.35,  188.10,  180.05,  179.95,
       180.40,  185.35,  182.90,  182.20,  133.70,  178.90,  187.25,  184.25,  184.15,  180.25,
       184.00,  184.40,  183.30,  178.95,  174.50,  173.85,  177.65,  178.70,  179.60,  173.15,
       172.45,  170.55,  175.85,  173.70,  173.00,  172.00,  173.80,  169.95,  166.90,  172.50,
       171.50,  169.05,  165.70,  165.80,  168.15,  164.00,  165.00,  164.15,  165.70,  166.25,
       162.05,  160.95,  162.00,  161.20,  161.20,  156.45,  160.20,  159.05,  137.30,  157.45,
       155.00,  146.95,  138.00
      };
      return E0;
   }

   public static final double[] getOzoneWavelengths() throws IOException {
      FlatFileReader ffr = new FlatFileReader(
       SystemUtil.getFile("org/mbari/model/solar/ozone.txt"));
      ffr.readFile();
      return ffr.getColumn(0);
   }

   public static final double[] getOzoneAbsorption() throws IOException {
      FlatFileReader ffr = new FlatFileReader(
       SystemUtil.getFile("org/mbari/model/solar/ozone.txt"));
      ffr.readFile();
      return ffr.getColumn(1);
   }

   public static final double[] getOzoneAbsorption(double[] lambda) throws IOException {
      FlatFileReader ffr = new FlatFileReader(
       SystemUtil.getFile("org/mbari/model/solar/ozone.txt"));
      ffr.readFile();
      return MathUtil.interpLinear(ffr.getColumn(0),
       ffr.getColumn(1), lambda);
   }

   /**
    * Low Precision
    * This is equal to 1/earthSunDistance
    */
   public static double getEarthSunIrradianceCorrectionLP(int dayOfYear) {
      double e = SolarUtil.EARTH_ORBIT_ECCENTRICITY;	 // Earth's orbital eccentricity
      return Math.pow((1 + e * Math.cos(2D * Math.PI * (dayOfYear - 3D)/365D)), 2D);
   }

   /**
    * Low Precision
    * Distance in astronomical units
    */
   public static double getEarthSunDistanceLP(int dayOfYear) {
      return 1D / SolarUtil.getEarthSunIrradianceCorrectionLP(dayOfYear);
   }

   public static double[] getCorrectedETIrradiance(double[] lambda, int dayOfYear) {
      double[] E0 = SolarUtil.getNeckelLabIrradiance(lambda);
      double earthSunDistance = SolarUtil.getEarthSunIrradianceCorrectionLP(dayOfYear);
      double[] Es = new double[E0.length];  // Extraterrestrial irradiance corrected for earth-sun distance
      for (int i = 0; i < E0.length; i++) {
          Es[i] = E0[i] * earthSunDistance;
      }
      return Es;
   }

   /** Adapted from PLOTephem.HPL and
    *  Internal parameter names (P10-P20) are identical to those in the
    *  HPL program, PLOTephem, from which this was extracted.
    *
    * This program agrees with Wilson (1980) and Duffet-Smith (1997)
    * W. Broenkow 21 Jul 1997
    *
    * Ported MLML/OCEANS matlab toolbox function almanac_.m
    */
   public static SolarPosition getSolarPosition(long millis, double longitude, double latitude) {
      return new SolarPosition(millis, longitude, latitude);
   }

   /**
    * Estimate Dobson units from climatology, given the month and
    * latitude.  Table has 12 columns, one per month, and 35 rows,
    * from 85 N to -85, in steps of 5 deg lat.
    *
    * Note: corrected two bad PAR values in table (eg, 396 -> 296)
    *
    * Subroutine EstimDobson( month, lat, Dobson )
    */
    public static double estimDobson(int month, double lat) {

       int        i1,
                  i2;

       double     Dobson,
                  fac,
                  diffLat;
	    double[][] TabDobson = new double[35][12];
       double[]   tmpDobson =
          {395, 395, 395, 395, 395,     392,     390,     387,     376,
           354, 322, 292, 269, 254,     248,     246,     247,     251,
           255, 260, 266, 271, 277,     286,     295,     306,     319,
           334, 344, 344, 338, 331,     324,     320,     316,
           433, 433, 433, 436, 432,     428,     426,     418,     402,
           374, 338, 303, 278, 261,     251,     246,     248,     250,
           254, 258, 262, 265, 270,     278,     286,     294,     303,
           313, 322, 325, 324, 317,     306,     299,     294,
           467, 470, 460, 459, 451,     441,     433,     420,     401,
           377, 347, 316, 291, 271,     260,     254,     254,     255,
           257, 259, 261, 264, 269,     277,     284,     289,     296,
           305, 312, 315, 317, 312,     305,     299,     295,
           467, 465, 462, 455, 444,     431,     421,     410,     395,
           373, 348, 325, 304, 287,     275,     267,     261,     259,
           258, 259, 260, 263, 271,     278,     284,     289,     297,
           306, 314, 318, 319, 313,     302,     302,     302,
           411, 414, 416, 415, 410,     406,     402,     394,     382,
           363, 342, 324, 307, 291,     279,     271,     264,     260,
           258, 257, 258, 264, 271,     281,     291,     303,     312,
           318, 322, 323, 322, 322,     322,     322,     322,
           371, 371, 370, 368, 367,     372,     375,     372,     360,
           341, 323, 311, 301, 290,     282,     275,     268,     263,
           259, 256, 258, 264, 273,     289,     306,     319,     327,
           328, 328, 337, 337, 337,     337,     337,     337,
           333, 332, 332, 334, 338,     346,     350,     346,     335,
           321, 310, 302, 296, 289,     284,     280,     274,     268,
           262, 259, 261, 268, 279,     295,     315,     331,     340,
           342, 338, 344, 340, 340,     340,     340,     340,
           311, 308, 308, 313, 320,     327,     330,     326,     319,
           310, 303, 298, 291, 286,     283,     281,     277,     273,
           268, 264, 266, 274, 288,     306,     327,     343,     353,
           355, 351, 339, 325, 307,     294,     294,     294,
           283, 291, 302, 308, 312,     317,     318,     313,     307,
           300, 295, 290, 284, 279,     279,     279,     278,     276,
           272, 270, 273, 282, 295,     313,     333,     348,     360,
           367, 368, 353, 324, 291,     267,     253,     230,
           299, 299, 299, 309, 315,     317,     317,     312,     302,
           291, 283, 280, 275, 270,     268,     267,     263,     263,
           265, 269, 277, 287, 301,     317,     336,     354,     371,
           387, 402, 402, 374, 333,     294,     274,     259,
           314, 314, 314, 314, 332,     332,     327,     322,     311,
           297, 284, 276, 270, 263,     261,     260,     258,     259,
           264, 270, 278, 286, 298,     311,     323,     335,     350,
           366, 381, 390, 388, 376,     357,     346,     341,
           358, 358, 358, 358, 358,     358,     353,     349,     338,
           320, 299, 281, 267, 256,     252,     251,     251,     253,
           257, 264, 272, 279, 287,     297,     307,     318,     332,
           347, 358, 365, 366, 364,     358,     356,     353};

       int n = -1;
       for (int c = 0; c < 12; c++) {
          for (int r = 0; r < 35; r++) {
             n = n + 1;
             TabDobson[r][c] = tmpDobson[n];
          }
       }

       month = month - 1;  // Java indexs arrays starting at 0
       if (lat >= 85) {
          Dobson  = TabDobson[0][month];
       } else if (lat <= -85) {
          Dobson  = TabDobson[34][month];
       } else if (lat >= 0) {
          i1      = 17 - (int) (lat / 5.0);
          i2      = i1 + 1;
          fac     = (TabDobson[i2][month] - TabDobson[i1][month]) / (-5.0);
	       diffLat = lat - (90.0 - (i1 * 5.0));
	       Dobson  = TabDobson[i1][month] + fac * diffLat;
       } else {
          i1      = 18 - (int) (lat / 5.0);
	       i2      = i1 + 1;
	       fac     = (TabDobson[i2][month] - TabDobson[i1][month]) / (-5.0);
	       diffLat = lat - ( 90.0 - (i1 * 5.0) );
	       Dobson  = TabDobson[i1][month] + fac * diffLat;
       }

       return Dobson;
   }



   public static final double EARTH_ORBIT_ECCENTRICITY = 0.0167; // Earth's orbital eccentricity

}