package org.mbari.test;

import java.util.HashSet;
import java.util.Iterator;
/**
 * Test case for publishing periodically updated data. Useful for testing
 * Langrangian Navigation and visAD programs
 */
public class RandomWalk {

    /**
     * Default constructor. Sets delay between point updates to 1000 millisecs.
     * The delay can be changed on the fly with <i>setDelay()</i>
     */
    public RandomWalk() {
        this(1000);
    }

    /**
     * Constructor which uses a user specified delay. The delay can be changed on
     * the fly with setDelay.
     *
     * @param delay Delay between fix updates in seconds
     */
    public RandomWalk(long delay) {
        this.delay = delay;

        ////////////////////////////////////////////////////
        // Implement an internal thread for the intial delay
        Runnable r = new Runnable() {
            public void run() {
                try {
                    runThis();
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        };

        walkThread = new Thread(r);
    }

    /**
     * Starts the random walk.
     */
    public void start() {
        isRunning = true;
         walkThread.start();
    }

    /**
     * Stops the random walk
     */
    public void stop() {
        isRunning = false;
        if (walkThread != null) {
            walkThread.interrupt();
        }
    }

    /**
     * Resets the current position of the point to x=0, y=0;
     */
    public synchronized void reset() {
        point.setX(0D);
        point.setY(0D);
    }

    public boolean isAlive() {
        return walkThread.isAlive();
    }

    /**
     * Listener should implement the IPointListener interface. All listeners will
     * be updated when a point is update
     * @param o An object
     * @todo How to I check to make sure this implements the IPointListener
     * interface? This needs to throw an  exception if its not the
     * right kind of listener.
     */
    public void addPointListener(Object o) {
        listeners.add(o);
    }

    public void removePointListener(Object o) {
        listeners.remove(o);
    }

    /**
     * Set the interval between point updates
     * @param delay The time between point updates in milliseconds
     */
    public void setDelay(long delay) {
        this.delay = delay;
    }

    /**
     * @return The time between point updates in milliseconds
     */
    public long getDelay() {
        return delay;
    }

    /**
     * THis is called by an internal thread. It updates the point by doing a
     * reandom walk progressive vector
     */
    private void runThis() {
        while (isRunning) {
            double x = Math.floor(Math.random() * 3) - 1;
            double y = Math.floor(Math.random() * 3) - 1;
            point.setX(point.getX() + x);
            point.setY(point.getY() + y);
            notifyListeners();
            System.out.println("dx = " + x + ", dy = " + y);
            System.out.println("Current Position: x = " + point.getX() + ", y = " + point.getY());
            try {
                Thread.sleep(delay);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
    }

    private void notifyListeners() {
        if (listeners != null) {
            Iterator iterator = listeners.iterator();
            while(iterator.hasNext()) {
                IPointListener listener = (IPointListener) iterator.next();
                try {
                    listener.update(point);
                } catch( Exception e ) {
                    System.err.println( "Failed update call to " + listener );
                    System.err.println( "  " + e.getMessage() );
                    e.printStackTrace();
                    removePointListener(listener);
                }
            }
        }

    }

    public static void main(String[] args) {

        RandomWalk rw = null;
        switch (args.length) {
        case 0:
            rw = new RandomWalk();
            break;
        case 1:
            rw = new RandomWalk(Long.parseLong(args[0]));
            break;
        }
        if (rw != null) {
            rw.start();
        } else {
            System.out.println("Unable to start RandomWalk!!");
        }
    }

    private volatile long delay; // Update delay in seconds
    private volatile Point point = new Point(0D, 0D); // Position
    private HashSet listeners = new HashSet(); // Objects to alert when a new point is published
    private Thread walkThread; // Internal thread for creating updates
    private volatile boolean isRunning = true; // False will stop the walkThread
}