package org.mbari.oasis;

import java.io.*;
import java.util.*;
import org.mbari.io.*;
import org.mbari.util.*;
import ucar.nc2.*;
import ucar.ma2.*;

/**
 * Constructs a netCDF file from pCO2 data files created by 'extract' (i.e.  OASIS)<br>
 *
 * This class contructs an archive of pCO2 data from daily files extracted from OASIS
 * downloads. It can be used in other Java programs or run from the command line.
 * @see OasisNetcdfReader
 *
 */
public class Pco2Coards {

    /**
     * @param mooring the name of the mooring (ex. M1, M2, EP1, Ep2, etc.)
     * @param infile The pco2 data file (from oasis)
     * @param calfile the path and name of the mooring.properties file.
     * @param outfile The name fo the netcdf archive to be created.
     */
    public Pco2Coards(String mooring, String infile, String calfile, String outfile)
      throws IOException {
        this(mooring, new File(infile), new File(calfile), new File(outfile));
    }

    public Pco2Coards(String mooring, File infile, File calfile, File outfile)
      throws IOException {
        this.mooring = mooring;
        this.infile = new OasisPco2Reader(infile);
        this.calfile = calfile;
        this.outfile = outfile;
        this.makeNetcdf();
    }

    public static void main(String[] args) {
        if (args.length != 4) {
         System.out.println("\nUsage: org.mbari.oasis.Pco2Coatds <mooring> <input file> <properties file>  <output file>\n");
         System.out.println("\nInputs: <mooring>         = mooring id (ex. M1 or M2)");
         System.out.println("        <input file>      = pCO2 data file (ex. /oasis/m1/data/pco2)");
         System.out.println("        <properties file> = mooring properties file (ex. /oasis/cfg/mooring.properties)");
         System.out.println("        <output file>     = name of the Coards netcdf file to create");
         System.exit(0);
      }

      try {
         Pco2Coards oasisPco2Netcdf1 = new Pco2Coards(args[0], args[1],
           args[2], args[3]);
      } catch (IOException e) {
         System.out.println(" Error: " + e.getMessage());
         e.printStackTrace();
      }

    }


    private void makeNetcdf() throws IOException {

        //////////////////////////
        // Load the mooring properties file to get at and lon
        Properties mooringProperties = new Properties();
        BufferedInputStream mooringPropertiesStream = new BufferedInputStream(
            new FileInputStream(this.calfile));
        mooringProperties.load(mooringPropertiesStream);
        mooringPropertiesStream.close();
        float[] lat = {Float.parseFloat(mooringProperties.getProperty(mooring +
            ".latitude", "-999"))};
        float[] lon = {Float.parseFloat(mooringProperties.getProperty(mooring +
            ".longitude", "-999"))};
        float[] depth = {Float.parseFloat(mooringProperties.getProperty(mooring +
            ".waterdepth", "-999"))};


        //////////////////////////
        // Create the schema
        NetcdfFileWriteable ncfile = new NetcdfFileWriteable();
        ncfile.setName(outfile.getCanonicalPath());

        ncfile.addGlobalAttribute("conventions", "MBARI/timeSeries/mooring/pco2");
        ncfile.addGlobalAttribute("creationDate", new Date().toString());
        ncfile.addGlobalAttribute("lastModified", new Date().toString());
        ncfile.addGlobalAttribute("mooring", this.mooring.toUpperCase());
        ncfile.addGlobalAttribute("description", "pCO2 data");
        ncfile.addGlobalAttribute("waterdepth", depth);
        ncfile.addGlobalAttribute("keywords", "pCO2, mooring");
        ncfile.addGlobalAttribute("instrumentType", "pCO2 Can");

        Dimension latDim     = ncfile.addDimension("latitude", 1);
        Dimension lonDim     = ncfile.addDimension("longitude", 1);

        ncfile.addVariable("latitude", float.class, new Dimension[] {latDim});
        ncfile.addVariableAttribute("latitude", "long_name", "Latitude");
        ncfile.addVariableAttribute("latitude", "units", "degrees_north (+N/-S)");

        ncfile.addVariable("longitude", float.class, new Dimension[] {lonDim});
        ncfile.addVariableAttribute("longitude", "long_name", "Longitude");
        ncfile.addVariableAttribute("longitude", "units", "degrees_east (+E/-W)");

        for (int i = 0; i < TYPES.length; i++) {
            double[] time;
            try {
                time = infile.getTime(TYPES[i]);
            } catch (IllegalArgumentException e) {
                continue;
            }

            int[]    order = MathUtil.uniqueSort(time);

            String timeName =  "time";
            Dimension timeDim = ncfile.addDimension(timeName, -1);

            Dimension[] dim3 = {timeDim, latDim, lonDim};

            System.out.println("Adding: " + timeName);
            ncfile.addVariable(timeName, double.class, new Dimension[] {timeDim});
            ncfile.addVariableAttribute(timeName, "long_name", "time GMT" + POSTFIX[i]);
            ncfile.addVariableAttribute(timeName, "units", "seconds since 01 Jan 1970 00:00:00 GMT");

            if (TYPES[i].equalsIgnoreCase("d")) {
                String varName = "pco2";
                System.out.println("Adding: " + varName);
                ncfile.addVariable(varName, float.class, dim3);
                ncfile.addVariableAttribute(varName, "long_name", "calibrated pCO2");
                ncfile.addVariableAttribute(varName, "units", "parts per million");
                ncfile.addVariableAttribute(varName, "_FillValue", new Float(Float.NaN));
                ncfile.addVariableAttribute(varName, "missing_value", new Float(Float.NaN));

                varName = "int_pco2_c";
                System.out.println("Adding: " + varName);
                ncfile.addVariable(varName, float.class, dim3);
                ncfile.addVariableAttribute(varName, "long_name",
                  "pCO2 of calibration samples interpolated to raw sample times");
                ncfile.addVariableAttribute(varName, "units", "parts per million");
                ncfile.addVariableAttribute(varName, "_FillValue", new Float(Float.NaN));
                ncfile.addVariableAttribute(varName, "missing_value", new Float(Float.NaN));
            }

            for (int j = 0; j < OasisPco2Reader.COLUMNS.length; j++) {

                String varName = OasisPco2Reader.COLUMNS[j] + "_" + TYPES[i].toLowerCase();
                System.out.println("Adding: " + varName);
                ncfile.addVariable(varName, float.class, dim3);
                ncfile.addVariableAttribute(varName, "long_name", OasisPco2Reader.LONG_NAME[j] +
                  POSTFIX[i]);
                ncfile.addVariableAttribute(varName, "units", OasisPco2Reader.UNITS[j]);
                ncfile.addVariableAttribute(varName, "_FillValue", new Float(Float.NaN));
                ncfile.addVariableAttribute(varName, "missing_value", new Float(Float.NaN));
            }

        }

        // Create the file
        ncfile.create();

        ncfile.write("latitude", ArrayAbstract.factory(lat));
        ncfile.write("longitude", ArrayAbstract.factory(lon));

        // Write the data
        for (int i = 0; i < TYPES.length; i++) {
            double[] timeRaw;
            try {
                timeRaw = infile.getTime(TYPES[i]);
            } catch (IllegalArgumentException e) {
                continue;
            }

            int[]    order      = MathUtil.uniqueSort(timeRaw);
            double[] timeSorted = MathUtil.orderVector(timeRaw, order);

            double[] timeSec = new double[timeSorted.length];
            for (int j = 0; j < timeSorted.length; j++) {
                timeSec[j] = timeSorted[j]/1000D;
            }

            String timeName = "time";
            System.out.println("Writing: " + timeName);
            ncfile.write(timeName,  ArrayAbstract.factory(timeSec));

            if (TYPES[i].equalsIgnoreCase("d")) {
                double[] timeCal       = infile.get("time_c");
                int[]    orderCal      = MathUtil.uniqueSort(timeCal);
                double[] timeCalSorted = MathUtil.orderVector(timeCal, orderCal);

                double[] pco2DataSorted = MathUtil.orderVector(infile.get("pco2_d"), order);
                double[] calDataSorted  = MathUtil.orderVector(infile.get("pco2_c"), orderCal);
                double[] calDataInterp  = MathUtil.interpLinear(timeCalSorted,
                  calDataSorted, timeSorted);

                float[][][] pco2 = new float[timeSec.length][1][1];
                float[][][] calI = new float[timeSec.length][1][1];
                for (int k = 0; k < timeSec.length; k++) {
                    calI[k][0][0] = (float) calDataInterp[k];
                    pco2[k][0][0] = (float) (pco2DataSorted[k] - calDataInterp[k]);
                }
                System.out.println("Writing: pco2");
                ncfile.write("pco2", ArrayAbstract.factory(pco2));
                System.out.println("Writing: int_pco2_c");
                ncfile.write("int_pco2_c", ArrayAbstract.factory(calI));
            }

            for (int j = 0; j < OasisPco2Reader.COLUMNS.length; j++) {

                String varName = OasisPco2Reader.COLUMNS[j] + "_" + TYPES[i];
                System.out.println("Writing: " + varName);
                double[] dataSorted = MathUtil.orderVector(infile.get(varName), order);

                float[][][] data = new float[order.length][1][1];
                for (int k = 0; k < order.length; k++) {
                    data[k][0][0] = (float) dataSorted[k];
                }

                ncfile.write(varName.toLowerCase(), ArrayAbstract.factory(data));
            }
        }

        // Close
        ncfile.close();


    }

    private OasisPco2Reader infile;
    private File outfile;
    private File calfile;
    private String mooring;
    private static final String[] TYPES = {"d"};
    private static final String[] POSTFIX  = {""};
    private static final String[] PREFIX  = {""};

}