
/**
 * Title:        HobiLabs data processing<p>
 * Description:  <p>
 * Copyright:    Copyright (c) Brian Schlining<p>
 * Company:      MBARI<p>
 * @author Brian Schlining
 * @version 1.0 19 Oct 2000
 */
package org.mbari.oasis;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Arrays;
import java.util.Date;
import java.util.Properties;
import org.mbari.hobilabs.HS2CalibrationFileReader;



/**
 * Used to process an HS2 data file or directory full of HS2 data files into a
 * single text file. The data fiels are assumed to be in the standard oasis
 * text format. The usage is:<br><br>
 *
 * OasisHS2Processer op = new OasisHS2Processer( ... );<br>
 * op.process();<br><br>
 *
 * Its also meant to be used as a stand-alone command line program.
 */
public class OasisHS2Processer {

   /**
    * @param src A HS2 datafile or a directoy containg only hs2 datafiles
    * @param calFile The name of the calibration file
    * @param destFile The name fo the file to write the calibrated data to.
    */
   public OasisHS2Processer(String src, String calFile, String destFile) {

      this.dst = destFile;

      try {

         this.cal = new HS2CalibrationFileReader(calFile).getProperties();

         File fid = new File(src);   // Open fid
         if (fid.isDirectory()) {     // If input arg is a directory
            String[] fileList = fid.list();  // Loop through all files in the directory
            Arrays.sort(fileList, String.CASE_INSENSITIVE_ORDER); // Sort the files
            this.files = new File[fileList.length];               // Initialize files array
            for (int i = 0; i < fileList.length; i++) {           // Loop through
                this.files[i] = new File(fid, fileList[i]);
            }
         } else {                     // Otherwise its a single data file
            files = new File[1];
            files[0] = fid;
         }

      } catch (IOException e) {
         e.printStackTrace();
      }
   }

   /**
    * Method called to read the calibration file and store it as a properties object.
    */
   public void process() {

      try {

         // Open the output file for writing
         BufferedWriter bos = new BufferedWriter(new FileWriter(this.dst));

         // Add the header
         bos.write("# File created on " + new Date() + "\n");
         bos.write("#");
         bos.write(" HS2 Data Header\n#\n# The overall format is:\n");
         bos.write("# Note: The data may not contain all the columns specified in " +
            "this header.\n# However, the data is always stored in the order thats " +
            "specified in \n# the header. If only four columns of data are present " +
            "then use the \n# first four column titles listed below.\n");
         bos.write("#\n# time");
         for (int i = 1; i < 9; i++) {
            String name = cal.getProperty("Channel" + i + ".Name");
            if (name != null) {
               if (!name.equalsIgnoreCase("null")) {
                  bos.write(" " + name);
               }
            }
         }
         bos.write("\n#\n# Time = epic seconds (seconds since Jan 01, 1970 00:00:00)\n");
         bos.write("# Missing_value = -999\n");

         for (int i = 0; i < this.files.length; i++) {
            OasisHS2Parser ofp = new OasisHS2Parser(files[i], bos, this.cal);
         }

         bos.close();

      } catch (IOException e) {
         e.printStackTrace();
      }
   }

   /**
    * arg[0] = directory or file to parse
    *  If it's a directory the processer will loop through all files in the directory.
    *  and write the result to the output file. Otherwise it will only prcess the file specified
    *
    * arg[1] = the calibration file to use. Thi sfile should be in a properties style format
    *
    * arg[2] =  The name of the file to create
    */
   public static void main(String[] args) {
      if (args.length != 3) {
         System.out.println("Usage: OasisHS2Proscesser <input directory or file> <calibration file> <output file>");
         System.exit(0);
      }
      OasisHS2Processer op = new OasisHS2Processer(args[0], args[1], args[2]);
      op.process();
   }

   File[] files;
   Properties cal;
   String dst;
}