
/**
 * Title:        HobiLabs data processing<p>
 * Description:  <p>
 * Copyright:    Copyright (c) Brian Schlining<p>
 * Company:      MBARI<p>
 * @author Brian Schlining
 * @version 1.0
 */
package org.mbari.oasis;

import java.io.*;
import java.text.*;
import java.util.*;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mbari.hobilabs.*;


public class OasisHS2Parser {
    
    public OasisHS2Parser(File inFile_, Writer out_, Properties cal_) {
        this.inFile = inFile_;
        this.out = out_;
        this.cal = cal_;
        
        log.debug("Using the following files:\n\tInput = " + inFile + "\n\tOutput = " + out + "\n\tCalibration = " + cal_);
        
        nf.setMaximumFractionDigits(8);
        nf.setMinimumFractionDigits(8);
        nf.setMinimumIntegerDigits(1);
        
        this.parse();
    }
    
    private void parse() {
        try {
            BufferedReader in = new BufferedReader(new FileReader(inFile));
            String s;
            while ((s = in.readLine()) != null) {
                if (s.startsWith("*D")) {
                    //System.out.println(s);
                    
                    try {
                        HSDPacket p = HSDParser.parse(s);
                        this.calibrate(p);
                    } catch (IOException e) {
                        e.printStackTrace();
                    } catch (NumberFormatException e) {
                        e.printStackTrace();
                    } catch (IllegalArgumentException e) {
                        System.out.println("Invalid string");
                    }
                }
            }
            
        } catch (IOException e) {
            e.printStackTrace();
        }
        
    }
    
    private void calibrate(HSDPacket p) throws IOException {
        boolean newLine = true;
        for (int i = 0; i < 8; i++) {
            if (p.getGain(i) != 0) {
                int j = i + 1;
                double T = ((double) p.getTemperature()) / 5D - 10D;
                
                double bb = Double.NaN;
                try {
                    double Tc = getValue("Channel" + j + ".TempCoeff");
                    double Tn = getValue("TempNom");
                    double mu = getValue("Channel" + j + ".Mu");
                    double beta = getValue("Channel" + j + ".Beta2Bb");
                    double gain = getValue("Channel" + j + ".Gain" + p.getGain(i));
                    double Rn = getValue("Channel" + j + ".RNominal");
                    
                    double bbu = (1 - Tc*T/Tn)*mu*beta*((double) p.getSnorm(i))/(gain*Rn);
                    
                    double B = getValue("Channel" + j + ".Bb2c");
                    double c = bbu * B;
                    
                    double sigma1 = getValue("Channel" + j + ".Sigma1");
                    double sigma2 = getValue("Channel" + j + ".Sigma2");
                    double sigma3 = getValue("Channel" + j + ".Sigma3");
                    double sigmaC = sigma1 + sigma2 * c + sigma3 * Math.pow(c, 2);
                    
                    bb = bbu * sigmaC;
                    
                } catch (Throwable t) {
                    log.error("Failed to get calibration info", t);
                    continue;
                }
                
                if (newLine) {
                    out.write(p.getTime()/1000 + "");
                    newLine = false;
                }
                
                // Does quality control. It's best to keep this turned off
            /*
            if (bbu > 1) {
               bbu = -999D;
            }
            if (bb > 1) {
               bb = -999D;
            }
             */
                
                out.write("\t" + nf.format(bb));
            }
        }
        out.write("\n");
    }
    
    /**
     * Retrieves the named property from the calibration properties file as a
     * double.
     */
    private double getValue(String propertyName) throws Throwable {
        double value = 0D;
        try {
            value = Double.parseDouble(cal.getProperty(propertyName));
        } catch (Exception e) {
            log.error("Unable to retrieve calibration property '" + propertyName + "'", e);
            throw e;
        }
        return value;
    }
    
    
    
    
    private File inFile;
    private Writer out;
    private Properties cal;
    private NumberFormat nf = NumberFormat.getNumberInstance();
    private static final Log log = LogFactory.getLog(OasisHS2Parser.class);
    
}