package org.mbari.oasis;

import java.io.*;
import java.util.*;
import java.text.*;

/**
 * Combine files created by Bob Herlein's extract program into a single data
 * file. This version replaces CoallateOasisFiles and can be called as
 * a static method.
 *
 * This class is designed to read a directory contianing files using 'yyddd' or
 * 'yyyyddd' naming schemes and combine them into a single file. It grabs the
 * header of the first file in the sequence and adds it to the output file.
 * The file created with this should contain identical data as the source files
 * with the exception that a column of years is added as the last column of the
 * data file.
 *
 * @author Brian Schlining
 * @version 1.2.1, 19 Jul 1999
 */
public class OasisFileCoallator {

   // Dummy constructor
   private OasisFileCoallator() {}

   /**
    * Combine files created by <i>extract</i>. An additional column is added
    * to the end and contains the year, which is parsed from the file name.
    *
    * @param outfile The file to be created
    * @param inPath The location of the data files to be combined.
    * @exception IOException
    */
   public synchronized static final void coallate(File outFile, File inPath)
     throws IOException {

      String[] fileList;
      int      Year;
      Locale   locale    = Locale.US;
      Collator collator  = Collator.getInstance(locale);

      // Read the file list and write out to a file
      fileList           = inPath.list();
      fileList           = sortFiles(collator, fileList);
      BufferedWriter out = new BufferedWriter(new FileWriter(outFile));

      for (int i = 0; i < fileList.length; i++) {

         File file = new File(inPath, fileList[i]);
         System.out.println(inPath + System.getProperty("file.separator") + fileList[i]);
         BufferedReader in = new BufferedReader(new FileReader(file));
         Year = getYear(fileList[i]);
         String line;
         while ((line = in.readLine()) != null) {
            if ((i != 0) & line.startsWith("#")) {
            }
            else if ((i == 0) & line.startsWith("#")) {
               out.write(line);
               out.newLine();
            }
            else {
               out.write(line);
               out.write(" ");
               out.write(Integer.toString(Year));
               out.newLine();
            }
         }
         out.flush();
         in.close();
      }
      out.close();
   }

   /**
    * Combine files created by <i>extract</i>. An additional column is added
    * to the end and contains the year, which is parsed from the file name. This
    * method assumes the files ae located in the current user directory
    *
    * @param outfile The file to be created
    * @exception IOException
    */
   public static final void coallate(File out) throws IOException {
      File in  = new File(System.getProperty("user.dir")); // pwd
      OasisFileCoallator.coallate(out, in);
   }

   /**
    * Combine files created by <i>extract</i>. An additional column is added
    * to the end and contains the year, which is parsed from the file name. This
    * method assumes the files ae located in the current user directory
    *
    * @param outfile The file to be created
    * @exception IOException
    */
   public static final void coallate(String outfile) throws IOException {
      File out = new File(outfile);
      File in  = new File(System.getProperty("user.dir")); // pwd
      OasisFileCoallator.coallate(out, in);
   }

   /**
    * Combine files created by <i>extract</i>. An additional column is added
    * to the end and contains the year, which is parsed from the file name.
    *
    * @param outfile The file to be created
    * @param inPath The location of the data files to be combined.
    * @exception IOException
    */
   public static final void coallate(String outfile, String inFilepath)
    throws IOException {
      File out  = new File(outfile);
      File in = new File(inFilepath);
      OasisFileCoallator.coallate(out, in);
   }



   /**
    * Extract the year from filenames of types 'yyddd' or 'yyyydd'.
    *
    * @param file The name of the file
    * @return The year represented in the file name
    */
   public static int getYear(String file) {

      String Year;
      if (file.length() == 5) {
         char[] buf = new char[2];
         file.getChars(0,2,buf,0);
         Year = new String(buf);
         return (Integer.parseInt(Year) + 1900);
      }
      else if (file.length() == 7) {
         char[] buf = new char[4];
         file.getChars(0,4,buf,0);
         Year = new String(buf);
         return Integer.parseInt(Year);
      }
      else {
         return 0;
      }
   }

   /**
    * Sorts the array of filename by name
    */
   public static String[] sortFiles(Collator collator, String[] strArray) {
      String tmp;
      for (int i = 0; i < strArray.length; i++) {
         for (int j = i + 1; j < strArray.length; j++) {
            if (collator.compare(strArray[i], strArray[j]) > 0) {
               tmp = strArray[i];
               strArray[i] = strArray[j];
               strArray[j] = tmp;
            }
         }
      }
      return strArray;
   }

   /**
    * Allows the user to call this class from the command line
    *
    * Type 'java org.mbari.oasis.OasisFileCoallator' for help.
    */
   public static void main(String[] args) {

      // Check arguments
      if (args.length == 0 || args.length > 2) {
         System.out.println("Usage: java org.mbari.oasis.OasisFileCoallator OutFile <InFilePath>");
         System.out.println("Inputs: ");
         System.out.println("   OutFile    = name of the file to write output to");
         System.out.println("   InFilePath = Optional argument. Specifies directory");
         System.out.println("                containing source data files");
         System.out.println(" ");
         System.out.println("Note: InFilePath should only contain data files");
         System.exit(0);
      } else {
         if (args.length == 1) {
            try {
               OasisFileCoallator.coallate(args[0]);
            } catch(Exception e) {
               System.out.println(e);
               e.printStackTrace();
            }
         } else if (args.length == 2) {
            try {
               OasisFileCoallator.coallate(args[0], args[1]);
            } catch(Exception e) {
               System.out.println(e);
               e.printStackTrace();
            }
         }
      }
   }


}