package org.mbari.oasis;

import java.io.*;
import java.lang.reflect.*;
import java.util.*;

/**
 * OMSVariableReader (OMS = Oasis Mooring Specprr) is used to get Variable
 * information about the configuration that is stored in the configuration files
 *
 * @author Brian Schlining
 * @version 29 Sep 1999
 */
public class OMSVariableReader implements Enumeration {

   /////////////////////////////////////////
   // Constructors
   /**
    * OMSVariableReader reads in the Variables in the main configuration
    * file
    *
    * @param filename The configuration file containing the mooring attributes
    */
   public OMSVariableReader(String filename) {
      this.cfgfile = filename;
      File file = new File(filename);
      this.init(file);
   }

   /**
    * OMSVariableReader reads in the Variables in the main configuration
    * file
    *
    * @param filename The configuration file containing the mooring attributes
    */
   public OMSVariableReader(File file) {
      this.cfgfile = file.toString();
      this.init(file);
   }

   /////////////////////////////////////////
   // Public Methods

   /**
    * Count the number of variables in the file
    */
   public int countVariables() {
      return Array.getLength(this.v);
   }

   /**
    * @return The name of the file used to create an instance of
    * MooringAttributeReader
    */
   public String getFilename() {
      return this.cfgfile;
   }

   /**
    * Get a single variable in the file
    *
    * @param i The index of the variable to return (Zero based indices)
    * @return MooringAttributeReader$MooringVariable
    * @exception ArrayIndexOutOfBoundsException
    */
   public OMSVariable getVariable(int i) {
      return this.v[i];
   }

   /**
    * Get all the variables in the file
    *
    * @return an array of MooringVariables
    */
   public OMSVariable[] getVariables() {
      return this.v;
   }

   /**
    * Enumeration interface
    *
    * @return true if more elements are available; false otherwise
    */
   public boolean hasMoreElements() {
      boolean ok = false;
      if (this.index < this.countVariables()) {
         ok = true;
      }
      return ok;
   }

   /**
    * Enumeration interface
    *
    * @return The next element
    */
   public Object nextElement() {
      if (this.hasMoreElements()) {
         this.index++;
         return this.v[this.index - 1];
      } else {
         return null;
      }
   }

   /////////////////////////////////////////
   // Private Methods
   private void init(File file) {
      try {
         this.count(file);
         this.parse(file);
      } catch (IOException e) {
      }
   }

   private void count(File file) throws IOException {
      int nVar = 0;
      String str;
      BufferedReader in = new BufferedReader(new FileReader(file));
      while ((str = in.readLine()) != null) {
         if (str.length() > 3) {
            if (str.startsWith("#")) {
               if (str.substring(1,2).equals("!")) {
                  // Ignore Attributes
               } else if (str.substring(1,2).equals(":")) {
                  nVar++;
               }
            }
         }
      }
      in.close();
      this.v = new OMSVariable[nVar];
   }

   private void parse(File file) throws IOException {
      String str;
      BufferedReader in = new BufferedReader(new FileReader(file));
      while ((str = in.readLine()) != null) {
         if (str.length() > 3) {
            if (str.startsWith("#")) {
               if (str.substring(1,2).equals(":")) {
                  this.readVariable(str);
               }
            }
         }
      }
      in.close();
   }

   private void readVariable(String str) {
      StringTokenizer st = new StringTokenizer(str);

      int     n      = st.countTokens() - 2;  // First Token is '#:', second is name
      float[] values = new float[n];
      String  buf    = st.nextToken(); // '#:'
      String  name   = st.nextToken(); // name
      int     i      = 0;
      while (st.hasMoreTokens()) {
         values[i] = new Float(st.nextToken()).floatValue();
         i++;
      }

      this.v[this.vIndex] = new OMSVariable(name, values);
      this.vIndex++;
   }


   /////////////////////////////////////////
   // Class Variables
   private String cfgfile;
   private OMSVariable[] v;
   private int vIndex = 0,
               index = 0;

   public static void main(String[] args) {
      OMSVariableReader ovr = new OMSVariableReader("g:/eqpac/cfg/ep1.cfg");
      System.out.println();
   }

}