package org.mbari.oasis;

import java.io.*;
import java.text.*;
import java.util.*;
import org.mbari.util.*;

/**
 * Utilties used for netcdf file construction
 *
 * @author Brian Schlining
 * @version 29 Sep 1999
 */
public class OMSUtil {

   private OMSUtil() {
   }

   /**
    * Assign a long descriptive name based on the short name found in the
    * calibration files.
    *
    * @param  name      String which is to be the name of this Attribute
    * @return longName  String describing the short name
    */
   public static String getLongName(String name) {

      String longName;
      name = name.toLowerCase();

      if (name.startsWith("ed")) {
         longName = "Downwelled irradiance at " + name.substring(2) + " nm";
      } else if (name.equals("fluor")) {
         longName = "Fluorescence";
      } else if (name.equals("gnd")) {
         longName = "Ground voltage";
      } else if (name.startsWith("lu")) {
         longName = "Upwelled radiance at " + name.substring(2) + " nm";
      } else if (name.equals("mcp")) {
         longName = "Downwelled irradiance at 490 nm";
      } else if (name.equals("par")) {
         longName = "Downwelled photosynthetically active radiation";
      } else if (name.equals("pitch")) {
         longName = "Pitch";
      } else if (name.equals("roll")) {
         longName = "Roll";
      } else if (name.equals("temp")) {
         longName = "Temperature";
      } else if (name.equals("latitude")) {
         longName = "Latitude";
      } else if (name.equals("longitude")) {
         longName = "Longitude";
      } else if (name.equals("bankdepth")) {
         longName = "Depths of the instruments by bank";
      } else {
         longName = "null";
      }

      return longName;
   }

   /**
    * Assign each variable in an OMSSensor array a unique name
    *
    * @param name The short descriptive name of a sensor variable
    * @param bank The bank number of the sensor
    * @param channel The channel number of the sensor
    * @return A unique name for the sensor created by comining the name, bank and channel
    */
   public static String getUniqueName(String name, int bank, int channel) {
      return name + "_" + bank + "_" + channel;
   }

   /**
    * Assign each variable in an OMSSensor array a unique name
    *
    * @param s An OMSSensor object
    * @return A unique name for the sensor created by comining the name, bank and channel
    */
   public static String getUniqueName(OMSSensor s) {
      return s.getName() + "_" + s.getBank() + "_" + s.getChannel();
   }

   /**
    * Assign each variable in an OMSSensor array a unique name that that can
    * be used for sorting
    *
    * @param s An OMSSensor object
    * @return A unique name for the sensor created by comining the name, bank and channel
    */
   public static String getSortableName(OMSSensor s) {
      DecimalFormat df = new DecimalFormat("00");
      return  df.format((long) s.getBank()) + "_" + df.format((long) s.getChannel())  + "_" + s.getName();
   }

   /**
    * Converts a file name like '99123' or '1999123' to a Date object
    *
    * @param filename The file name of the data file
    * @return A date object corresponding to the date of the filename
    */
   public static Date getDateFromName(String filename) {
      String name = new File(filename).getName();
      return DateConverter.doyToDate(OMSUtil.getDay(name),
       OMSUtil.getYear(name));
   }


   /**
    * Extract the year from filenames of types 'yyddd' or 'yyyyddd'.
    *
    * @param file The name of the file
    * @return The year represented in the file name
    */
   public static int getYear(String filename) {
      String name = new File(filename).getName();
      String Year;
      if (name.length() == 5) {
         char[] buf = new char[2];
         name.getChars(0,2,buf,0);
         Year = new String(buf);
         return (Integer.parseInt(Year) + 1900);
      }
      else if (name.length() == 7) {
         char[] buf = new char[4];
         name.getChars(0,4,buf,0);
         Year = new String(buf);
         return Integer.parseInt(Year);
      }
      else {
         return 0;
      }
   }

   /**
    * Extract the day from filenames of types 'yyddd' or 'yyyyddd'.
    *
    * @param file The name of the file
    * @return The year-day represented in the file name
    */
   public static int getDay(String filename) {
      String name = new File(filename).getName();
      String Day;
      if (name.length() == 5) {
         char[] buf = new char[3];
         name.getChars(2,5,buf,0);
         Day = new String(buf);
         return (Integer.parseInt(Day));
      }
      else if (name.length() == 7) {
         char[] buf = new char[3];
         name.getChars(4,7,buf,0);
         Day = new String(buf);
         return (Integer.parseInt(Day));
      }
      else {
         return 0;
      }
   }


   /**
    * Sorts an array of Strings alphabetically
    *
    * @param strArray An array of strings
    * @return The array of string sorted alphabetically
    */
   public static String[] sortFiles(String[] strArray) {

      Locale   locale    = Locale.US;
      Collator collator  = Collator.getInstance(locale);
      String tmp;
      for (int i = 0; i < strArray.length; i++) {
         for (int j = i + 1; j < strArray.length; j++) {
            if (collator.compare(strArray[i], strArray[j]) > 0) {
               tmp = strArray[i];
               strArray[i] = strArray[j];
               strArray[j] = tmp;
            }
         }
      }
      return strArray;
   }

   public static float getWavelength(OMSSensor s) {
      return OMSUtil.getWavelength(s.getName());
   }

   public static float getWavelength(String name) {
      float lambda = -999f;
      name = name.toLowerCase();
      if (name.startsWith("ed")) {
         lambda = Float.parseFloat(name.substring(2));
      } else if (name.startsWith("lu")) {
         lambda = Float.parseFloat(name.substring(2));
      } else if (name.equals("mcp")) {
         lambda = 490f;
      }
      return lambda;
   }

   /**
    *  Returns a short symbol used to represent the data in equations. For example
    *  Ed490 in Channel 0 will return "Es" as the symbol
    *
    *  @param s An OMSSensor object
    *  @return symbol A string of the symbol corresponding to the data collected
    *  by the sensor.
    */
   public static String getDataSymbol(OMSSensor s) {
      String symbol;
      String name = s.getName().toLowerCase();

      if (name.startsWith("ed")) {
         switch(s.getBank()) {
            case 0:
               symbol = "Es";
               break;
            default:
               symbol = "Ed";
               break;
         }
      } else if (name.equals("fluor")) {
         symbol = "F";
      } else if (name.equals("gnd")) {
         symbol = "V";
      } else if (name.startsWith("lu")) {
         symbol = "Lu";
      } else if (name.equals("mcp")) {
         symbol = "Ed";
      } else if (name.equals("par")) {
         symbol = "PAR";
      } else if (name.equals("pitch")) {
         symbol = "null";
      } else if (name.equals("roll")) {
         symbol = "null";
      } else if (name.equals("temp")) {
         symbol = "T";
      } else if (name.equals("latitude")) {
         symbol = "Lat";
      } else if (name.equals("longitude")) {
         symbol = "Lon";
      } else if (name.equals("bankDepth")) {
         symbol = "Z";
      } else if (name.startsWith("depth")) {
         symbol = "Z";
      } else {
         symbol = "null";
      }

      return symbol;
   }

}