package org.mbari.oasis;

import java.io.*;
import java.util.*;

/**
 * OMSSensor (Oasis Mooring Specprr = OMS)
 *
 * @author Brian Schlining
 * @version 06 Jul 1999
 */

public class OMSSensor implements Serializable {


   /////////////////////////////////////////
   // Constructors
   /**
    * Constructor creating an OMSSensor object from a line from a Calibration
    * file
    *
    * @param s  String of calibration data read directly from the cal file
    */
   public OMSSensor(String s) {
      this.parse(s);
   }

   /////////////////////////////////////////
   // Accessors
   /**
    * @return the bank number of the sensor
    */
   public int getBank() {
      return this.bank;
   }

   /**
    * @return the channel number of the sensor
    */
   public int getChannel() {
      return this.channel;
   }

   /**
    * @return the calibration equation type:
    *	 Type 0:       not converted or reported
    *	 Types 1 - 4:  value = (volts - c) / b  (a is unused)
    *	 Type 5:       value = a + (b * volts) + (c * volts^2)
    *  Type 6:       value = a * (volts - c) / b
    *  Type 7:       value = a * b * (volts - c)
    */
   public int getType() {
      return this.type;
   }

   /**
    * @return the name of the data collected by the sensor
    */
   public String getName() {
      return this.name;
   }

   /**
    * @return the calibration coefficients
    */
   public double[] getCoef() {
      return this.coef;
   }

   /**
    * @return the min and max of the valid range
    * Note: The use of this for processing is not currently implemented
    */
   public double[] getMinMax() {
      return this.minMax;
   }

   /**
    * @return the data units
    */
   public String getUnits() {
      return this.units;
   }

   /////////////////////////////////////////
   // Private Methods
   /**
    *  Parse a line of calibration data from an Oasis calibration file
    *
    * @param s
    */
   private void parse(String s) {

      StringTokenizer st   = new StringTokenizer(s);
      String[]        data = new String[10];
      int             i    = 0;

      try {
         while(st.hasMoreTokens()) {
            data[i] = st.nextToken();
            i++;
         }

         this.bank      = new Integer(data[0]).intValue();
         this.channel   = new Integer(data[1]).intValue();
         this.type      = new Integer(data[2]).intValue();
         this.name      = new String(data[3]);
         this.coef[0]   = new Double(data[4]).doubleValue();
         this.coef[1]   = new Double(data[5]).doubleValue();
         this.coef[2]   = new Double(data[6]).doubleValue();
         this.minMax[0] = new Integer(data[7]).intValue();
         this.minMax[1] = new Integer(data[8]).intValue();
         this.units     = new String(data[9]);

      } catch(Exception e) {
         System.err.println(e);
      }
   }


   /////////////////////////////////////////
   // Class variables
   private int      bank,
                    channel,
                    type;

   private String   name,
                    units;

   private double[] coef   = new double[3],
                    minMax = new double[2];
}