package org.mbari.oasis;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.text.Collator;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Locale;
import java.util.StringTokenizer;



/**
 * Setup a netCDF schema for specprr data collected through Oasis cans.
 * This class uses info contained in the cfg file to assign global attributes.
 * <BR><BR>
 * OMSSchemaConstructor reads through the main cfg file (i.e. ep1.cfg, m1.cfg, etc)
 * and creates a master template containing all the measured variables found in
 * the configruation files.
 *
 * @author Brian Schlining
 * @version 06 Jul 1999
 */
public class OMSSchemaConstructor extends OMSAbstractSchemaConstructor {

   /////////////////////////////////////////
   // Constructors
   /**
    * Create a netCDF schema using information in the configuration
    * files. This will create a schema using information in all the files in the
    * configuration history as well.
    *
    * @param file The configuration file to read
    */
   public OMSSchemaConstructor(String fileS) {
      super(new File(fileS));

   }

   public OMSSchemaConstructor(File file) {

      super(file);
   }

   protected OMSSensor[] getSensors(){
      // 1. Get start times for each specprr file in the cfg file
      Hashtable table = new Hashtable(151, 0.6f); // 60% load factor
      OMSConfigReader cr;
      OMSSensor[] s;
      try {
         // Add the default variables to the hashtable
         cr = new OMSConfigReader(cfgfile);
         s = cr.getConfiguration();
         for (int i = 0; i < s.length; i++) {
            table.put(OMSUtil.getUniqueName(s[i]), s[i]);
         }

         // Add the specprr history to the hashtable
         BufferedReader in = new BufferedReader(new FileReader(cfgfile));
         String str;
         while ((str = in.readLine()) != null) {
            //Process the string
            if (str.startsWith("9") || str.startsWith("1") || str.startsWith("2")) {
               StringTokenizer st = new StringTokenizer(str);
               if (st.countTokens() > 4) {                        // Make sure its long genough
                  double d1 = Double.parseDouble(st.nextToken()); //Date1
                  String buf = st.nextToken();                    // The '-' sign
                  double d2 = Double.parseDouble(st.nextToken()); //Date2
                  String type = st.nextToken();                   // is it a specprr file?
                  if (type.startsWith("specprr")) {               //If it is a string
                     cr = new OMSConfigReader(cfgfile, d1);
                     s = cr.getConfiguration();
                     for (int i = 0; i < s.length; i++) {
                        Object v = table.get(OMSUtil.getUniqueName(s[i]));
                        if (v == null) {
                           table.put(OMSUtil.getUniqueName(s[i]), s[i]);
                        }
                     }
                  }
               }
            }
         }
         in.close();
      } catch (Exception e) {
         // Deal with IO error here
      }
      // 2. Loop through each date using OMSConfigReader(filename, date)

      // 3. Store each unique variable in a hashtable
      OMSSensor[] sensor = new OMSSensor[table.size()];
      Enumeration elements = table.elements();
      int i = 0;
      while (elements.hasMoreElements()) {
         sensor[i] = (OMSSensor) elements.nextElement();
         i++;
      }

      sensor = this.sortFiles(sensor);

      return sensor;
   }

   /**
    * Sorts an array of Strings alphabetically
    *
    * @param strArray An array of strings
    * @return The array of string sorted alphabetically
    */
   public OMSSensor[] sortFiles(OMSSensor[] s) {

      Locale   locale    = Locale.US;
      Collator collator  = Collator.getInstance(locale);
      OMSSensor tmp;
      for (int i = 0; i < s.length; i++) {
         //System.out.println(OMSUtil.getSortableName(s[i]));
         for (int j = i + 1; j < s.length; j++) {
            if (collator.compare(OMSUtil.getSortableName(s[i]) , OMSUtil.getSortableName(s[j])) > 0) {
               tmp = s[i];
               s[i] = s[j];
               s[j] = tmp;
            }
         }
      }
      return s;
   }

}
