package org.mbari.oasis;

import java.io.*;
import java.util.*;
import org.mbari.io.*;
import org.mbari.util.*;
import ucar.multiarray.*;
import ucar.netcdf.*;

/**
 * OMSNetcdfAppender is designed to append data onto existing Oasis Mooring
 * specprr netcdf files. This version is aimed at appending the files from
 * Bob's extract program
 *
 * @author Brian Schlining
 * @version 04 Oct 1999
 */
public class OMSNetcdfAppender {

   /////////////////////////////////////////
   // Constructor
   public OMSNetcdfAppender(String cfgfile, String filename, String ncfile) throws IOException  {
      this.file = filename;
      this.nc = new NetcdfFile(ncfile, false);
      this.cfg = cfgfile;
      this.process();
      this.nc.close();
   }

   /////////////////////////////////////////
   // Public Methods
   public static void main(String[] args) {
      // Print out help if used incorrectly
      /*args = new String[3];
      args[0] = "g:/eqpac/cfg/ep1.cfg";
      args[1] = "d:/brian/temp/2s0data/users/brian/mooring/eqpac/ep1/specprr2/97001";
      args[2] = "d:/brian/temp/ep1test.nc";*/
      if (args.length != 3) {
         System.out.println();
         System.out.println("Usage: org.mbari.netcdf.OMSNetcdfAppender configfile datafile ncfile clobber");
         System.out.println();
         System.out.println("Inputs: configfile = name of the mooring configuration file");
         System.out.println("        datafile   = name of specprr data file");
         System.out.println("        ncfile     = name of netcdf file to be created");
         System.out.println();
      } else if (args.length == 3) { // Don't let files get clobbered without a fight
         try {
            File nc = new File(args[2]);
            if (nc.exists()) {
               OMSNetcdfAppender msn = new OMSNetcdfAppender(args[0], args[1], args[2]);
            } else {
               System.out.println(args[2] + " does not exist. Program aborted");
            }
         } catch (IOException e) {
            System.out.println("Unable to read or create one of the input files");
         }
      }
   }

   private void process() throws IOException {
      // *** Check the data to see if its already in the netcdf file ***
      Date     fileDate = OMSUtil.getDateFromName((this.file)),
               lastNcDate = new Date(DateConverter.serialDaysToUtc(1));
      Variable timeV    = nc.get("time");     // Get the time variable
      int[]    extent   = timeV.getLengths(); // i.e. length(timeV)
      int      n        = extent[0];
      int[]    lastRec  = new int[1];
      lastRec[0] = n - 1;                     // Index of the last record.
      if (n > 0) {                            // Do this if there is data in the nc file
         if (timeV.getDouble(lastRec) > 0 ) { // Do this if the last time exists

            // Get the date of the last reocrd in the file in millisec (UTC)
            lastNcDate = new Date((long) timeV.getDouble(lastRec)*1000);

            /*The method commented out here reads time values from the netcdf file
               in matlab's data format
            lastNcDate = new Date(DateConverter.serialDaysToUtc(
                                    timeV.getDouble(lastRec)));
            */
         }
      }


      if (fileDate.after(lastNcDate)) {       // Append data if the text files date is after
                                              // the last record in the netcdf file

         msg("Reading :" + this.file);

         // Read the data file
         FlatFileReader ffr = new FlatFileReader(this.file); // Read the data in the text file
         ffr.readFile();
         this.data = ffr.getData();                          // Store the data

         // Get the times from the data file
         int Year  = OMSUtil.getYear(this.file);        // Get the year from the text files name
         this.time = new double[ffr.countRows()];            // Create a time variable the same length as the data
         for (int j = 0; j < ffr.countRows(); j++) {         // Loop through each row of the data

            // Covnert decimal day of the year to UTC seconds
            this.time[j] = (double) DateConverter.doyToDate(this.data[j][0], Year).getTime()/1000;

            /* The methdo commented out here adds time values to the netcdf file
               in matlab's date format
            this.time[j] = DateConverter.utcToSerialDays(
                           DateConverter.doyToDate(this.data[j][0], Year).getTime());
            */
         }

         // Get the configuration used for the text data file
         OMSConfigReader cr = new OMSConfigReader(this.cfg,
                              Double.parseDouble(
                              new File(this.file).getName()));

         for (int r = 0; r < ffr.countRows(); r++) { // Loop through each row of data

            int[] ti = timeV.getLengths();
            timeV.setDouble(ti, this.time[r]);       // Add the time for the current row of data ot the netcdf fiel

            for (int c = 0; c < cr.countChannels(); c++) { // Loop through each column of the text file
               OMSSensor s       = cr.getConfiguration(c); // Get the configuration for the current column
               String    varName = OMSUtil.getUniqueName(s);
               Variable  v       = nc.get(varName);
               if (v == null) {
                  // do nothing
               } else  {
                 /* data[r][0] is the Julian day. This is an extra column.
                    so we use data starting from c = 1 */
                  v.setFloat(ti, (float) data[r][c+1]);
               }
            }
         }
      }
   }

   private void msg(String s) {
      if (DEBUG) {
         System.out.println("OMSNetcdfAppender: " + s);
      }
   }

   boolean DEBUG = true;

   String file,       // The source data file (output from extract)
          cfg;        // The configuration file
   NetcdfFile nc;
   double[] time;
   double[][] data;

}