/*--- formatted by Jindent 2.1, (www.c-lab.de/~jindent) ---*/

package org.mbari.oasis;

import java.io.File;
import java.io.IOException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import ucar.netcdf.NetcdfFile;
import ucar.netcdf.Schema;
import ucar.netcdf.Variable;



/**
 * OMSMain is the main program for creating or adding to Oasis Mooring Spectroradiometer
 * (OMS) netcdf files. If The netcdf file already exists it will check the last date
 * that occurs in the file and append any data that occurs after that to the netcdf file.
 * If it's creating a new file, it will construct a schema based on the information
 * in the calibration histiry of the calibration file supplied as an input
 * argument. It's best to have the complete calibration history listed in the
 * configuration file so that a complete schema is found. Otherwise data for
 * sensors not listed in the config files will be ignored.
 * @author Brian Schlining
 * @version %I%, %G%
 */
public class OMSMain {
    /**
     * @param args
     * @see
     */
    private OMSMain(File cfg, File dat, File nc) throws IOException {
        this.cfg = cfg;
        this.dat = dat;
        this.nc  = nc;
        if (nc.exists()) {
            this.netcdf = new NetcdfFile(nc, false); // Open for writing
        } else {
            this.netcdf = new NetcdfFile(nc, true, true, this.getSchema());
            this.addAxes();
        }
        this.addData();
        this.netcdf.close();
    }
    
    /**
     * Method declaration
     * @param args
     * @see
     */
    public static void main(String[] args) {
         /*String[] args = new String[3];
         args[0] = "h:/users/brian/mooring/eqpac/cfg/ep1.cfg.tmp";
         args[1] = "h:/users/brian/mooring/eqpac/ep2/specprr";
         args[2] = "h:/users/brian/mooring/eqpac/data/ep2.nc";
          */
        
        if (args.length != 3) {
            System.out.println();
            System.out.println("Usage: org.mbari.netcdf.OMSNetcdf configfile datafile ncfile");
            System.out.println();
            System.out.println("Inputs: configfile = name of the mooring configuration file");
            System.out.println("        datafile   = name of specprr data directory");
            System.out.println("        ncfile     = name of netcdf file to be created");
            System.out.println();
            System.exit(0);
        }
        try {
            boolean OK = true;
            
            File cfg = new File(args[0]);
            if (!cfg.exists()) {
                log.warn("The configuration file, " + args[0] +", was not found.");
                OK = false;
            } else if (!cfg.canRead()) {
                log.warn("Unable to read the configuration file, " + args[0] +".");
                OK = false;
            }
            
            File dat = new File(args[1]);
            if (!dat.exists()) {
                log.warn("The data directory, " + args[1] +", was not found.");
                OK = false;
            } else if (!dat.canRead()) {
                log.warn("Unable to read the data directory, " + args[1] +".");
                OK = false;
            }
            
            File nc  = new File(args[2]);
            
            if (OK) {
                OMSMain main = new OMSMain(cfg, dat, nc);
            } else {
                log.warn("The processing was aborted.");
            }
            
        } catch(IOException e) {
            log.fatal("Unable to read or create one of the input files", e);
        }
    }
    
    /**
     * Method declaration
     * @throws IOException
     * @see
     */
    private void addAxes() throws IOException {
        OMSVariableReader vr = new OMSVariableReader(this.cfg);
        while (vr.hasMoreElements()) {
            OMSVariable v = (OMSVariable)vr.nextElement();
            float[] values = v.getValues();
            Variable vnc = this.netcdf.get(v.getName());
            for (int i = 0; i < values.length; i++) {
                int[] index = {i};
                vnc.setDouble(index, (double)values[i]);
            }
        }
    }
    
    /**
     * Method declaration
     * @throws IOException
     * @see
     */
    private void addData() throws IOException {
        // 1. Sort the files by name
        String[] fileList;
        // Read the file list and write out to a file
        fileList = dat.list();
        fileList = OMSUtil.sortFiles(fileList);
        for (int i = 0; i < fileList.length; i++) {
            OMSNetcdfAppender ona = new OMSNetcdfAppender(cfg.getCanonicalPath(), new File(this.dat.getCanonicalPath(), fileList[i]).getCanonicalPath(),
                    this.nc.getCanonicalPath());
        }
    }
    
    /**
     * Method declaration
     * @return
     * @throws IOException
     * @see
     */
    private Schema getSchema() throws IOException {
        System.out.println("Creating Schema from " + this.cfg );
        OMSSchemaConstructor sc = new OMSSchemaConstructor(this.cfg);
        return sc.getSchema();
    }
    
    private File cfg, dat, nc;
    private NetcdfFile netcdf;
    private static final Log log = LogFactory.getLog(OMSMain.class);
}

/*--- formatting done in "Brian Schlining" style on 01-14-2000 ---*/
