package org.mbari.oasis;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.lang.reflect.Array;
import java.util.StringTokenizer;
import java.util.Vector;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;



public class OMSConfigReader {
    
    /////////////////////////////////////////
    // Constructors
    /**
     * OMSConfigReader Reads in the Calibration file tree used for
     * specprr instruments. It defaults to the most current Configuration
     * of the mooring
     *
     * @param filename String of the main configuration file
     */
    public OMSConfigReader(String filename) {
        this(new File(filename), 0);
        
    }
    
    /**
     * OMSConfigReader Reads in the Calibration file tree used for
     * specprr instruments.It defaults to the most current Configuration
     * of the mooring
     *
     * @param file File object representing the main configuration file
     */
    public OMSConfigReader(File file) {
        this(file, 0);
    }
    
    /////////////////////////////////////////
    // Constructors
    /**
     * OMSConfigReader Reads in the Calibration file tree used for
     * specprr instruments
     *
     * @param filename String of the main configuration file
     * @param d The date of the configuration to use
     */
    public OMSConfigReader(String filename, double d) {
        this(new File(filename), d);
    }
    
    /**
     * OMSConfigReader Reads in the Calibration file tree used for
     * specprr instruments
     *
     * @param file File object representing the main configuration file
     * @param d The date of the configuration to use
     */
    public OMSConfigReader(File file, double d) {
        this.cfgfile = file.toString();
        this.currentDate = d;
        this.getSensors(file);
    }
    
    // Debugging method
    public static void main(String[] args) {
        File f = new File("//tsunami.shore.mbari.org/oasis/cfg/m1.cfg");
        OMSConfigReader cr = new OMSConfigReader(f);
        cr.getConfiguration();
    }
    
    /////////////////////////////////////////
    // Accessors
    /**
     * Get the configuration information out of the files. Note that since
     * the names read in here are not nescessarly unique they will have to be
     * assigned unigue names before they can be applied to netCDF schema
     *
     * This method returns the configuration based on the number of channels
     * and the return type specified in the config files
     *
     * @return sensor  An array of OasisSensor objects
     */
    public OMSSensor[] getConfiguration() {
        return this.validSensor;
    }
    
    /**
     *  Return the configuration parameters from a single sensor
     *
     *  @return configuration from sensor number i
     */
    public OMSSensor getConfiguration(int i) {
        return this.validSensor[i];
    }
    
    /**
     * @return The number of specprr instrument banks. For example:
     * m1 has five banks
     * @deprecated Use countBanks
     */
    public int getNumberOfBanks() {
        return this.channels.length;
    }
    
    /**
     * @return The number of specprr instrument banks. For example:
     * m1 has five banks
     */
    public int countBanks() {
        return Array.getLength(this.channels);
    }
    
    /**
     * @return The number of channels in the specified bank
     */
    public int getChannelsInBank(int i){
        return this.channels[i] ;
    }
    
    /**
     * @return The total number of channels that are used by the mooring
     * @deprecicated Use countChannels
     */
    public int getNumberOfChannels() {
        // Add up the total number of channels
      /*int NumberOfChannels = 0;
      for (int j = 0; j < Array.getLength(this.channels); j++) {
         NumberOfChannels += this.channels[j];
      }
      return NumberOfChannels; */
        return validSensor.length;
    }
    
    /**
     * @return The total number of channels that are used by the mooring
     */
    public int countChannels() {
        return validSensor.length;
    }
    
    /**
     * Returns the complete configuration as listed in the config files. Note
     * that this is NOT what is used in processing. Rather it is away to view the
     * complete record in the config file.
     */
    public OMSSensor[] getCompleteConfiguration() {
        return this.sensor;
    }
    
    public OMSSensor getCompleteConfiguration(int i) {
        return this.sensor[i];
    }
    
    public String getFilename() {
        return this.cfgfile;
    }
    
    /////////////////////////////////////////
    // Private Methods
    /**
     * Sets the configuration as extract would read the config files
     */
    private void setValidConfiguration() {
        Vector buf = new Vector();
        int i = 0;
        log.debug("Setting Valid Configuration");
        for (int bank = 0; bank < this.getNumberOfBanks(); bank++) {
            for (int channel = 1; channel <= this.getChannelsInBank(bank); channel++) {
                if (this.getCompleteConfiguration(i).getBank() == bank) {
                    if (this.getCompleteConfiguration(i).getType() != 0) {
                  /*this.ID();
                  System.out.println("Name: " +
                     OMSUtilities.getUniqueName(this.getCompleteConfiguration(i)));*/
                        buf.addElement(this.getCompleteConfiguration(i));
                        
                        
                    }
                }
                i++;
            }
            if (bank < this.getNumberOfBanks()) {
                i = this.nextBank(bank+1);
            }
        }
        
        this.validSensor = new OMSSensor[buf.size()];
        buf.copyInto((OMSSensor[]) validSensor);
        
    }
    
    private int nextBank(int bank) {
        int channel;
        loop: {
            for (channel = 0; channel < Array.getLength(this.sensor); channel++) {
                if (this.getCompleteConfiguration(channel).getBank() == bank) {
                    break loop;
                }
            }
        }
        return channel;
    }
    
    private int cumSum(int bank) {
        int buf = 0;
        for (int i = 0; i <= bank; i++) {
            buf += this.getChannelsInBank(i);
        }
        return buf;
    }
    
    /**
     * Read in the configuration files for the specprr instruments on an Oasis can
     *
     * @param file  A file object specifiying the root configuration file
     */
    private void getSensors(File file) {
        File cfg = this.getSpecprrFile(file);
        log.debug("Reading the configuration files");
        this.readCfg(cfg);
        this.setSensor();
        this.setValidConfiguration();
    }
    
    private void setSensor() {
        this.sensor = new OMSSensor[this.sensorVector.size()];
        this.sensorVector.copyInto((OMSSensor[]) sensor);
    }
/*   private void setSensor() {
      this.sensor = new OasisSensor[this.sensorVector.size()];
      this.sensorVector.copyInto((OasisSensor[]) sensor);
   }*/
    
    
    /**
     * Open the main configuration file and find the line that points to the
     * specprr config file.
     */
    private File getSpecprrFile(File file) {
        
        String cfgFile = new String();
        
        // Need to check to see if the default specprr config file shoudl be used
        boolean useDefault = true;    // Use the default specprr config file
        if (this.currentDate != 0) {
            useDefault = false;        // If a date is specfied we'll look for a match
        }                             // in the history of the config file
        
        boolean exists = false;
        
        try {
            
            BufferedReader in = new BufferedReader(new FileReader(file));
            String str;
            
            if (!useDefault) {   // See if the specified data exists
                while ((str = in.readLine()) != null) {
                    //Process the string
                    if (str.startsWith("9") || str.startsWith("1") || str.startsWith("2")) {
                        StringTokenizer st = new StringTokenizer(str);
                        if (st.countTokens() > 4) { // Make sure its long genough
                            double d1 = Double.parseDouble(st.nextToken()); //Date1
                            String buf = st.nextToken();  // The '-' sign
                            double d2 = Double.parseDouble(st.nextToken()); //Date2
                            String type = st.nextToken(); // is it a specprr file?
                            if (type.startsWith("specprr")) { //If it is a string
                                if (this.currentDate >= d1 & this.currentDate <= d2) {  // Check the dates
                                    exists = true;
                                }
                            }
                        }
                    }
                }
                
            }
            in.close();
            
        } catch (IOException e) {
            log.fatal("Unable to access " + file.getAbsolutePath(), e);
            System.exit(-1);
        }
        
        if (!exists) {        // If a match in the histroy was not found then
            useDefault = true; // go ahead and use the default
        }
        
        // Get the correct line to use from the config history
        try {
            
            BufferedReader in = new BufferedReader(new FileReader(file));
            String str;
            
            if (useDefault) { // Use the default
            /* Open the root.cfg file (ex. m1.cfg) and find the line that starts
            with specprr.*/
                
                loop1: {
                    while ((str = in.readLine()) != null) {
                        //Process the string
                        if (str.startsWith("specprr")) {
                            String path = this.parsePath(str);
                            cfgFile = this.getRealPath(path);
                            break loop1;
                        }
                    }
                }
                
                
            } else {  // Get the line that has the current dates config file
                
                loop2: {
                    while ((str = in.readLine()) != null) {
                        //Process the string
                        if (str.startsWith("9") || str.startsWith("1") || str.startsWith("2")) {
                            StringTokenizer st = new StringTokenizer(str);
                            if (st.countTokens() > 4) {                                   // Make sure its long genough
                                double d1 = Double.parseDouble(st.nextToken());            //Date1
                                String buf = st.nextToken();                               // The '-' sign
                                double d2 = Double.parseDouble(st.nextToken());            //Date2
                                String type = st.nextToken();                              // is it a specprr file?
                                if (type.startsWith("specprr")) {                          //If it is a string
                                    if (this.currentDate >= d1 & this.currentDate <= d2) {  // Check the dates
                                        String path = this.parsePath(str);
                                        cfgFile     = this.getRealPath(path);
                                        break loop2;
                                    }
                                }
                            }
                        }
                    }
                }
            }
            in.close();
        } catch (Exception e) {
            // Deal with IO error here
        }
        
        return new File(cfgFile);
    }
    
    /**
     * Open a file containing configuration data and extract the calibration
     * information.
     */
    private void readCfg(File file) {
        try {
         /*
           Open the specprr configuration file (ex. specprr.m1.19990701)
          */
            log.info("Reading the configuration file " + file);
            BufferedReader in = new BufferedReader(new FileReader(file));
            String str;
            int[] type;
            
            while ((str = in.readLine()) != null) {
                //Process the string
                
                if (str.length() > 1) {
                    
                    if (str.startsWith("/")) {          // Case 1: path info
                        this.readCfg(new File(this.getRealPath(this.parsePath(str))));
                    } else if (str.startsWith("#")) {   // Case 2: Comment
                    } else {
                        StringTokenizer st = new StringTokenizer(str);
                        int n = st.countTokens();
                        if (n == 10) {                   // Case 3: Configuration info
                            this.sensorVector.addElement(new OMSSensor(str));
                        } else {                         // Case 4: Number of channels
                            this.channels      = new int[st.countTokens()];
                            // Get the number of channels in each bank
                            int i = 0;
                            while(st.hasMoreTokens()) {
                                this.channels[i] = new Integer(st.nextToken()).intValue();
                                i++;
                            }
                        }
                    }
                }
            }
            
            in.close();
            
        } catch (Exception e) {
            // Deal with IO error here
        }
    }
    
    /**
     * Extract the path from line in the configuration file
     */
    private String parsePath(String s) {
        
        StringTokenizer st   = new StringTokenizer(s);
        String          buf = new String();
        try {
            loop: {
                while (st.hasMoreTokens()) {
                    buf = st.nextToken();
                    if (buf.startsWith("/")) {
                        break loop;
                    }
                }
            }
        } catch (Exception e) {
            
        }
        return buf;
        
    }
    
    /**
     * Locate a configuration file used for an Oasis can. This method is
     * very specific to the location of the machine on the network. In the future
     * the parameters stored here should be moved to an ini file. It's assumed that
     * all cfg files are kept on the same machine.
     */
    private String getRealPath(String path) {
        String file = new String();
        String OS = System.getProperty("os.name").toLowerCase();
        if (OS.startsWith("win")) {
            file = "//tsunami.shore.mbari.org" + path;
        } 
        else if (OS.startsWith("sun")){
            file = "/hosts/tornado/vol/vol0" + path;
        }
        else if (OS.startsWith("mac")) {
            file = "/Volumes" + path;
            File f = new File(file);
            if (!f.exists()) {
                file = path;
            }
        } 
        else {
            file = "/hosts/tsunami" + path;
        }
        log.info("Generated a path of " + file);
        return file;
    }
    
    
    boolean DEBUG = false;
    /////////////////////////////////////////
    // Class Variables
    protected OMSSensor[] validSensor,
            sensor;
    
    private   String        cfgfile;
    
    private   Vector        sensorVector = new Vector();
    
    private   int[]         channels;
    
    private   double        currentDate;  // date of specprr file to use. ex. 99182
    
    private static final Log log = LogFactory.getLog(OMSConfigReader.class);
    
}
