package org.mbari.oasis;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.lang.reflect.Array;
import java.util.Enumeration;
import java.util.StringTokenizer;
import ucar.netcdf.Attribute;


/**
 * This class reads in data from a file in 'mooring'/data/specprr files and
 * collects the netcdf specific information for attributes
 *
 * @author Brian Schlining
 * @version 29 Sep 1999
 */
public class OMSAttributeReader implements Enumeration {

   /////////////////////////////////////////
   // Constructors
   /**
    * MooringAttributeReader reads in the Attributes in the main configuration
    * file
    *
    * @param filename The configuration file containing the mooring attributes
    */
   public OMSAttributeReader(String filename) {
      this.cfgfile = filename;
      File file = new File(filename);
      this.init(file);
   }

   /**
    * MooringAttributeReader reads in the Attributes in the main configuration
    * file
    *
    * @param filename The configuration file containing the mooring attributes
    */
   public OMSAttributeReader(File file) {
      this.cfgfile = file.toString();
      this.init(file);
   }

   /////////////////////////////////////////
   // Public Methods
   /**
    * @return The name of the file used to create an instance of
    * MooringAttributeReader
    */
   public String getFilename() {
      return this.cfgfile;
   }

   /**
    * Get all attributes
    *
    * @return All attributes occurring in this object
    */
   public Attribute[] getAttributes() {
      return this.attributes;
   }

   /**
    * Get a single attribute
    *
    * @param i The index of the attribute to return. <B>REMINDER</B> Java uses
    * zero based indices.
    * @return One attribute
    * @exception ArrayIndexOutOfBoundsException
    */
   public Attribute getAttribute(int i) {
      return this.attributes[i];
   }

   /**
    * Count the number of attributes in the file
    *
    * @return Number of attributes
    */
   public int countAttributes() {
      return Array.getLength(this.attributes);
   }

   /**
    * Enumeration interface
    *
    * @return true if more elements are available; false otherwise
    */
   public boolean hasMoreElements() {
      boolean ok = false;
      if (this.index < this.countAttributes()) {
         ok = true;
      }
      return ok;
   }

   /**
    * Enumeration interface
    *
    * @return The next element
    */
   public Object nextElement() {
      if (this.hasMoreElements()) {
         this.index++;
         return this.attributes[this.index - 1];
      } else {
         return null;
      }
   }

   /////////////////////////////////////////
   // Private Methods

   private void count(File file) throws IOException {
      int nAtt = 0;
      String str;
      BufferedReader in = new BufferedReader(new FileReader(file));
      while ((str = in.readLine()) != null) {
         if (str.length() > 3) {
            if (str.startsWith("#")) {
               if (str.substring(1,2).equals("!")) {
                  nAtt++;
               } else if (str.substring(1,2).equals(":")) {
                  // Ignore Variables
               }
            }
         }
      }
      in.close();
      this.attributes = new Attribute[nAtt];
   }

   private void init(File file) {
      try {
         this.count(file);
         this.parse(file);
      } catch (IOException e) {
      }
      this.index = 0;
   }

   private void parse(File file) throws IOException {
      String str;
      BufferedReader in = new BufferedReader(new FileReader(file));
      while ((str = in.readLine()) != null) {
         if (str.length() > 3) {
            if (str.startsWith("#")) {
               if (str.substring(1,2).equals("!")) {
                  this.readAttribute(str);
               } else if (str.substring(1,2).equals(":")) {
                  //Ignore variables
               }
            }
         }
      }
      in.close();
   }

   private void readAttribute(String str) {
      StringTokenizer st   = new StringTokenizer(str);
      String name = new String();
      int n  = 0;
      for (int i = 0; i < 2; i++) {
         name = st.nextToken();     // Get the name of the attribute (2nd token)
         n += name.length();
      }
      String value = str.substring(n+2); // The line after the name is the value
      this.attributes[this.index] = new Attribute(name, value);
      this.index++;
   }

   /////////////////////////////////////////
   // Class Variables
   private String cfgfile;
   protected Attribute[] attributes;
   private int index = 0;
}
