package org.mbari.oasis;

import java.io.File;
import java.lang.reflect.Array;
import java.text.SimpleDateFormat;
import java.util.Date;
import ucar.netcdf.Attribute;
import ucar.netcdf.Dimension;
import ucar.netcdf.ProtoVariable;
import ucar.netcdf.Schema;
import ucar.netcdf.UnlimitedDimension;



public abstract class OMSAbstractSchemaConstructor {

   public OMSAbstractSchemaConstructor(File file) {
      this.cfgfile = file;
      this.init();
   }

   /////////////////////////////////////////
   // Accessors
   /**
    * Return the schema for use in creating a netCDF file
    *
    * @return Schema
    */
   public Schema getSchema() {
      return this.schema;
   }

   public ProtoVariable[] getAxes() {
     return this.axesP;
   }

   /////////////////////////////////////////
   // Private Methods
   /**
    * Called by constructors to set up schema
     */
   private void init() {
      this.schema = new Schema();

      OMSAttributeReader ar = new OMSAttributeReader(cfgfile);
      this.setGlobalAttributes(this.createGlobalAttributes(ar));

      OMSVariableReader  vr = new OMSVariableReader(cfgfile);
      this.setAxes(vr);

      this.variables = vr.getVariables();

      OMSSensor[] s = this.getSensors();

      ProtoVariable[] pv = this.setAttributes(s);
      this.setProtoVariables(pv);
   }

   /////////////////////////////////////////
   // Protected methods

   ////////Global Attributes////////
   /**
    * Applies global attributes to a schema
    *
    * @param att An array of Attributes
    */
   private void setGlobalAttributes(Attribute[] att) {
      for (int i = 0; i < Array.getLength(att); i++) {
         this.schema.putAttribute(att[i]);
      }
   }

   /**
    * Collect the Attributes in the Config file (using a OMSAttributeReader)
    * and add them to additional attributes specified here.
    *
    * @param ar The OMSAttributeReader using the main config file
    */
   private Attribute[] createGlobalAttributes(OMSAttributeReader ar) {

      SimpleDateFormat sdf = new SimpleDateFormat("MM/dd/yyyy");

      // Global Attributes
      int i = 3;
      Attribute[] att = new Attribute[i + ar.countAttributes()];
      att[0] = new Attribute("conventions","MBARI/timeSeries/mooring/spectroradiometer");
      att[1] = new Attribute("creationDate", sdf.format(new Date()));
      att[2] = new Attribute("lastModified", sdf.format(new Date()));

      sdf = null;

      while (ar.hasMoreElements()) {
         att[i] = (Attribute) ar.nextElement();
         i++;
      }
      return att;
   }

   ////////Axes////////
   /**
    * Create the data Axes (i.e. Dimensions in netCDF terms). For moorings these
    * will generally be Axes with the same names as the dimensions.
    */
   private void setAxes(OMSVariableReader vr) {

      this.axesP = new ProtoVariable[1 + vr.countVariables()];

      axesP[0] = new ProtoVariable("time", double.class, this.timeD);
      axesP[0].putAttribute(new Attribute("long_name", "time GMT"));
      axesP[0].putAttribute(new Attribute("units", "seconds since 1970-01-01 00:00:00"));

      OMSVariable[] v = vr.getVariables();
      for (int i = 1; i < vr.countVariables() + 1; i++) { // i indexes axesP
         int j = i - 1;                               // j indexes the variables
         axesP[i] = new ProtoVariable(v[j].getName(),double.class,
                    new Dimension(v[j].getName(), v[j].countVariables()));
         axesP[i].putAttribute(new Attribute("long_name",
                               OMSUtil.getLongName(v[j].getName())));

         String vn =  v[j].getName().toLowerCase(),
                units;

         if (vn.equals("longitude")) {
            units = "degrees_east";
         } else if (vn.equals("latitude")) {
            units = "degrees_north";
         } else if (vn.equals("bankdepth")) {
            units = "meters";
            axesP[i].putAttribute(new Attribute("positive", "down"));
         } else {
            units = "";
         }

         axesP[i].putAttribute(new Attribute("units", units));
      }

      this.setProtoVariables(axesP);
   }



   ////////ProtoVariables////////
   /**
    * Convience method to add protovariables arrays to a schema
    *
    * @param pv The array of protoVariables
    */
   private void setProtoVariables(ProtoVariable[] pv) {
      for (int i = 0; i < Array.getLength(pv); i++) {
         this.schema.put(pv[i]);
      }
   }

   /**
    * Convience method to add protovariables arrays to a schema
    *
    * @param pv A single protovariable
    */
   private void setProtoVariables(ProtoVariable pv) {
      this.schema.put(pv);
   }

   ////////Attributes////////
   /**
    * This generates attributes for each NetCDF variable from information
    * contained in the configuration files
    *
    * @param sensor Infomation collected from the config files (using
    * the OMSConfigReader, OMSConfigTreeReader or some variant
    * @return An array of Attriutes for use in constuction the netCDF schema
    */
   private Attribute[] createAttributes(OMSSensor sensor) {
      float depth = -999f;

      for (int i = 0; i < this.variables.length; i++) {
         if (variables[i].getName().equalsIgnoreCase("bankDepth")) {
            float[] depths = variables[i].getValues();
            depth  = depths[sensor.getBank()];
            if (sensor.getBank() == 2) {
               switch (sensor.getChannel()) {  // Correct depth for MCP's
                  case 11:
                     depth = depth - 10f;
                     break;
                  case 12:
                     depth = depth - 10f;
                     break;
                  case 13:
                     depth = depth - 10f;
                     break;
                  case 14:
                     depth = depth - 10f;
                     break;
                  case 15:
                     depth = depth + 10f;
                     break;
                  case 16:
                     depth = depth + 10f;
                     break;
               }
            }
         }
      }

      String symbol     = OMSUtil.getDataSymbol(sensor);
      float waveLength = OMSUtil.getWavelength(sensor);
      int length = 6;

      // Create attribute array
      return new Attribute[] {
         new Attribute("long_name", OMSUtil.getLongName(sensor.getName())),
         new Attribute("units", sensor.getUnits()),
         new Attribute("bank", (int) sensor.getBank()),
         new Attribute("channel", (int) sensor.getChannel()),
         new Attribute("instrumentDepth", (double) depth),
         new Attribute("symbol", symbol),
         new Attribute("wavelength", (double) waveLength),
         new Attribute("_FillValue", new Float(-999f)),
         new Attribute("missing_value", new Float(-999f))

      };
   }

   /**
    * Assign attributes to each channels protovariable. Channels with type equal
    * to 0 are ignored.
    *
    * @param cr      A MooringSpecprrConfigReader object
    * @param dim              The unlimited dimension, representing time
    * @return ProtoVariable[] Array of protovariables representing each channel
    */
   private ProtoVariable[] setAttributes(OMSSensor[] s) {

      /* Create an array using the channels specified in the config files.
         However, exclude channels of type 0 */
      ProtoVariable[] pv = new ProtoVariable[s.length];
      int i = 0,
          k = 0;

      //Loop through each sensor
      while (i < s.length) {
         //Create a protovariable if the sensor type is not 0
         if (s[i].getType() != 0) {
            // Use the one dimension convienience constructor
            pv[k] = new ProtoVariable(OMSUtil.getUniqueName(s[i]),
                                      float.class, this.timeD);
            // Create the attributes
            Attribute[] att = this.createAttributes(s[i]);
            // Add the attributes to the protovariable
            for (int j = 0; j < Array.getLength(att); j++) {
               pv[k].putAttribute(att[j]);
            }
            k++;
         }
         i++;
      }

      return pv;
   }

   protected abstract OMSSensor[] getSensors();

   protected OMSVariable[] variables;

   protected Schema schema;

   protected File cfgfile;

   protected Dimension timeD = new UnlimitedDimension("time");

   protected ProtoVariable[] axesP;
}