package org.mbari.oasis;

/**
 * Title:        MBARI Java development
 * Description:
 * Copyright:    Copyright (c) 2001
 * Company:      MBARI
 * @author Brian Schlining
 * @version 1.0
 */

import java.io.*;
import java.util.*;
import org.mbari.io.*;
import org.mbari.util.*;
import ucar.nc2.*;
import ucar.ma2.*;

public class NetcdfTest {

    public NetcdfTest(String outfile) throws IOException {
        NetcdfFileWriteable ncfile = new NetcdfFileWriteable();
        ncfile.setName(outfile);

        ncfile.addGlobalAttribute("conventions", "MBARI/timeSeries/mooring/pco2");
        ncfile.addGlobalAttribute("creationDate", new Date().toString());
        ncfile.addGlobalAttribute("lastModified", new Date().toString());
        ncfile.addGlobalAttribute("mooring", "M1");
        ncfile.addGlobalAttribute("description", "pCO2 data");
        ncfile.addGlobalAttribute("waterdepth", "1050");
        ncfile.addGlobalAttribute("keywords", "pCO2, mooring");
        ncfile.addGlobalAttribute("instrumentType", "pCO2 Can");

        //define Dimensions
        Dimension timeDim    = ncfile.addDimension("time", -1);
        Dimension latDim     = ncfile.addDimension("latitude", 1);
        Dimension lonDim     = ncfile.addDimension("longitude", 1);
        Dimension calTimeDim = ncfile.addDimension("calTime",
            500);

        //Define Variables
        Dimension[] mainDim = new Dimension[3];
        mainDim[0] = timeDim;
        mainDim[1] = latDim;
        mainDim[2] = lonDim;

        Dimension[] calDim = new Dimension[3];
        calDim[0] = calTimeDim;
        calDim[1] = latDim;
        calDim[2] = lonDim;

        ncfile.addVariable("time", double.class, new Dimension[] {timeDim});
        ncfile.addVariableAttribute("time", "long_name", "time GMT");
        ncfile.addVariableAttribute("time", "units", "seconds since 01 Jan 1970 00:00:00 GMT");

        ncfile.addVariable("latitude", float.class, new Dimension[] {latDim});
        ncfile.addVariableAttribute("latitude", "long_name", "Latitude");
        ncfile.addVariableAttribute("latitude", "units", "degrees_north (+N/-S)");

        ncfile.addVariable("longitude", float.class, new Dimension[] {lonDim});
        ncfile.addVariableAttribute("longitude", "long_name", "Longitude");
        ncfile.addVariableAttribute("longitude", "units", "degrees_east (+E/-W)");

        ncfile.addVariable("calTime", double.class, new Dimension[] {calTimeDim});
        ncfile.addVariableAttribute("calTime", "long_name", "time GMT of calibration samples");
        ncfile.addVariableAttribute("calTime", "units", "seconds since 01 Jan 1970 00:00:00 GMT");

        ncfile.addVariable("pco2u", float.class, mainDim);
        ncfile.addVariableAttribute("pco2u", "long_name", "uncalibrated pCO2");
        ncfile.addVariableAttribute("pco2u", "units", "microatm");
        ncfile.addVariableAttribute("pco2u", "symbol", "pCO2u");
        ncfile.addVariableAttribute("pco2u", "_FillValue", new Float(-999f));
        ncfile.addVariableAttribute("pco2u", "missing_value", new Float(-999f));

        ncfile.addVariable("pco2", float.class, mainDim);
        ncfile.addVariableAttribute("pco2", "long_name", "calibrated pCO2");
        ncfile.addVariableAttribute("pco2", "units", "microatm");
        ncfile.addVariableAttribute("pco2", "symbol", "pCO2");
        ncfile.addVariableAttribute("pco2", "_FillValue", new Float(-999f));
        ncfile.addVariableAttribute("pco2", "missing_value", new Float(-999f));

        ncfile.addVariable("cal", float.class, calDim);
        ncfile.addVariableAttribute("cal", "long_name", "atmospheric calibration");
        ncfile.addVariableAttribute("cal", "units", "microatm");
        ncfile.addVariableAttribute("cal", "symbol", "pCO2c");
        ncfile.addVariableAttribute("cal", "_FillValue", new Float(-999f));
        ncfile.addVariableAttribute("cal", "missing_value", new Float(-999f));

        ncfile.addVariable("calI", float.class, mainDim);
        ncfile.addVariableAttribute("calI", "long_name", "atmospheric calibration " +
          "interpolated to pco2 time axis");
        ncfile.addVariableAttribute("calI", "units", "microatm");
        ncfile.addVariableAttribute("calI", "symbol", "pCO2ci");
        ncfile.addVariableAttribute("calI", "_FillValue", new Float(-999f));
        ncfile.addVariableAttribute("calI", "missing_value", new Float(-999f));

        // Create the file
        ncfile.create();

        // Make up some data
        double[]     data1 = new double[2902];
        float[][][] data3 = new float[2902][1][1];
        // write non-zero data to the rawTime array
        for (int i = 0; i < data1.length; i++){
            data1[i] = i;
            data3[i][0][0] = i;
        }
        double[]     calData1 = new double[500];
        float[][][] calData3 = new float[500][1][1];
        // write non-zero data to the rawTime array
        for (int i = 0; i < calData1.length; i++){
            data1[i] = i;
            data3[i][0][0] = i;
        }

        ncfile.write("time",  ArrayAbstract.factory(data1));
        ncfile.write("latitude", ArrayAbstract.factory(new float[] {36f}));
        ncfile.write("longitude", ArrayAbstract.factory(new float[] {-122f}));
        ncfile.write("calTime", ArrayAbstract.factory(calData1));
        ncfile.write("pco2u", ArrayAbstract.factory(data3));
        ncfile.write("pco2", ArrayAbstract.factory(data3));
        ncfile.write("cal", ArrayAbstract.factory(calData3));
        ncfile.write("calI", ArrayAbstract.factory(data3));

        // Close
        ncfile.close();
    }

    public static void main(String[] args) {
        try {
           NetcdfTest netcdfTest1 = new NetcdfTest(args[0]);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }
}