package org.mbari.oasis;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Arrays;
import java.util.Date;
import java.util.Properties;
import java.util.StringTokenizer;
import org.mbari.hobilabs.HRBinaryDecoder;
import org.mbari.hobilabs.HRCalFileNameFilter;
import org.mbari.hobilabs.HRCalibrationFileReader;
import org.mbari.hobilabs.HRPacket;
import org.mbari.util.GmtCalendar;
import org.mbari.util.HRUtil;
import ucar.multiarray.ArrayMultiArray;
import ucar.multiarray.MultiArray;
import ucar.netcdf.Attribute;
import ucar.netcdf.Dimension;
import ucar.netcdf.NetcdfFile;
import ucar.netcdf.ProtoVariable;
import ucar.netcdf.Schema;
import ucar.netcdf.UnlimitedDimension;
import ucar.netcdf.Variable;

/**
 * Title:        HobiLabs data processing
 * Description:
 * Copyright:    Copyright (c) Brian Schlining
 * Company:      MBARI
 * @author Brian Schlining
 * @version 1.0
 */



public class HRNetcdfConstructor {

   /**
    * @param cfgfile Configuration file for the instrument (i.e. hr2.hr1999345.2000345)
    * @param infile Name of one of the hr binary files to process. This program will
    *  attempt to find all other fiels form the same instrument in the same
    *  directoyr and combine them together
    * @param outfile The name fohte netcdf file to construct
    */
   public HRNetcdfConstructor(String mooring, File calDirectory, String outfile,
    File[] infile) {
      this.calDirectory = calDirectory;
      Arrays.sort(infile);
      this.infile = infile;
      this.outfile = outfile;
      this.mooring = mooring.toLowerCase();
   }

//   public HRNetcdfConstructor(String mooring, String calDirectory, String outfile,
//    String[] infile) {
//      File[] f = new File[infile.length];
//      for (int i = 0; i < infile.length; i++) {
//         f[i] = new File(infile[i]);
//      }
//      this(mooring, new File(calDirectory), outfile, f);
//   }

   public void process() throws IOException, IllegalArgumentException {

      if (!calDirectory.isDirectory())
         throw new IllegalArgumentException(calDirectory.getCanonicalPath() +
            " is not a directory.");

      ///////////////////////////////////
      // Load the mooring properties file
      Properties mooringProperties = new Properties();
      // Assume the mooring.properties file is in the calDirectory
      BufferedInputStream mooringPropertiesStream = new BufferedInputStream(
       new FileInputStream(new File(calDirectory, "mooring.properties")));
      mooringProperties.load(mooringPropertiesStream);
      mooringPropertiesStream.close();
      float[] lat = {Float.parseFloat(mooringProperties.getProperty(mooring + ".latitude", "-999"))};
      float[] lon = {Float.parseFloat(mooringProperties.getProperty(mooring + ".longitude", "-999"))};

      //////////////////////
      // Read the data files
      HRBinaryDecoder[] d = new HRBinaryDecoder[this.infile.length];
      for (int i = 0; i < d.length; i++) {
         // Read the data file
         d[i] = new HRBinaryDecoder(infile[i], false);
         d[i].decode();

         // Flag data with bad time stamps
         HRPacket[] buf = d[i].getData();
         int n = buf.length;
         for (int j = 0; j < buf.length; j++) {
            if (buf[j].RawTime < 883612800) { // If the time is before 01 Jan 1998 it's bad
               n--;                           // MBARI didn't have instruments before this date.
               buf[j].RawTime = 0;            // Set bad times to 0
            }
         }

         // Remove flagged data
         HRPacket[] newBuf = new HRPacket[n]; // temporary array to hold the good data
         int newN = 0;
         for (int j = 0; j < n; j++) {
            if (buf[j].RawTime != 0) {        // If not flagged copy it to the new array
               newBuf[j] = buf[j];
               newN++;
            }
         }
         d[i].setData(newBuf);                // Copy the array to the HRBinaryDecoder
      }

      /////////////////////////////////////////
      // Open the appropriate calibration files
      // Some needed info
      String[]   serialNumber0    = d[0].getSerialNumbers();
      String     instrumentType0  = d[0].getInstrumentType();
      HRPacket[] data0            = d[0].getData();

      String[] calList = this.calDirectory.list(new HRCalFileNameFilter(
       instrumentType0.toLowerCase(), serialNumber0[0]));
      Arrays.sort(calList);   // Sorted list of cal files
      HRCalibrationFileReader[] calFiles =
       new HRCalibrationFileReader[calList.length];
      double[] calTime = new double[calList.length];  // Get the time of each file
      for (int i = 0; i < calList.length; i++) {
         calFiles[i] = new HRCalibrationFileReader(new File(
          calDirectory, calList[i]));
         StringTokenizer st3 = new StringTokenizer(calList[i], ".");
         String tmp = st3.nextToken();
         tmp   = st3.nextToken();
         tmp   = st3.nextToken(); // Here's the word with the date
         int year, month, date;
         year  = Integer.parseInt(tmp.substring(0,4));
         month = Integer.parseInt(tmp.substring(4,6));
         date  = Integer.parseInt(tmp.substring(6,8));
         // Return time in seconds GMT
         calTime[i] = new GmtCalendar(year, month, date).getTime().getTime()/1000;
      }

      // Determine which cal file to use
      int good = 0;  // Index of calFile to use
      if (data0[0].RawTime < calTime[0]) {
            good = 0;
      } else if (data0[0].RawTime > calTime[calTime.length - 1]) {
            good = calTime.length - 1;
      } else {
         for (int i = 0; i < calTime.length - 1; i++) {
            if ((data0[0].RawTime >= calTime[i]) && (data0[0].RawTime < calTime[i+1])) {
               good = i;
               break;
            }
         }
      }


      //////////////////////////////
      // Construct the netcdf schema
      // schemaV will contain all the ProtoVariables
      ProtoVariable[] schemaV = new ProtoVariable[3 + 6 * d.length];
      int n = 0; // Counter for indexing into schemaV

      // Time
      Dimension timeD = new UnlimitedDimension("time");
      Attribute[] timeA = {
         new Attribute("long_name", "time GMT"),
         new Attribute("units", "seconds since 1970-01-01 00:00:00"),
      };
      ProtoVariable timeV = new ProtoVariable("time", Double.TYPE,
       new Dimension[] {timeD}, timeA);
      schemaV[n] = timeV;
      n++;

      // lambda variables
      Dimension[] lambdaD = new Dimension[infile.length];
      Attribute[] lambdaA = {                           // Wavelength attributes
         new Attribute("long_name", "wavelength"),
         new Attribute("units", "nanometers"),
      };
      ProtoVariable[] lambdaV = new ProtoVariable[d.length];

      // depth variables
      Dimension[] depthD  = new Dimension[infile.length];
      Attribute[] depthA = {                           // Depth attributes
         new Attribute("long_name", "Sensor depth"),
         new Attribute("units", "meters"),
      };
      ProtoVariable[] depthV = new ProtoVariable[d.length];


      // Adding lambda and depth variables here
      ProtoVariable[] timeXV = new ProtoVariable[d.length];
      for (int i = 0; i < d.length; i++) {            // Dimensions for each channel

         HRPacket[]      data            = d[i].getData();

         lambdaD[i] = new Dimension("wavelength" + d[i].getChannelNumber(),
          data[0].PixCount);
         lambdaV[i] = new ProtoVariable("wavelength" + d[i].getChannelNumber(),
          Float.TYPE, new Dimension[] {lambdaD[i]}, lambdaA);
         schemaV[n] = lambdaV[i];
         n++;

         depthD[i]  = new Dimension("depth" + d[i].getChannelNumber(), 1);
         depthV[i] = new ProtoVariable("depth" + d[i].getChannelNumber(),
          Float.TYPE, new Dimension[] {depthD[i]}, depthA);
         schemaV[n] = depthV[i];
         n++;

         // Add orginal files times
         // time variables
         Attribute[] timeXA = {                           // Depth attributes
            new Attribute("long_name", "Sample time for " + d[i].getDataType() + " sensor"),
            new Attribute("units", "seconds since 1970-01-01 00:00:00"),
            new Attribute("_FillValue", -999f),
         };
         timeXV[i] = new ProtoVariable("time" + d[i].getChannelNumber(),
          Double.TYPE, new Dimension[] {timeD}, timeXA);
         schemaV[n] = timeXV[i];
         n++;
      }

      // Latitude
      Dimension latD = new Dimension("latitude", 1);
      Attribute[] latA = {                           // Depth attributes
         new Attribute("long_name", "Latitude"),
         new Attribute("units", "degrees_east"),
      };
      ProtoVariable latV = new ProtoVariable("latitude",
          Float.TYPE, new Dimension[] {latD}, latA);
      schemaV[n] = latV;
      n++;

      // Longitude
      Dimension lonD = new Dimension("longitude", 1);
      Attribute[] lonA = {                           // Depth attributes
         new Attribute("long_name", "Longitude"),
         new Attribute("units", "degrees_north"),
      };
      ProtoVariable lonV  = new ProtoVariable("longitude",
          Float.TYPE, new Dimension[] {lonD}, lambdaA);
      schemaV[n] = lonV;
      n++;

      // Define Global Attributes
      Attribute[] globalAtt = {
         new Attribute("conventions", "MBARI/timeSeries/mooring/hydrorad"),
         new Attribute("creationDate", new Date().toString()),
         new Attribute("lastModified", new Date().toString()),
         new Attribute("mooring", "M1"),
         new Attribute("project", "Monterey Bay Time Series"),
         new Attribute("keywords", "spectroradiometer"),
         new Attribute("instrumentType", "Hobilabs " + instrumentType0),
         new Attribute("serialNumber", serialNumber0[0]),
         new Attribute("configurationFile", calList[good])
      };


         // Define the variable (temp, voltage, depth, intTime)
//         ProtoVariable tempV = new ProtoVariable("temp", Float.TYPE,
//          new Dimension[] {timeD, depthD, latD, lonD},
//          new Attribute[] {
//            new Attribute("long_name", "temperature"),
//            new Attribute("units", "celsius"),
//            new Attribute("symbol", "T"),
//            new Attribute("_FillValue", -999f),
//            new Attribute("missing_value", -999f),
//          });
//         ProtoVariable voltV = new ProtoVariable("voltage", Float.TYPE,
//          new Dimension[] {timeD},
//          new Attribute[] {
//            new Attribute("long_name", "power supply voltage"),
//            new Attribute("units", "volts"),
//            new Attribute("symbol", "V"),
//            new Attribute("_FillValue", -999f),
//            new Attribute("missing_value", -999f),
//          });

      // pressures
      ProtoVariable[] pressureV = new ProtoVariable[d.length];
      for (int i = 0; i < d.length; i++) {
          pressureV[i] = new ProtoVariable("pressure" + d[i].getChannelNumber(), Float.TYPE,
          new Dimension[] {timeD, depthD[i], latD, lonD},
          new Attribute[] {
            new Attribute("long_name", "Pressure for channel " + d[i].getChannelNumber()),
            new Attribute("units", "dbar"),
            new Attribute("symbol", "Z"),
            new Attribute("_FillValue", -999f),
            new Attribute("missing_value", -999f),
          });
          schemaV[n] = pressureV[i];
          n++;
      }

      // integration time
      ProtoVariable[] intTimeV = new ProtoVariable[d.length];
      for (int i = 0; i < d.length; i++) {
          intTimeV[i] = new ProtoVariable("intTime" + d[i].getChannelNumber(), Float.TYPE,
          new Dimension[] {timeD, depthD[i], latD, lonD},
          new Attribute[] {
            new Attribute("long_name", "Integration time for channel " + d[i].getChannelNumber()),
            new Attribute("units", "seconds"),
            new Attribute("symbol", "I"),
            new Attribute("_FillValue", -999f),
            new Attribute("missing_value", -999f),
          });
          schemaV[n] = intTimeV[i];
          n++;
      }

      // data
      ProtoVariable[] dataV = new ProtoVariable[d.length];
      for (int i = 0; i < d.length; i++) {
          dataV[i] = new ProtoVariable(d[i].getDataType(), Float.TYPE,
          new Dimension[] {timeD, lambdaD[i], depthD[i], latD, lonD},
          new Attribute[] {
            new Attribute("long_name", d[i].getDataDescription()),
            new Attribute("units", d[i].getUnits()),
            new Attribute("symbol", d[i].getDataType()),
            new Attribute("_FillValue", -999f),
            new Attribute("missing_value", -999f),
          });
          schemaV[n] = dataV[i];
          n++;
      }

      // Make the schema here
      Schema schema = new Schema(schemaV, globalAtt);
      NetcdfFile nc = new NetcdfFile(outfile, true, true, schema);

      int[] origin1 = {0};
      int[] origin4 = {0, 0, 0, 0};
      int[] origin5 = {0, 0, 0, 0, 0};

      // Add the data to the netcdf file

      // 1st Get the longest time in any of the files and assume thats
      // the complete time. All other variables will be mapped to this
      int timeLength = 0;;
      int timeIndex = 0;
      for (int i = 0; i < d.length; i++) {
         HRPacket[]      data            = d[i].getData();
         if (data.length > timeLength) {
            timeLength = data.length;
            timeIndex = i;
         }
      }

      HRPacket[] buf = d[timeIndex].getData();
      double[] timeBase = new double[buf.length];
      for (int r = 9; r < buf.length; r++) {
         timeBase[r] = (float) buf[r].RawTime;
      }

      MultiArray m = new ArrayMultiArray(timeBase); // Add Time
      Variable v = nc.get(timeV.getName());
      v.copyin(origin1, m);

      m = new ArrayMultiArray(lat);  // Add latitude
      v = nc.get(latV.getName());
      v.copyin(origin1, m);

      m = new ArrayMultiArray(lon);  // Add longitude
      v = nc.get(lonV.getName());
      v.copyin(origin1, m);

      // Loop to add depth, wavelength, intime, pressure, and the data
      for (int i = 0; i < d.length; i++) {

         HRPacket[]      data            = d[i].getData();
         String[]        serialNumber    = d[i].getSerialNumbers();
         String          dataDescription = d[i].getDataDescription();
         String          dataType        = d[i].getDataType();
         String          dataUnits       = d[i].getUnits();
         String          channel         = d[i].getChannelNumber() + "";
         String          instrumentType  = d[i].getInstrumentType();

         // Calculate wavelengths
         double[] lambda = HRUtil.calcWavelength(data[0], calFiles[good].getWave(channel));
         float[] lambdaF = new float[lambda.length];
         for (int j = 0; j < lambda.length; j++) {
            lambdaF[j] = (float) lambda[j];
         }

         // Assign the data to arrays
         double[] time = new double[data.length];
         float[]  temp = new float[data.length],
                  voltage = new float[data.length],
                  pressure = new float[data.length],
                  intT = new float[data.length];
         float[][] dat =  new float[data.length][lambda.length];
         float[] depth = {0};
         for (int r = 0; r < data.length; r++) {
            time[r] = (float) data[r].RawTime;
            temp[r] = (float) data[r].Temp;
            voltage[r] = (float) data[r].Voltage;
            pressure[r] = (float) data[r].Depth;
            intT[r] = (float) data[r].IntTime;
            dat[r] = new float[data[r].Pixel.length];
            for (int c = 0; c < dat[r].length; c++) {
               dat[r][c] = (float) data[r].Pixel[c];
            }
            depth[0] = depth[0] + pressure[r]; // Pressure = measured; depth = average pressure
         }
         depth[0] = depth[0] / data.length; // Use the mean pressure as the nominal depth

         // Will need to loop through the data to add it since they may be on
         // different time axes. 12/4/00

         m = new ArrayMultiArray(depth);     // Add Depth
         v = nc.get(depthV[i].getName());
         v.copyin(origin1, m);

         m = new ArrayMultiArray(lambdaF);   // Add wavelength
         v = nc.get(lambdaV[i].getName());
         v.copyin(origin1, m);

         for (int k = 0; k < time.length; k++) {
            // Find the correct time or insertion point
            int index = Arrays.binarySearch(timeBase, time[k]);

            if (index < -1) {
               index = -(index + 1); // Convert any insertion points to an index
               if (index < timeBase.length && index > 0) {
                  double dt1 = Math.abs(timeBase[index  - 1] - time[k]); // insert at nearest
                  double dt2 = Math.abs(timeBase[index] - time[k]);
                  if (dt1 < dt2)
                     index = index - 1;
               }
            }

            if (index >= 0) {
               int[] origin_5 = {index, 0, 0, 0, 0};
               int[] origin_4 = {index, 0, 0, 0};
               int[] origin_1 = {index};

               m = new ArrayMultiArray(dat);       // Add Data
               v = nc.get(dataV[i].getName());
               for (int ii = 0; ii < lambdaF.length; ii++) {
                  v.setFloat(new int[] {index, ii, 0, 0, 0}, dat[k][ii]);
               }

               v = nc.get(timeXV[i].getName());
               v.setDouble(origin_1, time[k]); // Add sensor time

               v = nc.get(pressureV[i].getName());
               v.setFloat(origin_4, pressure[k]); // Add pressure

               v = nc.get(intTimeV[i].getName());
               v.setFloat(origin_4, intT[k]); // Add

            }
         }

//         m = new ArrayMultiArray(temp);
//         v = nc.get(tempV.getName());
//         v.copyin(origin1, m);
//
//         m = new ArrayMultiArray(voltage);
//         v = nc.get(voltV.getName());
//         v.copyin(origin2, m);


      }

      nc.close();
   }

   /**
    * arg[0] = calfile directory
    * arg[1] = binary data file
    * arg[2] = destination directory
    */
   public static void main(String[] args) {
      if (args.length < 4) {
         System.err.println("Usage: java org.mbari.hobilabs.hr.HRNetcdfConstructor " +
          "<mooring> <Directory containing cal files> <outfile> <infile1> <infile2> ...");
         System.exit(0);
      }

      File[] infile = new File[args.length - 3];

      for (int i = 3; i < args.length; i++) {
         infile[i - 3] = new File(args[i]);
      }


      HRNetcdfConstructor main = new HRNetcdfConstructor(args[0], new File(args[1]),
       args[2], infile);
      try {
         main.process();
      } catch (Exception e) {
         e.printStackTrace();
      }

   }

   File calDirectory;
   File[] infile;
   String outfile,
          mooring;
}