
/**
 * Title:        HobiLabs data processing<p>
 * Description:  <p>
 * Copyright:    Copyright (c) Brian Schlining<p>
 * Company:      MBARI<p>
 * @author Brian Schlining
 * @version 1.0
 */

 /*
  * $Log: HRDataCleaner.java,v $
  * Revision 1.2  2005/07/01 00:01:57  brian
  * Adding bug fixes an logging in an effort to port everything over to Heckel (Mac OS X)
  *
  * Revision 1.1  2002/11/22 17:19:53  brian
  * no message
  *
  * Revision 1.3  2002/07/10 16:24:01  brian
  * Modified to accomadate files with a few bad records or other problems.
  *
  */
package org.mbari.oasis;

import java.io.*;
import java.util.*;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mbari.io.*;
import org.mbari.util.*;

/**
 * Converts OASIS hydrorad data files to binary. The binary can then be used by hobilabs or other
 * processing programs.
 *
 * Useful class for taking the hex dump of Hydorad data that the Oasis can transmits and converting
 * it back to binary. This class also splits out different instruments into
 * different files and removes extraneous information. It currently works only with
 * Oasis files...it chops off the first token of each line. Typically this is
 * the Oasis time. However, it should be easy enough to modify this class if
 * it needs to be applied to non-oasis files.
 *
 * This class will loop through ALL files if a directory is specified. It also
 * accepts individual files as arguments.
 * @see AsciiHexReader
 * @author Brian Schlining
 * @version 26 Mar 2001
 *
 */
public class HRDataCleaner {
    
    /**
     * The constructor doesn't check for read write access to any files. Thats
     * done in the <i>seperate</i> method. It reads the file specified or ALL the
     * files in a directory if a directory is specified (SO make sure they're all
     * HR data files)<BR><BR>
     *
     * The output file(s) are dumped to the destination directory specifed. The
     * output file(s) will be named <prefix>_<instrumentType>.bin. So, if you
     * specify prefix=HR1 the resulting files could be name HR1_HR2.bin or
     * HR1_HR4.bin
     * @param infile A file object referering to the fiel or directory of files
     * containing data to seperate
     * @param dest The directory to write the resulting files to.
     * @param prefix The prefix to prepend to the output files name
     * @throws IOException Thrown is errors reading either infile or dest occurs
     */
    public HRDataCleaner(File infile, File dest, String prefix) throws IOException {
        
        if (infile.isDirectory()) {
            files = infile.listFiles();
        } else if (infile.isFile()) {
            files = new File[1];
            files[0] = infile;
        }
        
        this.dest   = dest;
        this.prefix = prefix;
    }
    
    /**
     * @param infile A file object referering to the file or directory of files
     * containing data to seperate
     * @param dest The directory to write the resulting files to.
     * @param prefix The prefix to prepend to the output files name
     */
    public HRDataCleaner(String infile, String dest, String prefix) throws IOException {
        this(new File(infile), new File(dest), prefix);
    }
    
    /**
     * This form of constructor will not append a prefix on to the output files
     * @param infile A file object referering to the fiel or directory of files
     * containing data to seperate
     * @param dest The directory to write the resulting files to.
     */
    public HRDataCleaner(String infile, String dest) throws IOException {
        this(new File(infile), new File(dest), "");
    }
    
    /**
     * This form of constructor will not append a prefix on to the output files
     * @param infile A file object referering to the fiel or directory of files
     * containing data to seperate
     * @param dest The directory to write the resulting files to.
     */
    public HRDataCleaner(File infile, File dest) throws IOException {
        this(infile, dest, "");
    }
    
    /**
     * This method will start the processing. A typical use would be:<br><br>
     *
     * HRDataCleaner hdc = new HRDataCleaner(....);<br>
     * hdc.seperate();<br>
     */
    public void seperate() throws IOException {
        
        if (!dest.isDirectory()) {
            //throw new IOException(dest.getCanonicalPath() + " is not a directory");
        }
        
        if (!dest.canWrite()) {
            throw new IOException("Unable to write to " + dest.getCanonicalFile());
        }
        
        HashMap        hm = new HashMap();  // This stores the OutputStreams
        BufferedOutputStream out;           // Name to use as pointer to active OutputStream
        
        for (int i = 0; i < files.length; i++) {  // Loop through all the files
            
            if (!files[i].canRead()) {
                log.warn("Unable to read " + files[i].getCanonicalPath());
                continue; //Skip the file
            }
            
            BufferedReader in = new BufferedReader(new FileReader(files[i]));
            
            String         str  = "",        // Stores the orginal string
                    S    = "";        // Stores the edited string
            
            while((str = in.readLine()) != null) { // Loop through each line
                
                StringTokenizer st  = new StringTokenizer(str); // Get a line of data
                try {
                    S = st.nextToken();        // Chop off the OASIS Time
                    //System.out.println(S);   // Shows date - good for debugging
                    S = st.nextToken();
                } catch (NoSuchElementException e) {
                    continue;
                }
                
                int start = S.indexOf("4852"); // Find "HR" in hex ascii
                if (start < 0) {
                    continue;
                }
                
                int end = S.toLowerCase().lastIndexOf("0d0a"); // Find the <CR><LF> at end of data
                if (end < 0) {
                    continue;
                    //end = S.length();
                } else {
                    end = end + 4;             // Adjust increment to include the <CR><LF>
                }
                
                try {
                    S = S.substring(start, end);                  // Trim the line
                } catch (Exception e) {
                    log.warn("Choked on line '" + S + "' in " + files[i]);
                    continue;
                }
                char[]          C   = S.toCharArray();        // Convert to a Character array
                
                if (!MathUtil.isEven((double) C.length))
                    continue;
                
                // Files contain hex, use AsciiHexReader to handle this
                AsciiHexReader  ahr = new AsciiHexReader(new CharArrayReader(C));
                char[] id = new char[3];
                char[] sn = new char[8];
                char   trash;
                try {
                    id[0] = ahr.readChar();
                    id[1] = ahr.readChar();
                    id[2] = ahr.readChar(); // Read 1st 3 characters, contains type
                    trash = ahr.readChar(); // Comma...ignore
                    for(int n = 0; n < 8; n++) {
                        sn[n] = ahr.readChar();
                    }
                } catch (EOFException e) {
                    continue;
                }
                
                //String          key = new String(id);         // Convert id to a String
                String key = new String(sn);                    // Convert id to a string
                
                // Check to see if the key, i.e the instrument type, already exists
                // if not open a new OutputStream to write to.
                if (!hm.containsKey(key)) {
                    File destination = new File(this.dest, key + "_" + prefix + ".bin");
                    hm.put(key, new BufferedOutputStream(new FileOutputStream(destination)));
                }
                
                // Get the OutputStream that corresponds to a given key
                out = (BufferedOutputStream) hm.get(key);
                
                ahr.reset();                  // Rewind to the start of the trimmed line
                for (int k = 0; k < (int) C.length/2; k++) {
                    try {
                        out.write(ahr.readUnsignedByte()); // read 2 hex Char
                    } catch (EOFException e) {
                        continue;
                    }
                }
                
            }
            
            in.close();                      // Close the input stream
            
        }
        
        // Close the OutputStreams when finished
        Object[] keys = hm.keySet().toArray();
        for (int i = 0; i < keys.length; i++) {
            out = (BufferedOutputStream) hm.get((String) keys[i]);
            out.close();
        }
        
        
    }
    
    /**
     * HRDataCleaner can be run from the command line as well as integrated into other programs.
     * The command line systax is:<br>
     *
     * org.mbari.oasis.HRDataCleaner <filename or dataDirectory> <outputDirectory> <outputNamePrefix><br>
     *
     * where filename specifies the name of the file to process. Alternatively, you may specify a data directory that contains
     * only OASIS formatted hydrorad files. In this case all files in the directory will be converted to binary files. Note:The directory
     * should only contain data from one of the channels. (Oasis normally splits the data to directories name hr1, hr2, hr3, hr4 which
     * each contain channesl 1, 2, 3, or 4 respectively). One file for each instrument will be created from data in a directory.
     * @param args Command line arguments
     */
    public static void main(String[] args) {
        if (args.length != 3) {
            System.err.println("\nUse as: org.mbari.oasis.HRDataCleaner <filename or dataDirectory> <outputDirectory> " +
                    "<outputNamePrefix>\n" );
            System.err.println("<filename> specifies the name of the file to process." +
                    " Alternatively you may specify a <dataDirectory>, the name of a directory that only" +
                    " contains HR data files. All files in the <dataDirectory> will be processed\n\n" +
                    "<outputDirectory> specifies the destination directory to write to.\n\n" +
                    "<outputNamePrefix> specifies the prefix to prepend onto the output filenames." +
                    " Its generally best to name it the same as the directory that contains the" +
                    " data files. (ex. hr1)");
            System.err.println("\nHRDataCleaner is used to turn the hex dumped hydrorad" +
                    " data thats returned by Oasis into binary files");
            System.exit(0);
        }
        
        try {
            HRDataCleaner hds = new HRDataCleaner(args[0], args[1], args[2]);
            hds.seperate();
        } catch (FileNotFoundException e) {
            log.fatal("Failed!!", e);
        } catch (IOException e) {
            log.fatal("Failed!!", e);
        } catch (Exception e) {
            log.fatal("Failed!!", e);
        }
    }
    
    File[] files;
    File dest;
    String prefix;
    
    private static final Log log = LogFactory.getLog(HRDataCleaner.class);
    
}


