package org.mbari.oasis;

import java.io.*;
import java.util.*;
import org.mbari.util.*;
import org.mbari.io.*;

/**
 * Removes duplicate data lines and sorts the data from OASIS data files.<br><br>
 *
 * If the file does not contain duplicate or unsorted daa it will not be rewritten.
 * @author Brian Schlining
 * @version 28 Mar 2001
 *
 */
public class DuplicateRemover {


    /**
     * @param f Name of the oasis data file to clean up.
     * @throws IOException Thrown if unalbe to read the file. Will also be thrown if the file needs to be
     * rewriten but write permission is not available.
     */
    public DuplicateRemover(String f) throws IOException {
        this(new File(f));
    }

    /**
     * @param f Name of the oasis data file to clean up.
     */
    public DuplicateRemover(File f) throws IOException {
        this.file = f;
        this.process();
    }

    /**
     * Use as: org.mbari.oasis.DuplicateRemover <filename or dataDirectory><br><br>
     *
     * Inputs: filename = the name of the file to sort and remove duplicates from<br>
     * dataDirectory = A directory of oasis data files. Duplicate Remover will loop
     * through all files in the directory and clean them.
     */
    public static void main(String[] args) {

      if (args.length > 1 || args.length < 1) {
         System.out.println("Usage: DuplicateRemover filename/directory");
         System.exit(0);
      }

      File f = new File(args[0]);
      if (f.isDirectory()) {
         String[] list = f.list();
         for (int i = 0; i < list.length; i++) {
            try {
               DuplicateRemover duplicateRemover = new DuplicateRemover(new File(f, list[i]));
            } catch (IOException e) {
               e.printStackTrace();
            }
         }
      } else {
         try {
            DuplicateRemover duplicateRemover = new DuplicateRemover(f);
         } catch (IOException e) {
            e.printStackTrace();
         }
      }
      System.exit(1);
   }

    private void process() throws IOException {

       //////////////////////
       // Read the data file
       BufferedReader in = new BufferedReader(new FileReader(this.file));

       ArrayList date = new ArrayList();    // Stores the decimal day
       ArrayList data = new ArrayList();    // Stores the entire record as a string
       ArrayList header = new ArrayList();  // Stores the header info

       String s;
       while((s = in.readLine()) != null) {
           if (s.startsWith("#")) {
               header.add(s);
           } else {
               data.add(s);
               StringTokenizer st = new StringTokenizer(s);
               date.add(new Double(Double.parseDouble(st.nextToken())));
           }
       }

        in.close();

        /////////////////////////////////////////////////////////////////
        // Get the dates from the file and sort getting rid of duplicates
        double[] dates  = new double[date.size()];
        for (int i = 0; i < dates.length; i++) {
            Double tmp = (Double) date.get(i);
            dates[i] = tmp.doubleValue();
        }

        int[] order = MathUtil.uniqueSort(dates); // gets the unigue sorted order

        boolean rewrite = false;
        if (order.length != dates.length) {
            rewrite = true;
        } else {
            for (int i = 1; i < dates.length; i++) {
                if ((dates[i] - dates[i - 1]) < 0) {
                    rewrite = true;
                    break;
                }

            }
        }


        ////////////////////////////////////////////
        // rewrite the file if duplicates are found


        if (rewrite) {
            BufferedWriter out = new BufferedWriter(new FileWriter(this.file));

            System.out.println("Rewriting " + this.file.getCanonicalPath());

            // write the header
            for (int i = 0; i < header.size(); i++) {
                out.write((String) header.get(i) + "\n");
            }

            // Write the data
            for (int i = 0; i < order.length; i++) {
                out.write((String) data.get(order[i])  + "\n");
            }

        }

    }

    File file;

}