
/**
 * Title:        HobiLabs data processing<p>
 * Description:  <p>
 * Copyright:    Copyright (c) Brian Schlining<p>
 * Company:      MBARI<p>
 * @author Brian Schlining
 * @version 26 Mar 2001
 */
package org.mbari.oasis;

import java.util.*;
import java.io.*;
import org.mbari.util.*;

/**
 * Constructs a CalibrationHistory instance from a OASIS mooring configuration
 * file.
 *
 * Use as:
 * CalibrationHistoryFactory chf = CalibrationHistoryFactory(String cfgFile);
 * CalibrationHistory ch = chf.getHistory(String instrument);
 *
 * where:
 * cfgFile is the name of the oasis configuration file to parse
 * instrument is the name of the instrument history to read (ex. specprr,
 * ctd, metsys, gps, etc.)
 */
public class CalibrationHistoryFactory {


   /**
     * @param cfgFile Name of the OASIS configuration file to parse
     * @see CalibrationHistory
     *
     */
    public CalibrationHistoryFactory(String cfgFile) {
      this(new File(cfgFile));
   }

   /**
     * @param cfgFile Name of the OASIS configuration file to parse
     * @see CalibrationHistory
     *
     */
    public CalibrationHistoryFactory(File cfgFile) {
   }

   /**
     * @param instrumentType Name of the Instrument type to get. Examples include:
     *    specprr, gps, ctd, hs2, hr2, hr4, metsys
     * @throws IOException Thrown if an error reading the calibration file occurs
     * @see CalibrationHistory
     *
     */
    public CalibrationHistory getHistory(String instrumentType) throws IOException {


      BufferedReader in;
      String str;
      CalibrationHistory ch = new CalibrationHistory(instrumentType);

      in = new BufferedReader(new FileReader(cfgFile));

      while ((str = in.readLine()) != null) {

         if (str.toLowerCase().startsWith(instrumentType.toLowerCase())) {
            StringTokenizer st = new StringTokenizer(str);
            if (st.countTokens() == 2) {
               String buf = st.nextToken(); // The instrument type
               String filename = new File(st.nextToken()).getName(); // The default calibration file w/o the path
               ch.setDefault(filename);
            }
            break;
         }

         if (str.startsWith("9") || str.startsWith("1") || str.startsWith("2")) {
            StringTokenizer st = new StringTokenizer(str);
            if (st.countTokens() > 4) { // Make sure its long genough

               String date1 = st.nextToken();  //Date1
               int yearIndex = 2;
               if (date1.length() >= 9)
                   yearIndex = 4;
               double year1 = Double.parseDouble(date1.substring(0, yearIndex));
               double day1  = Double.parseDouble(date1.substring(yearIndex, date1.length()));
               Date   start = TimeUtil.dayOfYearToDate(day1, year1); //Date1

               String buf = st.nextToken();  // The '-' sign

               String date2 = st.nextToken(); //Date2
               yearIndex = 2;
               if (date2.length() >= 9)
                   yearIndex = 4;
               double year2 = Double.parseDouble(date2.substring(0, yearIndex));
               double day2  = Double.parseDouble(date2.substring(yearIndex, date2.length()));
               Date   end = TimeUtil.dayOfYearToDate(day2, year2); //Date2

               String type = st.nextToken(); // is it the right instrument file?
               if (type.toLowerCase().startsWith(instrumentType.toLowerCase())) { // If it is a string
                  String cfg = st.nextToken();
                  String history = new File(st.nextToken()).getName(); // The calibration file w/o the path
                  ch.set(start, end, history);
               }
            }
         }
      }

      in.close();
      return ch;
   }

   private File cfgFile;

}