
/**
 * Title:        MBARI development package<p>
 * Description:  Default package environment<p>
 * Copyright:    Copyright (c) 1999<p>
 * Company:      MBARI<p>
 * @author Brian Schlining
 * @version 1.0
 */
package org.mbari.io;

import java.io.*;
import java.util.*;
import org.mbari.util.*;

/**
 * Read a concatenated file of pC02 data. and provide methods to access that data
 */
public class OasisPco2Reader2 {

   //////////////////////////////////////////
   // Constructors

   /**
    * Create a OasisPco2Reader and intialize memory.
    *
    * @param s  Name of file to read
    */
   public OasisPco2Reader2(String s) throws IOException {
      this.file = new File(s);

      this.init();
      this.read();
   }

   /**
    * Create a OasisPco2Reader and intialize memory.
    *
    * @param f  File object representing the file to be read
    */
   public OasisPco2Reader2(File f) throws IOException {
      this.file = f;

      this.init();
      this.read();
   }

   //////////////////////////////////////////
   // Public Methods

    /**
     * Return the data
     */
     public double[] getData() {
        return this.data;
     }

     public long[] getTime() {
        return this.dataTime;
     }

     public double[] getCalData() {
        return this.cal;
     }

     public long[] getCalTime() {
         return this.calTime;
     }

   // ///////////////////////////////////////
   // Private Methods

  /**
    * Makes a quick pass through the file and makes an array of the correct size.
    * It find the length of the rows that occur most and assumes that other rows
    * are erroneous and ignores them.
    */
   private void init() throws IOException {

      BufferedReader in = new BufferedReader(new FileReader(this.file));

      int      n,
               numCal  = 0,
               numData = 0;
      String   line;

      /* Loop though each line and sum up the number of data lines
         and the number of calibration lines */

      while ((line = in.readLine()) != null) {
         if (line.startsWith("#") || line.startsWith("%")) {

            // ignore comments
         }
         else {
            StringTokenizer   st = new StringTokenizer(line);

            n = st.countTokens();

            switch (n) {
               case COLUMNS_CAL:
                  int OK = line.indexOf('C'); // Ignore 'I' lines
                  if (OK != -1) {
                     numCal++;
                  }
                  break;
               case COLUMNS_DATA:
                  numData++;
                  break;
            }

         }
      }
      in.close();

      this.cal      = new double[numCal];
      this.data     = new double[numData];
      this.calTime  = new long[numCal];
      this.dataTime = new long[numData];
   }

   /**
    * This method does the actual reading of the data file. It tokenizes the
    * line by using white spaces
    *
    * @exception IOException
    */
   private void read() throws IOException {

      BufferedReader in = new BufferedReader(new FileReader(this.file));
      String         line;
      int            calRow = 0,
                     dataRow = 0;

      while ((line = in.readLine()) != null) {
         //System.out.println(line);
         if (line.startsWith("#") || line.startsWith("%")) {
            // ignore comments
         } else {

            StringTokenizer   st = new StringTokenizer(line);
            int               NumOfCol = st.countTokens();

            switch (NumOfCol) {
               case COLUMNS_CAL: // Read Calibration data
                  int OK = line.indexOf('C');
                  if (OK != -1) {

                     int   currentCol = 0;
                     double dayOfYear = 0,
                            year = 0;

                     while (st.hasMoreTokens()) {
                        switch (currentCol) {
                           case 0:
                              dayOfYear = new Double(st.nextToken()).doubleValue();
                              break;
                           case 1:
                              this.cal[calRow] = new Double(st.nextToken()).doubleValue();
                              break;
                           case 6:
                              year = new Double(st.nextToken()).doubleValue();
                              this.calTime[calRow] = DateConverter.doyToDate(dayOfYear, year).getTime();
                              break;
                           default:
                              String tmp = st.nextToken();
                              break;
                        }
                        currentCol++;

                     }
                     calRow++;
                  }
                  break;

               case COLUMNS_DATA: // Read the pCO2 data
                  int   currentCol = 0;
                  double dayOfYear = 0D,
                         year = 0D;

                  while (st.hasMoreTokens()) {
                     switch (currentCol) {
                        case 0:
                           dayOfYear = new Double(st.nextToken()).doubleValue();
                           break;
                        case 1:
                           this.data[dataRow] = new Double(st.nextToken()).doubleValue();
                           break;
                        case 5:
                           year = new Double(st.nextToken()).doubleValue();
                           this.dataTime[dataRow] = DateConverter.doyToDate(dayOfYear, year).getTime();
                           break;
                        default:
                           String tmp = st.nextToken();
                           break;
                     }
                     currentCol++;

                  }
                  dataRow++;
                  break;

            }
         }
      }
   }

   public static void main(String[] args) {
      try {
      OasisPco2Reader2 r = new  OasisPco2Reader2("g:/m1/data/pco2");
      } catch (Exception e) {
         e.printStackTrace();
      }
      System.out.println();
   }

   private File       file;
   private long[]     dataTime,
                      calTime;
   private double[]   data,
                      cal;
   private static final int COLUMNS_CAL = 7,
                            COLUMNS_DATA = 6;
}