
/**
 * Title:        MBARI development package<p>
 * Description:  Default package environment<p>
 * Copyright:    Copyright (c) 1999<p>
 * Company:      MBARI<p>
 * @author Brian Schlining
 * @version 1.0
 */
package org.mbari.io;

import java.io.*;
import java.util.*;
import org.mbari.util.*;

/**
 * Read a concatenated file of pC02 data. and provide methods to access that data
 */
public class OasisPco2Reader {

   //////////////////////////////////////////
   // Constructors

   /**
    * Create a OasisPco2Reader and intialize memory.
    *
    * @param s  Name of file to read
    */
    public OasisPco2Reader(String file) throws IOException {
        this(new File(file));
    }

   /**
    * Create a OasisPco2Reader and intialize memory.
    *
    * @param f  File object representing the file to be read
    */
    public OasisPco2Reader(File file) throws IOException {
        this.file = file;
        this.read();
    }

   //////////////////////////////////////////
   // Public Methods

    /**
     * Return the data
     */


     public double[] get(String key) throws IllegalArgumentException {
         key = key.toLowerCase();
         if (!hm.containsKey(key)) {
             throw new IllegalArgumentException("The specified type was not found");
         }
         ArrayList a = (ArrayList) hm.get(key);
         Object[] o = a.toArray();
         double[] data = new double[o.length];
         for (int i = 0; i < data.length; i++) {
             data[i] = ((Double) o[i]).doubleValue();
         }
         return data;
     }

     public double[] getTime() throws IllegalArgumentException{
         String key = "time_d";
         return get(key);
     }

     public double[] getTime(String type) throws IllegalArgumentException {
         String key = "time_" + type;
         return get(key);
     }

     public double[] getPco2() throws IllegalArgumentException {
         String key = "pco2_d";
         return get(key);
     }

     public double[] getPco2(String type) throws IllegalArgumentException {
         String key = "pco2" + type;
         return get(key);
     }

     public double[] getMillivolts() throws IllegalArgumentException {
         String key = "mv_d";
         return get(key);
     }

     public double[] getMillivolts(String type) throws IllegalArgumentException {
         String key = "mv" + type;
         return get(key);
     }

     public double[] getTemperature() throws IllegalArgumentException {
         String key = "temp_d";
         return get(key);
     }

     public double[] getTemperature(String type) throws IllegalArgumentException {
         String key = "temp" + type;
         return get(key);
     }

     public double[] getPressure() throws IllegalArgumentException {
         String key = "pressure_d";
         return get(key);
     }

     public double[] getPressure(String type) throws IllegalArgumentException {
         String key = "pressure" + type;
         return get(key);
     }

     public Set getTypes() {
         return hm.keySet();
     }



   // ///////////////////////////////////////
   // Private Methods

  /**
    * This method does the actual reading of the data file. It tokenizes the
    * line by using white spaces
    *
    * @exception IOException
    */
   private void read() throws IOException {

      BufferedReader in = new BufferedReader(new FileReader(this.file));
      String         line;
      hm = new HashMap();  // This stores the OutputStreams

      while ((line = in.readLine()) != null) {
         //System.out.println(line);
         if (line.startsWith("#") || line.startsWith("%")) {
            // ignore comments
         } else {

            ArrayList a;
            StringTokenizer   st       = new StringTokenizer(line);
            int               NumOfCol = st.countTokens();
            if(NumOfCol < MIN_NUM_COLUMNS) {
                continue;
            }

            String            key      = StringUtil.getToken(line, 6); // Look for I or C

            if (key.length() > 1) { // Keys can be I or C not 2000, 2001, etc.
                key = "D";          // If not I or C flag set key to D (for data)
            }

            String keyTime = "time_" + key.toLowerCase();
            if (!hm.containsKey(keyTime)) {
               hm.put(keyTime, new ArrayList());
            }
            double dayOfYear = new Double(st.nextToken()).doubleValue();
            double year      = new Double(StringUtil.getToken(line, NumOfCol)).doubleValue();
            Double time      = new Double((double) TimeUtil.dayOfYearToDate(dayOfYear, year).getTime());
            a = (ArrayList) hm.get(keyTime);
            a.add(time);

            for (int i = 0; i < COLUMNS.length; i++) {
                String tempKey = COLUMNS[i] + "_" + key.toLowerCase();
                if (!hm.containsKey(tempKey)) {
                    hm.put(tempKey, new ArrayList());
                }
                Double tempVar;
                try {
                    tempVar = new Double(st.nextToken());
                } catch (NumberFormatException e) {
                    tempVar = new Double(Double.NaN);
                }
                a = (ArrayList) hm.get(tempKey);
                a.add(tempVar);
            }

         }
      }
   }

   public static void test() {
      try {
      OasisPco2Reader r = new  OasisPco2Reader("g:/m1/data/pco2");
      Set s = r.getTypes();
      } catch (Exception e) {
         e.printStackTrace();
      }
      System.out.println();
   }

   private File file;
   private HashMap hm;

   // Specifies the minimum number of data columns that a row of data can have.
   // Rows with less than this are ignored.
   private static final int MIN_NUM_COLUMNS = 6;

   // Column 1 is dayOfYear and the last column is year so COLUMNS specifies
   // the name of columns 2:end-1
   public static final String[] COLUMNS = {"pco2", "mv", "temp", "pressure"};
   public static final String[] UNITS = {"parts per million", "millivolts",
     "degrees Celsius", "kilopascals"};
   public static final String[] LONG_NAME = {"pCO2", "sensor output voltage",
     "sensor internal temperature", "sensor internal pressure"};
}