
/**
 * Title:        HobiLabs data processing<p>
 * Description:  <p>
 * Copyright:    Copyright (c) Brian Schlining<p>
 * Company:      MBARI<p>
 * @author Brian Schlining
 * @version 1.0 19 Oct 2000
 */
package org.mbari.hobilabs;

import java.io.*;
import java.util.*;

/**
 * Reads the HS2 calibration files. The usage is:<br><br>
 *
 * HS2CalibrationFileReader hcfr = new HS2CalibrationFileReader( ... )<br>
 * Properties p = hcfr.getProperties();<br><br>
 *
 * The calibration information is stored in the properties object.
 */
public class HS2CalibrationFileReader {

   /**
    * @param filename Name of the calibration file to read
    */
   public HS2CalibrationFileReader(String filename) throws IOException {
      this.parse(new File(filename));

   }

   /**
    * @param filename File object of the calibration file to read
    */
   public HS2CalibrationFileReader(File file) throws IOException {
      this.parse(file);
   }

   /**
    * Returns a Properties object for the values found in the calibration file.
    *
    * @return A properties object of the calibration values
    */
   public Properties getProperties() {
       return prop;
   }

   /**
    * Private method used to parse the Hobilabs cal file format
    */
   private void parse(File file) throws IOException {
      if (file.isDirectory()) {
         throw new IOException("You dolt! " + file.getName() +
            " is a directory, not a file.");
      }

      if (!file.canRead()) {
         throw new IOException("Unable to read " + file.getName() + ". Check the files permissions.");
      }

      BufferedReader in = new BufferedReader(new FileReader(file));
      String S = null;

      // Loop variables
      boolean isTag = true;
      String tag = "none";
      String prefix = "";

      // Read the good stuff in this loop
      processLoop: while((S = in.readLine()) != null) {

         isTag = checkForTag(S);   // Look for markup tags

         if (isTag) {              // Process the markup tags
            StringTokenizer st1 = new StringTokenizer(S, "]");
            S = st1.nextToken();
            tag = S.substring(1,S.length());   // Get the tags name

            if (tag.startsWith("Channel")) {
               prefix = tag + ".";
            } else if (tag.startsWith("General") | tag.startsWith("Start")) {
               prefix = "";
            } else if (tag.startsWith("End")) {
               break processLoop;
            }
         } else {                   // Process the name-value pairs

            StringTokenizer st2 = new StringTokenizer(S, "=");
            try {
               String param = st2.nextToken();
               if (param != null) {
                  String name = prefix + param;
                  String value = st2.nextToken();
                  if (value == null) { value = "";}
                  this.prop.put(name, value);
               }
            } catch (NoSuchElementException e) {
               // Do nothing
            }
         }

      }

      in.close();

      String tempNom = prop.getProperty("TempNom");
      if (tempNom == null) {
         prop.put("TempNom", new String("1.0"));
      }

   }

   private boolean checkForTag(String S) {
      boolean isTag = false;
      if (S.startsWith("[") & S.endsWith("]")) {
         isTag = true;
      }

      return isTag;
   }



   private Properties prop = new Properties();

}