
/**
 * Title:        HobiLabs data processing<p>
 * Description:  <p>
 * Copyright:    Copyright (c) Brian Schlining<p>
 * Company:      MBARI<p>
 * @author Brian Schlining
 * @version 1.0
 */
package org.mbari.hobilabs;

import java.io.*;
import java.util.*;

/**
 * HRParser is the main data parser engine for Decoders and Converters in the
 * org.mbari.hobilabs.hr parckage. Most decoders in this package layer some
 * functionality on top of HRParser but all user this parsing engine.
 *
 * HRParser is used to parse a binary stream of data from a Hobilabs hydrorad
 * instrument. Data collected using MBARI's OASIS controller will be in ASCII
 * hexadecimal format and will need to be converted back to binary (using
 * AsciiHexInputStream)<br><br>
 *
 * A typical use of this class:<br>
 *   HRParser p = new HRParser( ... )
 *   p.parseFile();
 *   HRPacket[] data = p.get();
 *
 * @author Brian Schlining
 */
public class HRParser {

   /**
    * HRParser will accept any InputStream. Internally this will be changed
    * to a DataInputStream. To start the parsing of the file call the parseFile
    * method.
    *
    * @param in InputStream from the file to parse
    */
   public HRParser(InputStream in) throws IOException {
      this(new DataInputStream(in));
   }

   /**
    * @param in A DataInputStream from the file to parse
    */
   public HRParser(DataInputStream in) throws IOException {
      this.in = in;
      this.storage = new ArrayList();
   }

   /**
    * Call this method to start the actual processing.
    * @deprecated Use parse() instead
    */
   public void parseFile() throws IOException {
      this.parse();
   }

   /**
    * Call this method to start the actual processing.
    */
   public void parse() throws IOException {
      int taste;
      try {
         while (true) { // look for 0FF0
            taste = (int) in.readByte();
            // Each data packet starts with '0FF0'
            if (taste == 15) { // find OF
              taste = (int) in.readUnsignedByte();
               if (taste == 240) { // find F0
                  this.parsePacket();
               }
            }
         }
      } catch (EOFException e) {
          // Do Nothing
      }
   }

   /**
    * The actual reading is performed here.
    */
   private void parsePacket() throws IOException {
      HRPacket tmp = new HRPacket();
      tmp.RawTime  = in.readInt();
      tmp.Temp     = in.readFloat();
      tmp.Voltage  = in.readFloat();
      tmp.Depth    = in.readFloat();
      tmp.Process  = in.readShort();
      tmp.N        = in.readShort();
      tmp.Scale    = in.readFloat();
      tmp.Do       = in.readFloat();
      tmp.Dt       = in.readFloat();
      tmp.IntTime  = in.readInt();
      tmp.FirstPix = in.readShort();
      tmp.PixInc   = in.readShort();
      tmp.PixCount = in.readShort();
      tmp.Pixel = new double[tmp.PixCount];
      for (int i = 0; i < tmp.PixCount; i++) {
         if (tmp.Process == 0) {
            tmp.Pixel[i] = in.readShort();
         } else {
            tmp.Pixel[i] = in.readFloat();
         }
      }
      if (tmp.PixCount != 0) {   // Occasionally an error will make this zero. Ignore those cases
         this.storage.add(tmp);
      }
   }

   /**
    * This method is used to get the data after it's been parsed
    *
    * @return An array of HRPackets. Each Packet contains one sample.
    */
   public HRPacket[] get() {
      Object[] buf = this.storage.toArray();
      HRPacket[] out = new HRPacket[buf.length];
      for (int i = 0; i < out.length; i++) {
         out[i] = (HRPacket) buf[i];
      }
      return out;
   }

   private DataInputStream in;
   private ArrayList storage;

}