
/**
 * Title:        HobiLabs data processing<p>
 * Description:  <p>
 * Copyright:    Copyright (c) Brian Schlining<p>
 * Company:      MBARI<p>
 * @author Brian Schlining
 * @version 1.0
 */
package org.mbari.hobilabs;

import java.io.*;
import java.util.*;
import org.mbari.io.*;
import org.mbari.util.*;
import ucar.multiarray.*;
import ucar.netcdf.*;

public class HRMain {

   public HRMain(File calDirectory, File infile, String outfile) {
      this.calDirectory = calDirectory;
      this.infile = infile;
      this.outfile = outfile;
   }

   public HRMain(String calDirectory, String infile, String outfile) {
      this(new File(calDirectory), new File(infile), outfile);
   }

   public void process() throws IOException, IllegalArgumentException {

      if (!calDirectory.isDirectory())
         throw new IllegalArgumentException(calDirectory.getCanonicalPath() +
            " is not a directory.");

      // Read the data file
      HRBinaryDecoder d               = new HRBinaryDecoder(infile, false);
      d.decode();
      HRPacket[]      data            = d.getData();
      String[]        serialNumber    = d.getSerialNumbers();
      String          dataDescription = d.getDataDescription();
      StringTokenizer st              = new StringTokenizer(dataDescription, ",");
      String          channelS        = st.nextToken();
      String          dataType        = st.nextToken();
      String          dataUnits       = st.nextToken();
      StringTokenizer st2             = new StringTokenizer(channelS);
      String          channel         = st2.nextToken();
      channel = st2.nextToken().trim();
      String          instrumentType  = d.getInstrumentType();

      // Open the appropriate calibration files
      // Implement the following lines n the future
      String[] calList = this.calDirectory.list(new HRCalFileNameFilter(instrumentType.toLowerCase(), serialNumber[0]));
      Arrays.sort(calList);   // Sorted list of cal files
      HRCalibrationFileReader[] calFiles = new HRCalibrationFileReader[calList.length];
      double[] calTime = new double[calList.length];  // Get the time of each file
      for (int i = 0; i < calList.length; i++) {
         calFiles[i] = new HRCalibrationFileReader(new File(calDirectory, calList[i]));
         StringTokenizer st3 = new StringTokenizer(calList[i], ".");
         String tmp = st3.nextToken();
         tmp   = st3.nextToken();
         tmp   = st3.nextToken(); // Here's the word with the date
         int year, month, date;
         year  = Integer.parseInt(tmp.substring(0,4));
         month = Integer.parseInt(tmp.substring(4,6));
         date  = Integer.parseInt(tmp.substring(6,8));
         // Return time in seconds GMT
         calTime[i] = new GmtCalendar(year, month, date).getTime().getTime()/1000;
      }

      ////////////////////////////////
      // Determine which cal file to use
      int good = 0;  // Index of calFile to use
      if (data[0].RawTime < calTime[0]) {
            good = 0;
      } else if (data[0].RawTime > calTime[calTime.length - 1]) {
            good = calTime.length - 1;
      } else {
         for (int i = 0; i < calTime.length - 1; i++) {
            if ((data[0].RawTime >= calTime[i]) && (data[0].RawTime < calTime[i+1])) {
               good = i;
               break;
            }
         }
      }

      //////////////////////////////
      // Construct the netcdf schema
      // Define Dimensions
      Dimension timeD = new UnlimitedDimension("time");
      Dimension lambdaD = new Dimension("wavelength", data[0].PixCount);

      // Define Global Attributes
      Attribute[] globalAtt = {
         new Attribute("conventions", "MBARI/timeSeries/mooring/hydrorad"),
         new Attribute("creationDate", new Date().toString()),
         new Attribute("lastModified", new Date().toString()),
         new Attribute("mooring", "M1"),
         new Attribute("project", "Monterey Bay Time Series"),
         new Attribute("keywords", "spectroradiometer"),
         new Attribute("instrumentType", "Hobilabs " + instrumentType),
         new Attribute("serialNumber", serialNumber[0]),
         new Attribute("configurationFile", calList[good]),
         new Attribute("sourceFile", this.infile.getName()),
         new Attribute("contents", dataDescription),
      };

      // Define Coordinate variables
      Attribute[] lambdaA = {
         new Attribute("long_name", "wavelength"),
         new Attribute("units", "nanometers"),
      };
      ProtoVariable lambdaV = new ProtoVariable("wavelength", Float.TYPE,
       new Dimension[] {lambdaD}, lambdaA);

      Attribute[] timeA = {
         new Attribute("long_name", "time GMT"),
         new Attribute("units", "seconds since 1970-01-01 00:00:00"),
      };
      ProtoVariable timeV = new ProtoVariable("time", Double.TYPE,
       new Dimension[] {timeD}, timeA);

      // Define the variable (temp, voltage, depth, intTime)
      ProtoVariable tempV = new ProtoVariable("temp", Float.TYPE,
       new Dimension[] {timeD},
       new Attribute[] {
         new Attribute("long_name", "temperature"),
         new Attribute("units", "celsius"),
         new Attribute("symbol", "T"),
         new Attribute("_FillValue", -999f),
         new Attribute("missing_value", -999f),
       });
      ProtoVariable voltV = new ProtoVariable("voltage", Float.TYPE,
       new Dimension[] {timeD},
       new Attribute[] {
         new Attribute("long_name", "power supply voltage"),
         new Attribute("units", "volts"),
         new Attribute("symbol", "V"),
         new Attribute("_FillValue", -999f),
         new Attribute("missing_value", -999f),
       });
       ProtoVariable depthV = new ProtoVariable("depth", Float.TYPE,
       new Dimension[] {timeD},
       new Attribute[] {
         new Attribute("long_name", "sensor depth"),
         new Attribute("units", "unknown"),
         new Attribute("symbol", "Z"),
         new Attribute("_FillValue", -999f),
         new Attribute("missing_value", -999f),
       });
       ProtoVariable intV = new ProtoVariable("intTime", Float.TYPE,
       new Dimension[] {timeD},
       new Attribute[] {
         new Attribute("long_name", "integration time"),
         new Attribute("units", "seconds"),
         new Attribute("symbol", "I"),
         new Attribute("_FillValue", -999f),
         new Attribute("missing_value", -999f),
       });
       ProtoVariable dataV = new ProtoVariable("data", Float.TYPE,
       new Dimension[] {timeD, lambdaD},
       new Attribute[] {
         new Attribute("long_name", dataType),
         new Attribute("units", dataUnits),
         new Attribute("symbol", "Z"),
         new Attribute("_FillValue", -999f),
         new Attribute("missing_value", -999f),
       });

       // Create the schema
       Schema schema = new Schema(
        new ProtoVariable[] {lambdaV, timeV, tempV, voltV, depthV, intV, dataV},
        globalAtt);

       NetcdfFile nc = new NetcdfFile(outfile, true, true, schema);

      // Calculate wavelengths
      double[] lambda = HRUtil.calcWavelength(data[0], calFiles[good].getWave(channel));
      float[] lambdaF = new float[lambda.length];
      for (int i = 0; i < lambda.length; i++) {
         lambdaF[i] = (float) lambda[i];
      }

      // Assign the data to arrays
      double[] time = new double[data.length];
      float[]  temp = new float[data.length],
               voltage = new float[data.length],
               depth = new float[data.length],
               intT = new float[data.length];
      float[][] dat =  new float[data.length][lambda.length];
      for (int r = 0; r < data.length; r++) {
         time[r] = (float) data[r].RawTime;
         temp[r] = (float) data[r].Temp;
         voltage[r] = (float) data[r].Voltage;
         depth[r] = (float) data[r].Depth;
         intT[r] = (float) data[r].IntTime;
         dat[r] = new float[data[r].Pixel.length];
         for (int c = 0; c < dat[r].length; c++) {
            dat[r][c] = (float) data[r].Pixel[c];
         }

      }

      // Add the data to the netcdf file
      int[] origin = {0, 0};
      MultiArray m = new ArrayMultiArray(time);
      Variable v = nc.get(timeV.getName());
      v.copyin(origin, m);

      m = new ArrayMultiArray(dat);
      v = nc.get(dataV.getName());
      v.copyin(origin, m);

      int[] origin2 = {0};
      m = new ArrayMultiArray(lambdaF);
      v = nc.get(lambdaV.getName());
      v.copyin(origin2, m);

      m = new ArrayMultiArray(temp);
      v = nc.get(tempV.getName());
      v.copyin(origin2, m);

      m = new ArrayMultiArray(voltage);
      v = nc.get(voltV.getName());
      v.copyin(origin2, m);

      m = new ArrayMultiArray(depth);
      v = nc.get(depthV.getName());
      v.copyin(origin2, m);

      m = new ArrayMultiArray(intT);
      v = nc.get(intV.getName());
      v.copyin(origin2, m);

      nc.close();
   }

   /**
    * arg[0] = calfile directory
    * arg[1] = binary data file
    * arg[2] = destination directory
    */
   public static void main(String[] args) {
      if (args.length != 3) {
         System.err.println("Usage: java org.mbari.hobilabs.hr.HRMain " +
          "<Directory containing cal files> <infile> <outfile>");
         System.exit(0);
      }
      HRMain main = new HRMain(args[0], args[1], args[2]);
      try {
         main.process();
      } catch (Exception e) {
         e.printStackTrace();
      }

   }

   File calDirectory,
        infile;

   String outfile;
}