//$Header: /home/cvs/iag/brian/mbari/src/org/mbari/hobilabs/HRBinaryDecoder.java,v 1.3 2002/10/04 20:04:13 brian Exp $
package org.mbari.hobilabs;

import java.io.*;
import java.util.*;

/**
 * <p>A decoder that reads information from binary Hydrorad files. They're are two
 * ways to use this class.<br><br>
 *
 * 1) If you think all the data in the file is from the same instrument then use
 * this form (It's faster). This form does not read all serial numbers in a
 * binary file. This is the default method.<br><br>
 *
 * HRBinaryDecoder d = new HRBinaryDecoder( ..., false);<br>
 * d.decode();<br>
 * HRPacket[] data = d.getData();<br>
 * String desc = d.getDataDescription();<br>
 * String type = d.getInstrumentType();<br>
 * String[] sn = d.getSerialNumbers(); // sn will be a String[1] array<br><br>
 *
 * 2) If you need the serial numbers from each message.<br><br>
 *
 * HRBinaryDecoder d = new HRBinaryDecoder( ..., true);<br>
 * d.decode();<br>
 * HRPacket[] data = d.getData();<br>
 * String desc = d.getDataDescription();<br>
 * String type = d.getInstrumentType();<br>
 * String[] sn = d.getSerialNumbers(); // sn will be a String[data.length] array<br><br>
 * </p><hr>
 *
 * @author  : $Author: brian $
 * @version : $Revision: 1.3 $
 *
 * <hr><p><font size="-1" color="#336699"><a href="http://www.mbari.org">
 * The Monterey Bay Aquarium Research Institute (MBARI)</a> provides this
 * documentation and code &quot;as is&quot;, with no warranty, express or
 * implied, of its quality or consistency. It is provided without support and
 * without obligation on the part of MBARI to assist in its use, correction,
 * modification, or enhancement. This information should not be published or
 * distributed to third parties without specific written permission from
 * MBARI.</font></p><br>
 *
 * <font size="-1" color="#336699">Copyright 2002 MBARI.<br>
 * MBARI Proprietary Information. All rights reserved.</font><br><hr><br>
 *
 */

/*
 * $Log: HRBinaryDecoder.java,v $
 * Revision 1.3  2002/10/04 20:04:13  brian
 * no message
 *
 * Revision 1.2  2002/07/02 23:41:55  brian
 * Updating comments
 *
 */

/**
 * A decoder that reads information from binary Hydrorad files. They're are two
 * ways to use this class.<br><br>
 *
 * 1) If you think all the data in the file is from the same instrument then use
 * this form (It's faster). This form does not read all serial numbers in a
 * binary file. This is the default method.<br><br>
 *
 * HRBinaryDecoder d = new HRBinaryDecoder( ..., false);<br>
 * d.decode();<br>
 * HRPacket[] data = d.getData();<br>
 * String desc = d.getDataDescription();<br>
 * String type = d.getInstrumentType();<br>
 * String[] sn = d.getSerialNumbers(); // sn will be a String[1] array<br><br>
 *
 * 2) If you need the serial numbers from each message.<br><br>
 *
 * HRBinaryDecoder d = new HRBinaryDecoder( ..., true);<br>
 * d.decode();<br>
 * HRPacket[] data = d.getData();<br>
 * String desc = d.getDataDescription();<br>
 * String type = d.getInstrumentType();<br>
 * String[] sn = d.getSerialNumbers(); // sn will be a String[data.length] array<br><br>
 */
public class HRBinaryDecoder {

   /**
    * @param infile File object representing the file to decode
    */
   public HRBinaryDecoder(File infile) {
      this(infile, false);
   }

   /**
    * @param infile Name of the file to decode
    */
   public HRBinaryDecoder(String infile) {
      this(new File(infile), false);
   }

   /**
    * @param infile Name of the fiel to decode
    * @param getAllSerialNumbers true indicates that all serial numbers in the
    * file need to read. false indicates that only the very first one in the
    * file will be returned
    */
   public HRBinaryDecoder(String infile, boolean getAllSerialNumbers) {
      this(new File(infile), getAllSerialNumbers);
   }

   /**
    * @param infile File object of the file to decode
    * @param getAllSerialNumbers true indicates that all serial numbers in the
    * file need to read. false indicates that only the very first one in the
    * file will be returned.
    */
   public HRBinaryDecoder(File infile, boolean getAllSerialNumbers) {
      this.infile = infile;
      this.getAllSerialNumbers = getAllSerialNumbers;
   }

   /**
    * Method to process the binary file. THis must be called before using the
    * accessor methods.
    */
   public void decode() throws IOException {
      ////////////////////////
      // Check the inputs
//      if (!infile.canRead()) {
//         throw new IOException("Unable to read " + infile.getCanonicalPath());
//      }

      //////////////////////////
      // Converting the data file
      DataInputStream in = new DataInputStream(new BufferedInputStream(
         new FileInputStream(infile)));

      char         type;

      char[]       c            = new char[1],
                   serialNumber = new char[8];
      StringBuffer channelInfo  = new StringBuffer();
      boolean      firstLoop    = true;

      // This loop parses instrument info from the first complete message found
      // in the data file. It continues to loop to find all the serial numbers
      try {
         fullFileLoop: while (true) {

            c[0] = (char) in.readUnsignedByte();

            if (new Character(c[0]).equals(new Character('H'))) { // First find H

               c[0] = (char) in.readUnsignedByte();

               if (new Character(c[0]).equals(new Character('R'))) { // Second find R

                  c[0] = (char) in.readUnsignedByte();

                  if(Character.isDigit(c[0])) { // Third find a digit

                     type = c[0];
                     c[0] = (char) in.readUnsignedByte();

                     if(new Character(c[0]).equals(new Character(','))) { // Fourth find a comma

                        c[0] = (char) in.readUnsignedByte();

                        if(new Character(c[0]).equals(new Character('H'))) { // Fifth find H

                           c[0] = (char) in.readUnsignedByte();

                           if(new Character(c[0]).equals(new Character('R'))) { // Sixth find R

                              serialNumber[0] = 'H';
                              serialNumber[1] = 'R';
                              for(int i = 2; i < 8; i++) {
                                 serialNumber[i] = (char) in.readUnsignedByte();
                              }
                              // Store each records serial number
                              this.serialNumbers.add(new String(serialNumber));

                              if (firstLoop) {
                                 this.instrumentType = "HR" + type;
                                 c[0] = (char) in.readUnsignedByte(); // skip a line feed
                                 infoLoop: while (true) {
                                    firstLoop = false;
                                    int a = in.readUnsignedByte();
                                    if (a == 10) {      // Look for the final line feed
                                       break infoLoop; // If found get out of loop
                                    }
                                    channelInfo.append((char) a);
                                 }
                                 this.channelInformation = new String(channelInfo);
                                 // Don't read the whole file if the user thinks that
                                 // all the serial numbers are the same in a file
                                 if (!this.getAllSerialNumbers) {
                                    break fullFileLoop;
                                 }
                              }
                           }
                        }
                     }
                  }
               }
            }

         }
      } catch (EOFException e) {
         // Do nothing
      } finally {
         in.close();
      }

      // This is where the actual parsing of the data occurs
      in = new DataInputStream(new BufferedInputStream(
         new FileInputStream(infile)));
      HRParser p = new HRParser(in);
      p.parse();
      this.data = p.get();
      in.close();

   }

   /**
    * @return HRPacket[] an array of HRPackets. Each HRPacket contains a single
    * data record. Each HRPacket corresponds to a serial number returned by
    * getSerialNumbers if the getAllSerialNumbers is set to <b>true</b>.
    */
   public HRPacket[] getData() {
      return data;
   }

   /**
    * @param data this method is required by HRNetcdfConstructor
    */
   public void setData(HRPacket[] data) {
      this.data = data;
   }

   /**
    * @return String[] an array of serial Numbers. Each serial number corresponds
    * to an HRPacket (serialNumber[3] is for HRPacket[3]).
    */
   public String[] getSerialNumbers() {
      Object[] buf = this.serialNumbers.toArray();
      String[] out = new String[buf.length];
      for (int i = 0; i < out.length; i++) {
         out[i] = (String) buf[i];
      }
      return out;
   }

   /**
    * @return String Name of the instrument type. Will be "HR2" or "HR4".
    */
   public String getInstrumentType() {
      return instrumentType;
   }

   /**
    * Returns a string of the channel description. All data packets are assumed
    * to contain the same description so only a single descrtiption is returned
    * for a file.
    */
   public String getDataDescription() {
      return channelInformation;
   }

   public int getChannelNumber() {
      if (this.channel == -1) {  // Parse the channel out if it hasn't already been parsed
         StringTokenizer st       = new StringTokenizer(channelInformation, ",");
         String          channelS = st.nextToken(); // Get the first token (ex. Channel 1)
         st = new StringTokenizer(channelS);
         String          buf      = st.nextToken(); // Ignore 'Channel'
         this.channel = Integer.parseInt(st.nextToken().trim());   // Convert '1' to an int
      }
      return this.channel;
   }

   public String getDataType() {
      if (this.dataType == null) {
         StringTokenizer st       = new StringTokenizer(channelInformation, ",");
         String buf = st.nextToken(); // Ignore the first token (ex. Channel 1)
         this.dataType = st.nextToken().trim(); // Use the second token (ex. Lu_wet1)
      }
      return this.dataType;
   }

   public String getUnits() {
      if (this.units == null) {
         StringTokenizer st       = new StringTokenizer(channelInformation, ",");
         String buf = st.nextToken(); // Ignore the first token (ex. Channel 1)
         buf = st.nextToken();           // Ignore the data Type (ex. Lu_wet1
         this.units = st.nextToken().trim(); // Use the third token (ex. w/cm^2/nm))
      }
      return this.units;
   }

   private String     instrumentType,
                      channelInformation,
                      dataType,
                      units;
   private File       infile;
   private ArrayList  serialNumbers = new ArrayList();
   private HRPacket[] data;
   private boolean    getAllSerialNumbers;
   private int        channel = -1;

}