//$Header: /home/cvs/iag/brian/mbari/src/org/mbari/hobilabs/HRBinaryConverter.java,v 1.3 2002/10/04 20:04:04 brian Exp $
package org.mbari.hobilabs;

import java.io.*;
import java.text.*;
import java.util.*;
import org.mbari.io.*;
import org.mbari.util.*;

/**
 * <p>Useful class for converting binary hydrorad data files to ascii. This class requires
 * that complete Hydrorad messages are in the data. This includes the instrument
 * id,serial number, and channel information. If you need to parse a data file
 * that doesn't contain complete messages, ony data, then use HRParser. HRParser
 * is the engine used by HRBinaryDecoder to decode the data.</p>
 *
 * <p>The output from this is essentially the same as HRCAL.EXE from hobilabs. The
 * main differences are C-style comments (lines that begin with '#') and tab-
 * delimited rather than comma delimited data.</p><br>
 *
 * Use this class as:<br><br>
 *
 * <code>
 * HRBinaryConverter hbd = new HRBinaryConverter( ... );<br>
 * hbd.decode();<br><br>
 * </code>
 *
 * or call the main method.</p><hr>
 *
 * @author  : $Author: brian $
 * @version : $Revision: 1.3 $
 *
 * <hr><p><font size="-1" color="#336699"><a href="http://www.mbari.org">
 * The Monterey Bay Aquarium Research Institute (MBARI)</a> provides this
 * documentation and code &quot;as is&quot;, with no warranty, express or
 * implied, of its quality or consistency. It is provided without support and
 * without obligation on the part of MBARI to assist in its use, correction,
 * modification, or enhancement. This information should not be published or
 * distributed to third parties without specific written permission from
 * MBARI.</font></p><br>
 *
 * <font size="-1" color="#336699">Copyright 2002 MBARI.<br>
 * MBARI Proprietary Information. All rights reserved.</font><br><hr><br>
 *
 */

/*
 * $Log: HRBinaryConverter.java,v $
 * Revision 1.3  2002/10/04 20:04:04  brian
 * no message
 *
 * Revision 1.2  2002/07/02 23:41:55  brian
 * Updating comments
 *
 */

public class HRBinaryConverter {

   /**
    * @param infile Name of the binary data file to read
    * @param calfile Name of the calibration file
    * @param outfile Name of the ascii text file to create
    */
   public HRBinaryConverter(File infile, File calfile, File outfile, String channel)
    throws IllegalArgumentException {
      this.infile = infile;
      this.calfile = calfile;
      this.outfile = outfile;
      this.channel = channel;
   }

   public HRBinaryConverter(String infile, String calfile, String outfile, String channel)
    throws IllegalArgumentException {
      this(new File(infile), new File(calfile), new File(outfile), channel);
   }

   /**
    * @exception IOException
    */
   public void convert() throws IOException {

      ////////////////////////
      // Check the inputs
//      if (!infile.canRead()) {
//         System.err.println("Unable to read " + infile.getCanonicalPath());
//         return; //Skip the file
//      }
//
//      if (!calfile.canRead()) {
//         System.err.println("Unable to read " + calfile.getCanonicalPath());
//         return; //Skip the file
//      }
//
//      if (outfile.isDirectory()) {
//         throw new IOException(outfile.getCanonicalPath() + " already exists as a directory.");
//      }

      ////////////////////////
      // Read the needed info from the cal file
      HRCalibrationFileReader cal = new HRCalibrationFileReader(this.calfile);
      double[] W = cal.getWave(this.channel);
      String id = cal.getID(); // serialNumber found in calibration file

      //////////////////////////
      // Converting the data file
      HRBinaryDecoder in = new HRBinaryDecoder(this.infile, false);
      in.decode();
      HRPacket[] data = in.getData();
      String channelInfo = in.getDataDescription();
      String instrumentType = in.getInstrumentType();
      String[] sn = in.getSerialNumbers();
      String serialNumber;
      try {
          serialNumber = sn[0];
      }
      catch (Exception e) {
          e.printStackTrace();
          serialNumber = id;
      }
      ///////////////////////
      // Write the output file

      // Open the file for writing
      BufferedWriter out = new BufferedWriter(new FileWriter(outfile));

      // Write the header
      out.write("# Hydrorad processed data file, " +
         "org.mbari.hobilabs.hr.HRBinaryDecoder v1.1\n");
      out.write("# File created on " + new Date() + "\n#\n");
      out.write("# Instrument: " + instrumentType + ", " + serialNumber + "\n" );
      out.write("# Source file: " + infile.getCanonicalPath() + "\n#\n");
      out.write("# Calibration file: " + calfile.getCanonicalPath() + "\n");

      if (!serialNumber.equals(id))
         out.write("# WARNING: The serial number in the calibration file used to process this data did not match the one in the data file!\n");

      out.write("# This file contains: " + channelInfo + "\n");
      out.write("# RawTime\tTemp\tVoltage\tDepth\tProcess\tN\tScale\tDo\tDt\tIntTime\t" +
         "FirstPix\tPixInc\tPixCount\t");

      // Calculate the wavelengths
      double firstPix = data[0].FirstPix,
             pixInc = data[0].PixInc,
             pixCount = data[0].PixCount,
             pixelNum = 0;

      double[] lambda = new double[(int) pixCount];

      NumberFormat nf = NumberFormat.getNumberInstance();
      nf.setMaximumFractionDigits(1);
      nf.setMinimumFractionDigits(1);
      nf.setMinimumIntegerDigits(3);

      for (int i = 0; i < pixCount; i++) {
         pixelNum = firstPix + pixInc*i;
         lambda[i] = W[0] + W[1] * pixelNum + W[2] * pixelNum * pixelNum;
         String term = null;
         term = i==(pixCount - 1) ? "\n" : "\t";
         out.write(nf.format(lambda[i]) + term); // write out wavelengths
      }

      nf.setMaximumFractionDigits(10);
      nf.setMinimumFractionDigits(10);
      nf.setMinimumIntegerDigits(1);

      NumberFormat nf2 = NumberFormat.getNumberInstance();
      nf2.setMaximumFractionDigits(2);
      nf2.setMinimumFractionDigits(2);
      nf2.setMinimumIntegerDigits(1);

      // Loop through each data sample and write it out to a file
      for (int i = 0; i < data.length; i++) {
         out.write(data[i].RawTime + "\t");
         out.write(nf2.format(data[i].Temp) + "\t");
         out.write(nf2.format(data[i].Voltage) + "\t");
         out.write(nf2.format(data[i].Depth) + "\t");
         out.write(data[i].Process + "\t");
         out.write(data[i].N + "\t");
         out.write(data[i].Scale + "\t");
         out.write(data[i].Do + "\t");
         out.write(data[i].Dt + "\t");
         out.write(data[i].IntTime + "\t");
         out.write(data[i].FirstPix + "\t");
         out.write(data[i].PixInc + "\t");
         out.write(data[i].PixCount + "\t");
         for (int j = 0; j < data[i].Pixel.length; j++) {
            out.write(nf.format(data[i].Pixel[j]) + "\t");
         }
         out.newLine();

      }

      out.close();


   }

   public static void main(String[] args) {
      if (args.length != 4) {
         System.out.println("\nUsage: org.mbari.hobilabs.HRBinaryConverter <input file> <calibration file>  <output file> <channel>");
         System.exit(0);
      }
      HRBinaryConverter op = new HRBinaryConverter (args[0], args[1], args[2], args[3]);

      try {
         op.convert();
      } catch (IOException e) {
         System.out.println(" Error: " + e.getMessage());
         e.printStackTrace();
      }

   }



   File infile,
        outfile,
        calfile;

   String channel;

}