/************************************************************************/
/* Copyright 1997 MBARI							*/
/************************************************************************/
/* $Header: argextract.c,v 3.0 99/05/12 10:11:30 bobh Exp $		    */
/* Summary  : Program to extract data files from ARGOS data		*/
/* Filename : argextract.c						*/
/* Author   : Bob Herlien (rah)						*/
/* Project  : OASIS							*/
/* $Revision: 3.0 $							    */
/* Created  : 13 Jan 1997 from argos.c					*/
/************************************************************************/
/* Modification History:						*/
/* 13jan97 rah - created from argos.c					*/
/* $Log:	argextract.c,v $
 * Revision 3.0  99/05/12  10:11:30  10:11:30  bobh (Bob Herlien)
 * Added tstring, misc changes
 * 
 * Revision 2.9  98/08/24  13:45:58  13:45:58  bobh (Bob Herlien)
 * Archiving sources after M2/M3 & Eqpac deployments of 1998
 * 
 * Revision 2.8  98/03/17  11:11:45  11:11:45  bobh (Bob Herlien)
 * Archiving sources prior to porting to DOS/Windows
 * 
 * Revision 1.1  97/10/27  09:53:19  09:53:19  bobh (Bob Herlien)
 * Initial revision
 * 
*/
/************************************************************************/

#include <stdio.h>			/* Unix standard I/O		*/
#include <stdlib.h>			/* Unix standard library	*/
#include <mbariTypes.h>			/* MBARI standard types		*/
#include <mbariConst.h>			/* MBARI standard constants	*/
#include <time.h>			/* Unix time library		*/
#include <decode.h>			/* OASIS data definitions	*/
#include <string.h>			/* Unix string library		*/
#include <argex.h>			/* ARGOS buffer definitions	*/
#include <hobi.h>

#define SPEC_VOLTS	FALSE
#define NUM_MSGS	100		/* Max messages to store & compare */
#define BUFSIZE		1024		/* Line buffer size		   */
#define FLUOR_CHAN	0		/* Analog channel for Wetstar fluor*/
#define TEMP_CHAN	2		/* Analog channel for temperature  */
#define OBATT_CHAN	3		/* Analog channel for OASIS battery*/
#define ABATT_CHAN	6		/* Analog channel for ARGOS battery*/

#define CHEM_TIME(i)	((Flt64)(i)/(8.0*365.0))

typedef enum				/********************************/
{					/* Return value from getXXXMsg	*/
    MSG_OK,				/* Successful conversion	*/
    BAD_MSG,				/* Bad message data		*/
    END_OF_FILE				/* End of input file		*/
} MsgRtn;				/********************************/


/********************************/
/*	External Functions	*/
/********************************/

Extern char	*get_can_name( char *dataname );
Extern Status	read_cfg( char **cfgname );
Extern double	decode_prr_chan( Int16 ival, SpecChanCal *sccp );
Extern Flt64	decode_ac9_temp( Int32 count );
Extern Void	print_sensor( Int sensor, char *fmt, ... );
Extern Void	print_error( char *fmt, ... );
Extern Void	close_sensor_files( Void );


/********************************/
/*	External Data		*/
/********************************/

Extern char	*optarg;		/* Option argument from getopt()    */
Extern Int	optind;			/* Option index from getopt()	    */
Extern Analog	analog[ANALOG_CHANS];	/* Analog calibration		    */
Extern SpecCal  spec_cal;		/* Struct to hold Spect calibrations*/
Extern Int32	gmt_offset;		/* GMT offset in hours		    */
Extern Nat32	temp_offset;		/* Offset for NO3 temperatures	    */
Extern MBool	smallConfig;		/* TRUE for small PTT data config   */
Extern MBool	reverseNO3;		/* TRUE to reverse 0m & 20m NO3s    */
Extern Int32	revision;		/* Software & msg type revision nmbr*/


/********************************/
/*	Module Local Data	*/
/********************************/

MLocal char		*cfgp = NULL;	/* Ptr to name of OASIS config file */
MLocal MBool		decode_all = TRUE;
MLocal MBool		decode_msg[NUM_MSG_TYPES];

MLocal Nat32		numMsgs = 0;
MLocal Nat32		msgCnt = 0;
MLocal char		buffer[BUFSIZE];
MLocal ArgosInMsg	msgs[NUM_MSGS];

MLocal Int32		msgTypesRev1[] =
{ CHEM, SPEC_0M_NOON, SPEC_0M_AM, SPEC_20M, AC9_1, AC9_2, MISC };
MLocal Nat32            mType;


/*
MessageItemMap: indexes message contents by message type (10 argos buffers daily) 
and revision (message mapping change).
*/
MLocal DataItem *MessageItemMap[MAX_MESSAGE_TYPES][MAX_MESSAGE_ITEMS];

/********************************/
/*	Forward Declarations	*/
/********************************/

MBool	processCommandLine ( Int32 argc, char **argv );
Void	decodeFile( char *filename );
Void	printOneMsg( ArgosInMsg *msgp );
Void    printIsus(Nat32 filenum, Flt64 day, void *msgp);
Void    printHS2(Nat32 filenum, Flt64 day, void *msgp);
DataItem *createDataItem(int id,Nat16 offset, Nat16 size, void (*exfunc(ArgosMsg *msg)));
Void    exPCO2(ArgosMsg *pMsg, Nat16 mapIndex);
Void    exOSMO_NO3();
Void    exPRR();
Void    exCTD();
Void    exISUS();
Void    exSAT();
Void    exWETSTAR();
Void    exMCP();
Void    exSHUTTER();
Void    exHS2();

/************************************************************************/
/* Function    : main							*/
/* Purpose     : Main routine						*/
/* Inputs      : argc, argv						*/
/* Outputs     : Integer, 0 for success, 1 for failure			*/
/************************************************************************/
	Int32
main( Int32 argc, char **argv )
{
    char	*filename, *cp;
    Reg Int32	i;

    if ( !processCommandLine(argc, argv) )
	exit( 1 );

    putenv( "TZ=GMT0" );

    for ( i = optind; i < argc; i++ )
    {
	filename = argv[i];
	get_can_name( filename );
	cp = cfgp;
	if ( read_cfg(&cp) != OK )
	    printf("Can't read configuration file \"%s\"\n", cp);
	else
	    decodeFile( filename );
    }

} /* main() */


/************************************************************************/
/* Function : use_msg							*/
/* Purpose  : Print Usage Message					*/
/* Inputs   : Name of program						*/
/* Outputs  : None							*/
/************************************************************************/
	Void
use_msg( char *s )
{
    fprintf( stderr, "Usage: %s [-c cfg_file] [-i msg_number]\n" );
    fprintf( stderr, "-c specifies a configuration file\n" );
    fprintf( stderr, "-i decodes a specified message number\n" );
    fprintf( stderr, "default is for data from Service Argos\n");

} /* use_msg() */


/************************************************************************/
/* Function : processCommandLine					*/
/* Purpose  : Read the arguments from the command line			*/
/* Inputs   : argc, argv from main() routine				*/
/* Outputs  : TRUE if arguments OK, else FALSE				*/
/************************************************************************/
	MBool
processCommandLine ( Int32 argc, char **argv )
{
    Int32	c, i;

    for ( i = 0; i < NUM_MSG_TYPES; i++ )
	decode_msg[i] = FALSE;

    while ( (c = getopt(argc, argv, "c:i:")) != EOF )
	switch( c )
	{
	  case 'c':
	    cfgp = optarg;
	    break;

	  case 'i':
	    decode_all = FALSE;
	    decode_msg[atoi(optarg)] = TRUE;
	    break;

	  default:
	    use_msg( argv[0] );
	    return( FALSE );
	}

    return( TRUE );

} /* processCommandLine() */


/************************************************************************/
/* Function    : getLine						*/
/* Purpose     : Get one line of ARGOS message				*/
/* Inputs      : FILE pointer, buffer ptr				*/
/* Outputs     : MSG_OK or END_OF_FILE					*/
/************************************************************************/
	MsgRtn
getLine( FILE *fp, char * buffer )
{
    Reg Int32	len;

    buffer[0] = '\0';

    while( TRUE )
    {
	if ( fgets(buffer, BUFSIZE, fp) == NULL )
	    return( END_OF_FILE );

	if ( (len = strlen(buffer)) > 0 )
	{
	    if ( buffer[len] == '\n' )
		buffer[len] = '\0';
	    return( MSG_OK );
	}
    }

} /* getLine() */


/************************************************************************/
/* Function    : getSvcArgosMsg						*/
/* Purpose     : Get an ARGOS message in Service Argos format		*/
/* Inputs      : FILE ptr, Buffer ptr, message ptr, time ptr, number of msgs*/
/* Outputs     : Message return code					*/
/************************************************************************/
	MsgRtn
getSvcArgosMsg( FILE *fp, char *buffer, WordMsg *msgp,
	        struct tm *tmPtr, Int32 *nmsgs )
{
    Nat32		i, j, dat[4];

    if ( getLine(fp, buffer) == END_OF_FILE )
	return( END_OF_FILE );

    if ( sscanf(buffer, " %d-%d-%d %d:%d:%d %d %x %x %x %x",
		&tmPtr->tm_year, &tmPtr->tm_mon, &tmPtr->tm_mday,
		&tmPtr->tm_hour, &tmPtr->tm_min, &tmPtr->tm_sec,
		nmsgs, &dat[0], &dat[1], &dat[2], &dat[3]) < 11 )
	return( BAD_MSG );

    tmPtr->tm_year -= 1900;
    tmPtr->tm_mon -= 1;

    for ( i = 0; i < 4; i++ )
	msgp->wd_data[i] = (Nat16)dat[i];

    for ( i = 4; i < ARGOS_WORDS; i += 4 )
    {
	if ( getLine(fp, buffer) == END_OF_FILE )
	    return( END_OF_FILE );

	if ( sscanf(buffer, " %x %x %x %x", 
		    &dat[0], &dat[1], &dat[2], &dat[3]) < 4 )
	    return( BAD_MSG );

	for ( j = 0; j < 4; j++ )
	    msgp->wd_data[i + j] = (Nat16)dat[j];
    }

    return( MSG_OK );

} /* getSvcArgosMsg() */


/************************************************************************/
/* Function    : msgMatch						*/
/* Purpose     : See if two ARGOS messages are the same			*/
/* Inputs      : Two Message ptrs					*/
/* Outputs     : TRUE if match, else FALSE				*/
/************************************************************************/
	MBool
msgMatch( ByteMsg *msgp1, ByteMsg *msgp2 )
{
    Int32	i;

    for ( i = 0; i < ARGOS_LEN; i++ )
	if ( msgp1->by_data[i] != msgp2->by_data[i] )
	    return( FALSE );

    return( TRUE );

} /* msgMatch() */


/************************************************************************/
/* Function    : findMatchingMessage					*/
/* Purpose     : Find a matching ARGOS message from those already received*/
/* Inputs      : Message ptr						*/
/* Outputs     : Message number of matching message, or ERROR		*/
/************************************************************************/
	Int32
findMatchingMessage( ByteMsg *msgp )
{
    Nat32	i;

    for ( i = 0; i < msgCnt; i++ )
	if ( msgMatch(msgp, &msgs[i].msg.rawb) )
	    return( i );

    return( ERROR );

} /* findMatchingMessage() */


/************************************************************************/
/* Function    : getMessages						*/
/* Purpose     : Get all incoming ARGOS messages			*/
/* Inputs      : FILE pointer						*/
/* Outputs     : None							*/
/************************************************************************/
	Void
getMessages( FILE *fp, char *buffer )
{
    struct tm		msgTime, *tmPtr;
    time_t		msg_time_t;
    MsgRtn		state;
    ArgosUnion		msg;
    Int32		msgNum, nmsgs;

    numMsgs = msgCnt = 0;
    bzero( (void *)msgs, sizeof(msgs) );
    time( &msg_time_t );
    tmPtr = gmtime( &msg_time_t );
    memcpy( (void *)&msgTime, (void *)tmPtr, sizeof(struct tm) );
    msgTime.tm_isdst = 0;
    state = MSG_OK;

    while ( state != END_OF_FILE )
    {
	nmsgs = 1;

	state = getSvcArgosMsg( fp, buffer, &msg.raww, &msgTime, &nmsgs );

	if ( state == MSG_OK )
	{
	    msg_time_t = mktime( &msgTime );

	    if ( (msgNum = findMatchingMessage(&msg.rawb)) != ERROR )
	    {
		if ( msg_time_t < msgs[msgNum].first_msg )
		    msgs[msgNum].first_msg = msg_time_t;
		if ( msg_time_t > msgs[msgNum].last_msg )
		    msgs[msgNum].last_msg = msg_time_t;
		msgs[msgNum].msg_cnt += nmsgs;
	    }
	    else
	    {
		if ( msgCnt >= NUM_MSGS )
		{
		    printOneMsg( &msgs[0] );
		    memmove( (void *)msgs, (void *)&msgs[1],
			     (NUM_MSGS - 1) * sizeof(ArgosInMsg) );
		    msgCnt--;
		}
		msgs[msgCnt].first_msg = msg_time_t;
		msgs[msgCnt].last_msg = msg_time_t;
		msgs[msgCnt].msg_cnt = nmsgs;
		memcpy( (void *)&msgs[msgCnt].msg.rawb, (void *)&msg,
		        sizeof(ArgosUnion) );
		numMsgs += nmsgs;
		msgCnt++;
	    }
	}
    }

} /* getMessages() */


/************************************************************************/
/* Function    : fileDate						*/
/* Purpose     : Convert Unix time_t to file date spec			*/
/* Inputs      : Time of first ARGOS hit				*/
/* Outputs     : Decimal year, e.g. 97.003 for Jan 2, 1997		*/
/************************************************************************/
	Flt64
fileDate( time_t timeToConvert )
{
    time_t	adjustedTime;
    struct tm	*tmPtr;
    Int32	year;
    Flt64	daysPerYear;

    adjustedTime = timeToConvert + (3600 * gmt_offset) - (24 * 3600);
    tmPtr = gmtime( &adjustedTime );

/*    year = tmPtr->tm_year + 1900;*/
    year = tmPtr->tm_year;

    daysPerYear = ((year % 4) == 0) ? 366.0 : 365.0;

    return( (Flt64)year + ((Flt64)tmPtr->tm_yday / daysPerYear) );
    
} /* fileDate() */


/************************************************************************/
/* Function    : printChem						*/
/* Purpose     : Print data from Chem message				*/
/* Inputs      : File day, Chem message ptr				*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printChem( Flt64 day, ChemMsg *msgp )
{
    Nat32	i, msbs;
    Int32	co2;
    
    msbs = getIntelword( (Byte *)&msgp->ch_pco2_msbs );

    for ( i = 0; i < CHEM_SAMPLES; i++ )
    {
	co2 = msgp->ch_pco2[i] | (((msbs >> (i + i)) & 3) << 8);
	if ( co2 & 0x200 )
	    co2 |= ~0x3ff;				/* Sign extend	*/
	print_sensor( CO2_FILE, "%8.4f  %4d\n",  day + CHEM_TIME(i), co2 );
    }

} /* printChem() */


/************************************************************************/
/* Function    : printChem4						*/
/* Purpose     : Print data from Chem message (rev 4 format)		*/
/* Inputs      : File day, Chem message ptr				*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printChem4( Flt64 day, Chem4Msg *msgp )
{
    Reg Byte	*p;
    Int32	co2;
    Nat16	ipitch, iroll, itemp, idepth;
    Nat32	i;

    p = msgp->ch_pco2;

    for ( i = 0; i < CHEM_SAMPLES; i+=2 )
    {
	co2 = *p++ << 4;
	co2 |= ((*p >> 4) & 0x0f);
	if ( co2 & 0x800 )
	    co2 |= 0xfffff000;				/* Sign extend	*/
	print_sensor( CO2_FILE, "%8.4f  %4d\n",  day + CHEM_TIME(i), co2 );

	co2 = ((*p++ & 0x0f) << 8);
	co2 |= *p++;
	if ( co2 & 0x800 )
	    co2 |= 0xfffff000;				/* Sign extend	*/
	print_sensor( CO2_FILE, "%8.4f  %4d\n",  day + CHEM_TIME(i+1), co2 );
    }

    ipitch = getIntelword( (Byte *)&msgp->ch_prr_pitch );
    iroll = getIntelword( (Byte *)&msgp->ch_prr_roll );
    itemp = getIntelword( (Byte *)&msgp->ch_prr_temp );
    idepth = getIntelword( (Byte *)&msgp->ch_prr_depth );

    print_sensor( SPEC_MISC_FILE, "%8.4f  %6.1f %6.1f %7.2f %7.2f", day,
		  decode_prr_chan(ipitch, &spec_cal.spc_cal[0][10]),
		  decode_prr_chan(iroll, &spec_cal.spc_cal[0][11]),
		  decode_prr_chan(itemp, &spec_cal.spc_cal[2][8]),
		  decode_prr_chan(idepth, &spec_cal.spc_cal[2][9]) );
    
    ipitch = getIntelword( (Byte *)&msgp->ch_prr_pitch1230 );
    iroll = getIntelword( (Byte *)&msgp->ch_prr_roll1230 );
    itemp = getIntelword( (Byte *)&msgp->ch_prr_temp1230 );
    idepth = getIntelword( (Byte *)&msgp->ch_prr_depth1230 );

    print_sensor( SPEC_MISC_FILE, " %6.1f %6.1f %7.2f %7.2f\n",
		  decode_prr_chan(ipitch, &spec_cal.spc_cal[0][10]),
		  decode_prr_chan(iroll, &spec_cal.spc_cal[0][11]),
		  decode_prr_chan(itemp, &spec_cal.spc_cal[2][8]),
		  decode_prr_chan(idepth, &spec_cal.spc_cal[2][9]) );
    
} /* printChem4() */

/************************************************************************/
/* Function    : printChemR5						*/
/* Purpose     : Print data from Chem message (rev 5 fmt)		*/
/* Inputs      : File day, Chem message ptr				*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printChemR5( Flt64 day, Chem5Msg *msgp )
{
    Reg Byte	*p;
    Int32	co2;
    Nat16	ipitch, iroll, itemp, idepth;
    Nat32	i;

    p = msgp->ch_pco2;

    for ( i = 0; i < CHEM_SAMPLES; i+=2 )
    {
	co2 = *p++ << 4;
	co2 |= ((*p >> 4) & 0x0f);
	if ( co2 & 0x800 )
	    co2 |= 0xfffff000;				/* Sign extend	*/
	print_sensor( CO2_FILE, "%8.4f  %4d\n",  day + CHEM_TIME(i), co2 );

	co2 = ((*p++ & 0x0f) << 8);
	co2 |= *p++;
	if ( co2 & 0x800 )
	    co2 |= 0xfffff000;				/* Sign extend	*/
	print_sensor( CO2_FILE, "%8.4f  %4d\n",  day + CHEM_TIME(i+1), co2 );
    }

    itemp = getIntelword( (Byte *)&msgp->ch_prr_temp );
    idepth = getIntelword( (Byte *)&msgp->ch_prr_depth );

    print_sensor( SPEC_MISC_FILE, "%8.4f  %7.2f %7.2f\n", day,
		  decode_prr_chan(itemp, &spec_cal.spc_cal[2][8]),
		  decode_prr_chan(idepth, &spec_cal.spc_cal[2][9]) );

    printIsus( ISUS_FILE,day,(void *)msgp );
    
} /* printChemR5() */

/************************************************************************/
/* Function    : printNO3						*/
/* Purpose     : Print an NO3 buffer					*/
/* Inputs      : File number, file day, NO3 msg ptr			*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printNO3( Nat32 filenum, Flt64 day, No3Msg *no3p )
{
    Reg Nat32	i, no3;
    Reg Byte	*p;

    p = no3p->no3;

    for ( i = 0; i < CHEM_SAMPLES; )
    {
	no3 = *p++ << 4;
	no3 |= ((*p >> 4) & 0x0f);
	print_sensor( filenum, "%8.4f  %4d\n", day + CHEM_TIME(i), no3 );
	i++;
	no3 = ((*p++ & 0x0f) << 8);
	no3 |= *p++;
	print_sensor( filenum, "%8.4f  %4d\n", day + CHEM_TIME(i), no3 );
	i++;
    }

} /* printNO3() */


/************************************************************************/
/* Function    : printTemp						*/
/* Purpose     : Print Temperature from NO3				*/
/* Inputs      : File number, file day, NO3 msg ptr			*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printTemp( Nat32 filenum, Flt64 day, No3Msg *no3p )
{
    Reg Nat32	i;

    for ( i = 0; i < CHEM_SAMPLES; i++ )
	print_sensor( filenum, "%8.4f  %5.2f\n", day + CHEM_TIME(i),
		      (Flt64)((Nat32)(no3p->temp[i]) + temp_offset)/20.0);

} /* printTemp() */

/************************************************************************/
/* Function    : printIsus						*/
/* Purpose     : Print ISUS data (stored in sp_lu[] of SpecMsg)		*/
/* Inputs      : ?							*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printIsus(Nat32 filenum, Flt64 day, void *msgp)
{
 Nat16 i;
 unsigned char w[2];
 Nat16 ival;

  print_sensor(filenum,"%8.4f ",day);

  if (mType == SPEC_0M_1000){
    for(i=0;i<(ASPEC_CHANS);i++){

      bcopy((Byte *)&((SpecMsg *)msgp)->sp_lu[i],(Byte *)&w[0],2);
      print_sensor(filenum,"%0.2x %0.2x ",w[0],w[1]);

    }
    print_sensor(filenum,"S");
  }else{
  if (mType == CHEM){
    for(i=0;i<(CH_UNUSED_SIZE);i++){

      bcopy((Byte *)&((Chem5Msg *)msgp)->ch_unused[i],(Byte *)&w[0],2);
      print_sensor(filenum,"%0.2x %0.2x ",w[0],w[1]);

    }
    print_sensor(filenum,"C");
  }
  }
  print_sensor(filenum,"\n");

  return;
}/*printIsus()*/

/************************************************************************/
/* Function    : printHS2						*/
/* Purpose     : Print HS2 data (stored in sp_lu[] of SpecMsg)		*/
/* Inputs      : ?							*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printHS2(Nat32 filenum, Flt64 day, void *msgp)
{
 Nat16 i;
 unsigned char hs2buf[HS2BUF_SIZE];
 Nat16 ival;

  print_sensor(filenum,"%8.4f ",day);

  bcopy((Byte *)&((SpecMsg *)msgp)->sp_lu[0],(Byte *)&hs2buf[0],HS2BUF_SIZE);
  print_sensor(filenum,"%0.2x%0.2x ",hs2buf[BOS_SNORM1],hs2buf[BOS_SNORM1+1]);
  print_sensor(filenum,"%0.2x%0.2x ",hs2buf[BOS_SNORM2],hs2buf[BOS_SNORM2+1]);
  print_sensor(filenum,"%0.2x%0.2x ",hs2buf[BOS_SNORM3],hs2buf[BOS_SNORM3+1]);
  print_sensor(filenum,"%0.2x ",(hs2buf[BOS_GAIN1]>>4));
  print_sensor(filenum,"%0.2x ",(hs2buf[BOS_GAIN2]&0x0F));
  print_sensor(filenum,"%0.2x ",(hs2buf[BOS_GAIN3]&0x0F));
  print_sensor(filenum,"%0.2x ",hs2buf[BOS_TEMPRAW]);

  print_sensor(filenum,"\n");

  return;
}/*printHS2()*/

/************************************************************************/
/* Function    : printSpec						*/
/* Purpose     : Print a PRR Spectro message				*/
/* Inputs      : File number, File day, Spectro message ptr,		*/
/*		  Address tags for Ed, Lu chans				*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printSpec( Nat32 filenum, Flt64 day, SpecMsg *msgp, Int32 edAdr,
	   Int32 luAdr, Int32 edChan, Int32 luChan )
{
    Reg Int32	i;
    Reg Nat16	ival;
    Flt64	ed, lu;

    print_sensor( filenum, "%8.4f  ", day );
    
    for ( i = 0; i < ASPEC_CHANS; i++ )
    {
	ival = getIntelword( (Byte *)&msgp->sp_ed[i] );
#if SPEC_VOLTS
	ed = decode_prr_chan( ival, NULL );
#else
	ed = decode_prr_chan( ival, &spec_cal.spc_cal[edAdr][i+edChan] );
#endif
	print_sensor( filenum, "%9.5f ", ed );
    }

    for ( i = 0; i < ASPEC_CHANS; i++ )
    {
	ival = getIntelword( (Byte *)&msgp->sp_lu[i] );
#if SPEC_VOLTS
	lu = decode_prr_chan( ival, NULL );
#else
	lu = decode_prr_chan( ival, &spec_cal.spc_cal[luAdr][i+luChan] );
#endif
	print_sensor( filenum, "%9.5f ", lu );
    }

} /* printSpec() */

/************************************************************************/
/* Function    : printSpecR5						*/
/* Purpose     : Print a PRR Spectro message	rev 5 format		*/
/* Inputs      : File number, File day, Spectro message ptr,		*/
/*		  Address tags for Ed, Lu chans				*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printSpecR5( Nat32 filenum, Flt64 day, SpecMsg *msgp, Int32 edAdr,
	   Int32 luAdr, Int32 edChan, Int32 luChan )
{
    Reg Int32	i;
    Reg Nat16	ival;
    Flt64	ed, lu;

    print_sensor( filenum, "%8.4f  ", day );
    
    for ( i = 0; i < ASPEC_CHANS; i++ )
    {
	ival = getIntelword( (Byte *)&msgp->sp_ed[i] );
#if SPEC_VOLTS
	ed = decode_prr_chan( ival, NULL );
#else
	ed = decode_prr_chan( ival, &spec_cal.spc_cal[edAdr][i+edChan] );
#endif
	print_sensor( filenum, "%9.5f ", ed );
    }

    if (mType == SPEC_0M_1000)
    {
       printIsus( ISUS_FILE, day, (void *)msgp );/* Lu has ISUS */
    }else
      /*if( filenum == SAT1_FILE )*/
    if ( mType == SAT1 )
    {
      printHS2( HS2_FILE, day, (void *)msgp );/* Lu has HS2 */	
    }else
    {
      for ( i = 0; i < ASPEC_CHANS; i++ )
	{
	  ival = getIntelword( (Byte *)&msgp->sp_lu[i] );
#if SPEC_VOLTS
	  lu = decode_prr_chan( ival, NULL );
#else
	  lu = decode_prr_chan( ival, &spec_cal.spc_cal[luAdr][i+luChan] );
#endif
	  print_sensor( filenum, "%9.5f ", lu );
	}
    }

} /* printSpecR5() */

/************************************************************************/
/* Function    : printAc9First						*/
/* Purpose     : Print First AC-9 message buffer			*/
/* Inputs      : File day, AC-9.1 message ptr				*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printAc9First( Flt64 day, Ac9Msg *ac9p )
{
    Reg Int32	i;
    Flt64	ac9;

    print_sensor( AC9_1_FILE, "%8.4f  ", day );

    for ( i = 0; i < AC9_1_CHANS; i++ )
    {
	ac9 = (Flt64)(getIntelword((Byte *)&ac9p->ac_data[i])) / 1000.0;
	print_sensor( AC9_1_FILE, "%9.5f ", ac9 );
    }

    print_sensor( AC9_1_FILE, "\n" );

} /* printAc9First() */


/************************************************************************/
/* Function    : printAc9Second						*/
/* Purpose     : Print Second AC-9 message buffer			*/
/* Inputs      : File day, AC-9.2 message ptr				*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printAc9Second( Flt64 day, Ac92Msg *ac9p )
{
    Reg Int32	i;
    Flt64	ac9, temp, chl, flow;
    Nat16	pump;

    print_sensor( AC9_2_FILE, "%8.4f  ", day );

    for ( i = 0; i < AC9_2_CHANS; i++ )
    {
	ac9 = (Flt64)(getIntelword((Byte *)&ac9p->ac_data[i])) / 1000.0;
	print_sensor( AC9_2_FILE, "%9.5f ", ac9 );
    }

    temp = decode_ac9_temp( getIntelword((Byte *)&ac9p->ac_temp) );
    print_sensor( AC9_2_FILE, "%9.5f ", temp );

    for ( i = 0; i < CHEM_SAMPLES; i++ )
    {
	chl = (Flt64)(getIntelword((Byte *)&ac9p->ac_chl[i])) / 1000.0;
	print_sensor( AC9_2_FILE, "%9.5f ", chl );
    }

    pump = getIntelword( (Byte *)&ac9p->ac_pump );
    flow = ((analog[FLOW_CHAN].a * (Flt64)(pump)) +
	     analog[FLOW_CHAN].b) * analog[FLOW_CHAN].c;

    print_sensor( AC9_2_FILE, "%9.5f %d\n", flow, pump );

} /* printAc9Second() */


/************************************************************************/
/* Function    : printMisc						*/
/* Purpose     : Print Miscellaneous message buffer			*/
/* Inputs      : File day, Misc msg ptr					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printMisc( Flt64 day, MiscMsg *msgp )
{
    Nat32	i, val;
    Int32	co2;
    Flt64	flt1, flt2, flt3;

    val = getIntelword( (Byte *)&msgp->ms_pco2 );
    co2 = val & 0x3ff;
    if ( co2 & 0x200 )
	co2 |= ~0x3ff;					/* Sign extend	*/

    print_sensor( CO2_CAL_FILE, "%8.4f  %3d  %02d00\n", 
		  day, co2, ((val >> 10) & 7) * 3 );
	    
    val = getIntellong( (Byte *)&msgp->ms_oasis );
    flt1 = (((analog[TEMP_CHAN].a * (Flt32)(val & 0x3ff)) +
	     analog[TEMP_CHAN].b) * analog[TEMP_CHAN].c) + analog[TEMP_CHAN].d;
    flt2 = ((analog[OBATT_CHAN].a * (Flt32)((val >> 10) & 0x3ff)) +
	    analog[OBATT_CHAN].b) * analog[OBATT_CHAN].c;
    flt3 = ((analog[ABATT_CHAN].a * (Flt32)((val >> 20) & 0x3ff)) +
	    analog[ABATT_CHAN].b) * analog[ABATT_CHAN].c;

    print_sensor( OASIS_FILE, "%8.4f  %5.2f %8.2f %8.2f\n",
		  day, flt1, flt2, flt3 );

    if ( !smallConfig )
    {
#if SPEC_VOLTS
	flt1 = decode_prr_chan( getIntelword((Byte *)&msgp->ms_mcp10), NULL );
	flt2 = decode_prr_chan( getIntelword((Byte *)&msgp->ms_mcp30), NULL );
#else
	flt1 = decode_prr_chan( getIntelword((Byte *)&msgp->ms_mcp10),
			        &spec_cal.spc_cal[2][10] );
	flt2 = decode_prr_chan( getIntelword((Byte *)&msgp->ms_mcp30),
			        &spec_cal.spc_cal[2][11] );
#endif
	print_sensor( MCP_FILE, "%8.4f  %8.2f %8.2f\n",
		     day, flt1, flt2 );

    }

} /* printMisc() */


/************************************************************************/
/* Function    : printMiscRev2						*/
/* Purpose     : Print Miscellaneous message buffer (Rev 1/2)		*/
/* Inputs      : File day, Misc msg ptr					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printMiscRev2( Flt64 day, MiscMsg *msgp )
{
    printMisc( day, msgp );

    if ( reverseNO3 )
    {
	printNO3( NO3_0_FILE, day, &msgp->ms_no3 );
	printTemp( TEMP0_FILE, day, &msgp->ms_no3 );
    }
    else
    {
	printNO3( NO3_20_FILE, day, &msgp->ms_no3 );
	printTemp( TEMP20_FILE, day, &msgp->ms_no3 );
    }

} /* printMiscRev2() */


/************************************************************************/
/* Function    : printMiscRev3						*/
/* Purpose     : Print Miscellaneous message buffer (Rev 3)		*/
/* Inputs      : File day, Misc msg ptr					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printMiscRev3( Flt64 day, Misc3Msg *msgp )
{
    Reg Nat32	i, ifluor;
    Reg Byte	*fluorp;
    Flt64	fluor;

    printMisc( day, (MiscMsg *)msgp );

    fluorp = msgp->ms_wetstar.fl_0m;

    for ( i = 0; i < CHEM_SAMPLES; i++ )
    {
	ifluor = (Nat32)(fluorp[i]);
	fluor = (((analog[FLUOR_CHAN].a * (Flt64)(ifluor)) +
		  analog[FLUOR_CHAN].b) * analog[FLUOR_CHAN].c) +
		      analog[FLUOR_CHAN].d;
	print_sensor( WETSTAR_FILE, "%8.4f  %5.2f\n", day + CHEM_TIME(i),
		      fluor );
    }

} /* printMiscRev3() */


/************************************************************************/
/* Function    : printMiscRev4						*/
/* Purpose     : Print Miscellaneous message buffer (Rev 4)		*/
/* Inputs      : File day, Misc msg ptr					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printMiscRev4( Flt64 day, Misc4Msg *msgp )
{
    Nat32	val, index, index20m;
    Reg Byte	*fluorp;
    Int32	co2;
    Flt64	flt1, flt2, flt3;

    val = getIntelword( (Byte *)&msgp->ms_pco2 );
    co2 = val & 0xfff;
    if ( co2 & 0x800 )
	co2 |= ~0xfff;					/* Sign extend	*/

    print_sensor( CO2_CAL_FILE, "%8.4f  %3d  %02d00\n", 
		  day, co2, 3 * ((val >> 12) & 7) );
	    
    val = getIntellong( (Byte *)&msgp->ms_oasis );
    flt1 = (((analog[TEMP_CHAN].a * (Flt32)(val & 0x3ff)) +
	     analog[TEMP_CHAN].b) * analog[TEMP_CHAN].c) + analog[TEMP_CHAN].d;
    flt2 = ((analog[OBATT_CHAN].a * (Flt32)((val >> 10) & 0x3ff)) +
	    analog[OBATT_CHAN].b) * analog[OBATT_CHAN].c;
    flt3 = ((analog[ABATT_CHAN].a * (Flt32)((val >> 20) & 0x3ff)) +
	    analog[ABATT_CHAN].b) * analog[ABATT_CHAN].c;

    print_sensor( OASIS_FILE, "%8.4f  %5.2f %8.2f %8.2f\n",
		  day, flt1, flt2, flt3 );

    fluorp= msgp->fl_0m;
    for ( index = index20m = 0; index < CHEM_SAMPLES; index++ )
    {
	if ( index & 1 )
	{
	    val = (*fluorp++ & 0x0f) << 8;
	    val |= *fluorp++;
	}
	else
	{
	    val = *fluorp++ << 4;
	    val |= ((*fluorp >> 4) & 0x0f);
	}

	flt1 = (((analog[FLUOR_CHAN].a * (Flt64)(val>>2)) +
		  analog[FLUOR_CHAN].b) * analog[FLUOR_CHAN].c) +
		      analog[FLUOR_CHAN].d;
	print_sensor( WETSTAR_FILE, "%8.4f  %6.3f", day + CHEM_TIME(index),
		      flt1 );
	if ( (index != 1) && (index != 7) )
	{
	    val = getIntelword( (Byte *)&msgp->fl_20m[index20m] );
	    flt1 = decode_prr_chan(val, &spec_cal.spc_cal[2][11]);
	    if ( flt1 < 20.0 )
		print_sensor( WETSTAR_FILE, " %6.3f", flt1 );
	    index20m++;
	}
	print_sensor( WETSTAR_FILE, "\n" );
    }

} /* printMiscRev4() */


/************************************************************************/
/* Function    : crcOneWord						*/
/* Purpose     : Check CRC-12 on one 16 bit word			*/
/* Inputs      : Initial CRC, word to check				*/
/* Outputs     : Resulting CRC						*/
/************************************************************************/
	Nat16
crcOneWord( Nat16 initCRC, Nat16 wordToCheck )
{
    Reg Nat32	bitCnt;
    Reg Nat16	crc;

    crc = initCRC;

    for ( bitCnt = 0; bitCnt < 16; bitCnt++ )
    {
	crc <<= 1;
	if ( wordToCheck & (0x8000 >> bitCnt) )
	    crc |= 1;
	if ( crc & CRC_MSB )
	    crc ^= CRC12;
    }

    return( crc );

} /* crcOneWord() */


/************************************************************************/
/* Function    : checkCRC						*/
/* Purpose     : Check CRC-12 on incoming message			*/
/* Inputs      : Word Message Pointer					*/
/* Outputs     : TRUE if CRC is OK, else FALSE				*/
/* Comment     : OASIS uses the standard CRC-12 polynomial 0x180f	*/
/*		 However, it puts the checksum and type information in	*/
/*		 the first word of the message, which it checks LAST	*/
/************************************************************************/
	MBool
checkCRC( WordMsg *msgp )
{
    Reg Nat32	cnt;
    Reg Nat16	crc;

    for ( cnt = 1, crc = 0; cnt < ARGOS_WORDS; cnt++ )
	crc = crcOneWord( crc, getIntelword((Byte *)&msgp->wd_data[cnt]) );

    return( crcOneWord(crc, getIntelword((Byte *)&msgp->wd_data[0])) == 0 );

} /* checkCRC() */


/************************************************************************/
/* Function    : checkSum						*/
/* Purpose     : Calculate Checksum on incoming message (Revision 1)	*/
/* Inputs      : Byte Message Pointer					*/
/* Outputs     : TRUE if checksum is OK, else FALSE			*/
/* Comment     : Revision 1 of OASIS software used a simple checksum	*/
/************************************************************************/
	MBool
checkSum( ByteMsg *msgp )
{
    Reg Nat32	cnt;
    Byte	cksum;

    for ( cnt = 2, cksum = 0; cnt < ARGOS_LEN; cnt++ )
	cksum ^= msgp->by_data[cnt];

    return( cksum == msgp->by_data[1] );

} /* checkSum() */


/************************************************************************/
/* Function    : printOneMsgRev2					*/
/* Purpose     : Print one received ARGOS message in Rev 1/2 format	*/
/* Inputs      : Message Pointer					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printOneMsgRev2( ArgosInMsg *msgp )
{
    Nat32	msgType;
    Nat16	ifluor;
    Flt64	fdate, fluor;
    
    if ( revision == 1 )
    {
	msgType = msgp->msg.rawb.by_data[0];
	if ( (msgType & 0x0f) != ((msgType >> 4) & 0x0f) )
	    return;
	if ( !checkSum(&msgp->msg.rawb) )
	    return;
	msgType = msgTypesRev1[msgType & 0x0f];
    }
    else
    {
	msgType = (msgp->msg.rawb.by_data[1] >> 4) & 0x0f;
	if ( !checkCRC(&msgp->msg.raww) )
	    return;
    }

    if ( !decode_all && !decode_msg[msgType] )
	return;

    fdate = fileDate( msgp->first_msg );

    switch( msgType )
    {
      case CHEM:
	printChem( fdate, &msgp->msg.chem );
	if ( reverseNO3 )
	{
	    printNO3( NO3_20_FILE, fdate, &msgp->msg.chem.ch_no3 );
	    printTemp( TEMP20_FILE, fdate, &msgp->msg.chem.ch_no3 );
	}
	else
	{
	    printNO3( NO3_0_FILE, fdate, &msgp->msg.chem.ch_no3 );
	    printTemp( TEMP0_FILE, fdate, &msgp->msg.chem.ch_no3 );
	}
	break;

      case SPEC_0M_NOON:
	ifluor = getIntelword( (Byte *)&msgp->msg.spec0m.sp_misc );
	fluor = (((analog[FLUOR_CHAN].a * (Flt64)(ifluor)) +
		   analog[FLUOR_CHAN].b) * analog[FLUOR_CHAN].c) +
		   analog[FLUOR_CHAN].d;
	printSpec( SPEC_NOON_FILE, fdate, &msgp->msg.spec0m,
		   SPEC_0M_ED, SPEC_0M_LU, 0, SPEC_NOON_LU_CHAN );
	print_sensor( SPEC_NOON_FILE, "%9.5f\n", fluor );
	break;

      case SPEC_0M_AM:
	ifluor = getIntelword( (Byte *)&msgp->msg.spec0m.sp_misc );
	fluor = (((analog[FLUOR_CHAN].a * (Flt64)(ifluor)) +
		   analog[FLUOR_CHAN].b) * analog[FLUOR_CHAN].c) +
		   analog[FLUOR_CHAN].d;
	printSpec( SPEC_1030_FILE, fdate, &msgp->msg.spec0m_1030,
		   SPEC_0M_ED, SPEC_0M_LU, 0, SPEC_AM_LU_CHAN );
	print_sensor( SPEC_1030_FILE, "%9.5f\n", fluor );
	break;

      case SPEC_20M:
	ifluor = getIntelword( (Byte *)&msgp->msg.spec0m.sp_misc );
#if SPEC_VOLTS
	fluor = decode_prr_chan( ifluor, NULL );
#else
	fluor = decode_prr_chan( ifluor, &spec_cal.spc_cal[2][12] );
#endif
	printSpec( SPEC_20_FILE, fdate, &msgp->msg.spec20m,
		   SPEC_20M_ED, SPEC_20M_LU, 0, 0 );
	print_sensor( SPEC_20_FILE, "%9.5f\n", fluor );
	break;

      case AC9_1:
	printAc9First( fdate, &msgp->msg.ac9 );
	break;

      case AC9_2:
	printAc9Second( fdate, &msgp->msg.ac92 );
	break;

      case MISC:
	printMiscRev2( fdate, &msgp->msg.misc );
	break;

      default:
	print_error( "Bad Message Type %d\n", msgType );
    }

    print_sensor( ARGOS_FILE, "%8.4f  %2d  %3d\n", 
		  fdate, msgType, msgp->msg_cnt );

} /* printOneMsgRev2() */


/************************************************************************/
/* Function    : printOneMsgRev3_4					*/
/* Purpose     : Print one received ARGOS message in Rev 3 or 4 format	*/
/* Inputs      : Message Pointer					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printOneMsgRev3_4( ArgosInMsg *msgp, Int32 rev )
{
    Nat32	msgType;
    Nat16	misc, val;
    Reg Nat32	i;
    Flt64	fdate, lu;
    
    if ( (rev != 3) && (rev != 4) )
    {
	print_error("Internal error - wrong message revision\n");
	return;
    }

    msgType = (msgp->msg.rawb.by_data[1] >> 4) & 0x0f;
    if ( !checkCRC(&msgp->msg.raww) )
	return;

    if ( !decode_all && !decode_msg[msgType] )
	return;

    fdate = fileDate( msgp->first_msg );

    switch( msgType )
    {
      case CHEM:
	if ( rev == 3 )
	{
	    printChem( fdate, &msgp->msg.chem );
	    printNO3( NO3_0_FILE, fdate, &msgp->msg.chem.ch_no3 );
	    printTemp( TEMP0_FILE, fdate, &msgp->msg.chem.ch_no3 );
	}
	else
	    printChem4( fdate, &msgp->msg.chem4 );
	break;

      case SPEC_0M_NOON:
	misc = getIntelword( (Byte *)&msgp->msg.spec0m.sp_misc );
	printSpec( SPEC_NOON_FILE, fdate, &msgp->msg.spec0m,
		   SPEC_0M_ED, SPEC_0M_LU, 0, SPEC_NOON_LU_CHAN );
	print_sensor( SPEC_NOON_FILE, "%u\n", misc );
	break;

      case SPEC_0M_1230:
	misc = getIntelword( (Byte *)&msgp->msg.spec0m.sp_misc );
	printSpec( SPEC_0M_1230_FILE, fdate, &msgp->msg.spec0m_1230,
		   SPEC_0M_ED, SPEC_0M_LU, 0,
		   ((rev == 3) ? SPEC_1230_LU_CHAN : SPEC_1230_LU_CHAN_REV4) );
	print_sensor( SPEC_0M_1230_FILE, "%u\n", misc );
	break;

      case SPEC_20M:
	misc = getIntelword( (Byte *)&msgp->msg.spec0m.sp_misc );
	printSpec( SPEC_20_FILE, fdate, &msgp->msg.spec20m,
		   SPEC_20M_ED, SPEC_20M_LU, 0, 0 );
	if ( rev == 3 )
	    print_sensor( SPEC_20_FILE, "%u\n", misc );
	else
	{
	    print_sensor( SPEC_20_FILE, "\n" );
#if SPEC_VOLTS
	    print_sensor( MCP_FILE, "%8.4f 10 %9.5f\n", fdate,
			  decode_prr_chan(misc, NULL));
#else
	    print_sensor( MCP_FILE, "%8.4f 10 %9.5f\n", fdate,
		   decode_prr_chan(misc, &spec_cal.spc_cal[2][10]));
#endif
	}
	break;

      case SPEC_20M_1230:
	misc = getIntelword( (Byte *)&msgp->msg.spec0m.sp_misc );
	printSpec( SPEC_20M_1230_FILE, fdate, &msgp->msg.spec20m,
		   SPEC_20M_ED, SPEC_20M_LU, 0, 0 );
	if ( rev == 3 )
	    print_sensor( SPEC_20M_1230_FILE, "%u\n", misc );
	else
	{
	    print_sensor( SPEC_20M_1230_FILE, "\n" );
#if SPEC_VOLTS
	    print_sensor( MCP_FILE, "%8.4f 30 %9.5f\n", fdate,
			  decode_prr_chan(misc, NULL));
#else
	    print_sensor( MCP_FILE, "%8.4f 30 %9.5f\n", fdate,
		   decode_prr_chan(misc, &spec_cal.spc_cal[2][14]));
#endif
	}
	break;

      case SPEC_0M_DARK:
	printSpec( SPEC_0M_DARK_FILE, fdate, &msgp->msg.spec0m,
		   SPEC_0M_ED, SPEC_0M_LU, 0, SPEC_NOON_LU_CHAN );
	print_sensor( SPEC_0M_DARK_FILE, "\n" );
	break;

      case SPEC_20M_DARK:
	printSpec( SPEC_20M_DARK_FILE, fdate, &msgp->msg.spec20m,
		   SPEC_20M_ED, SPEC_20M_LU, 0, 0 );
	print_sensor( SPEC_20M_DARK_FILE, "\n" );
	break;

      case SAT1:
	printSpec( SAT1_FILE, fdate, &msgp->msg.sat1, 
		   SAT1_ED, SAT1_LU, SAT1_ED_CHAN, SAT1_LU_CHAN );
	print_sensor( SAT1_FILE, "\n" );
	break;

      case SHUTTR:
	print_sensor(SHUTTER_FILE, "%8.4f %3d %3d %3d %2d %2d %2d %2d %04x\n",
		     fdate,
		     getIntelword((Byte *)&msgp->msg.shutter.sh_attempts),
		     msgp->msg.shutter.sh_opens, msgp->msg.shutter.sh_closes,
		     msgp->msg.shutter.sh_toterrs, msgp->msg.shutter.sh_errs[0],
		     msgp->msg.shutter.sh_errs[1], msgp->msg.shutter.sh_errs[2],
		     getIntelword((Byte *)&msgp->msg.shutter.sh_oasisErrs));

	print_sensor( SAT1_DARK_FILE, "%8.4f ", fdate );

	for ( i = 0; i < ASPEC_CHANS; i++ )
	{
	    val = getIntelword( (Byte *)&msgp->msg.shutter.sh_satDark[i] );
#if SPEC_VOLTS
	    lu = decode_prr_chan( val, NULL );
#else
	    lu = decode_prr_chan( val,
			 &spec_cal.spc_cal[SPEC_0M_LU][SPEC_1230_LU_CHAN+i] );
#endif
	    print_sensor( SAT1_DARK_FILE, "%9.5f ", lu );
	}
	print_sensor( SAT1_DARK_FILE, "\n" );

	if ( rev == 4 )
	{
	    val  = getIntelword( (Byte *)&msgp->msg.shutter.sh_ctd_temp );
	    misc = getIntelword( (Byte *)&msgp->msg.shutter.sh_ctd_cond );
	    print_sensor( TEMP0_FILE, "%8.4f  %7.3f %8.4f\n", fdate,
			 ((double)val) / 1000., ((double)misc) / 10000.);
	}

	break;

      case MISC:
	if ( rev == 3 )
	    printMiscRev3( fdate, &msgp->msg.misc3 );
	else
	    printMiscRev4( fdate, &msgp->msg.misc4 );
	break;

      default:
	print_error( "Bad Message Type %d\n", msgType );
    }

    print_sensor( ARGOS_FILE, "%8.4f  %2d  %3d\n", 
		  fdate, msgType, msgp->msg_cnt );

} /* printOneMsgRev3_4() */

/************************************************************************/
/* Function    : printOneMsgR5					*/
/* Purpose     : Print one received ARGOS message in Rev 5 format	*/
/* Inputs      : Message Pointer					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printOneMsgR5( ArgosInMsg *msgp, Int32 rev )
{
    Nat32	msgType;
    Nat16	misc, val;
    Reg Nat32	i;
    Flt64	fdate, lu;
    
    if ( (rev != 5) )
    {
	print_error("Internal error - wrong message revision\n");
	return;
    }

    msgType = (msgp->msg.rawb.by_data[1] >> 4) & 0x0f;
    if ( !checkCRC(&msgp->msg.raww) )
	return;

    if ( !decode_all && !decode_msg[msgType] )
	return;

    fdate = fileDate( msgp->first_msg );

    mType=msgType;
    switch( msgType )
    {
      case CHEM:
	    printChemR5( fdate, &msgp->msg.chem5 );
	break;

      case SPEC_0M_NOON:
	misc = getIntelword( (Byte *)&msgp->msg.spec0m.sp_misc );
	printSpecR5( SPEC_NOON_FILE, fdate, &msgp->msg.spec0m,
		   SPEC_0M_ED, SPEC_0M_LU, 0, SPEC_NOON_LU_CHAN );
	print_sensor( SPEC_NOON_FILE, "%u\n", misc );
	break;

      case SPEC_0M_1000:
	misc = getIntelword( (Byte *)&msgp->msg.spec0m.sp_misc );
	printSpecR5( SPEC_0M_1000_FILE, fdate, &msgp->msg.spec0m_1000,
		   SPEC_0M_ED, SPEC_0M_LU, 0,
		   (SPEC_1000_LU_CHAN_REV5) );
	print_sensor( SPEC_0M_1000_FILE, "%u\n", misc );
	break;

      case SPEC_20M:
	printSpecR5( SPEC_20_FILE, fdate, &msgp->msg.spec20m,
		   SPEC_20M_ED, SPEC_20M_LU, 0, 0 );
	print_sensor( SPEC_20_FILE, "\n" );
	break;

      case SPEC_20M_1000:
	printSpecR5( SPEC_20M_1000_FILE, fdate, &msgp->msg.spec20m,
		   SPEC_20M_ED, SPEC_20M_LU, 0, 0 );
	print_sensor( SPEC_20M_1000_FILE, "\n" );
	break;

      case SPEC_0M_DARK:
	printSpecR5( SPEC_0M_DARK_FILE, fdate, &msgp->msg.spec0m,
		   SPEC_0M_ED, SPEC_0M_LU, 0, SPEC_NOON_LU_CHAN );
	print_sensor( SPEC_0M_DARK_FILE, "\n" );
	break;

      case SPEC_20M_DARK:
	printSpecR5( SPEC_20M_DARK_FILE, fdate, &msgp->msg.spec20m,
		   SPEC_20M_ED, SPEC_20M_LU, 0, 0 );
	print_sensor( SPEC_20M_DARK_FILE, "\n" );
	break;

      case SAT1:
	printSpecR5( SAT1_FILE, fdate, &msgp->msg.sat1, 
		   SAT1_ED, SAT1_LU, SAT1_ED_CHAN, SAT1_LU_CHAN );
	print_sensor( SAT1_FILE, "\n" );
	break;

      case SHUTTR:
	print_sensor(SHUTTER_FILE, "%8.4f %3d %3d %3d %2d %2d %2d %2d %04x\n",
		     fdate,
		     getIntelword((Byte *)&msgp->msg.shutter.sh_attempts),
		     msgp->msg.shutter.sh_opens, msgp->msg.shutter.sh_closes,
		     msgp->msg.shutter.sh_toterrs, msgp->msg.shutter.sh_errs[0],
		     msgp->msg.shutter.sh_errs[1], msgp->msg.shutter.sh_errs[2],
		     getIntelword((Byte *)&msgp->msg.shutter.sh_oasisErrs));

	print_sensor( SAT1_DARK_FILE, "%8.4f ", fdate );

	for ( i = 0; i < ASPEC_CHANS; i++ )
	{
	    val = getIntelword( (Byte *)&msgp->msg.shutter.sh_satDark[i] );
#if SPEC_VOLTS
	    lu = decode_prr_chan( val, NULL );
#else
	    lu = decode_prr_chan( val,
			 &spec_cal.spc_cal[SPEC_0M_LU][SPEC_1000_LU_CHAN+i] );
#endif
	    print_sensor( SAT1_DARK_FILE, "%9.5f ", lu );
	}
	print_sensor( SAT1_DARK_FILE, "\n" );

	if ( rev == 5 )
	{
	    val  = getIntelword( (Byte *)&msgp->msg.shutter.sh_ctd_temp );
	    misc = getIntelword( (Byte *)&msgp->msg.shutter.sh_ctd_cond );
	    print_sensor( TEMP0_FILE, "%8.4f  %7.3f %8.4f\n", fdate,
			 ((double)val) / 1000., ((double)misc) / 10000.);
	}

	break;

      case MISC:
	    printMiscRev4( fdate, &msgp->msg.misc4 );
	break;

      default:
	print_error( "Bad Message Type %d\n", msgType );
    }

    print_sensor( ARGOS_FILE, "%8.4f  %2d  %3d\n", 
		  fdate, msgType, msgp->msg_cnt );

} /* printOneMsgR5() */




/************************************************************************/
/* Function    : printOneMsg						*/
/* Purpose     : Print one received ARGOS message			*/
/* Inputs      : Message Pointer					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printOneMsg( ArgosInMsg *msgp )
{
    switch( revision )
    {
      case 1:
      case 2:
	printOneMsgRev2( msgp );
	break;

      case 3:
	printOneMsgRev3_4( msgp, 3 );
	break;

      case 4:
	printOneMsgRev3_4( msgp, 4 );
	break;

      case 5:
	printOneMsgR5( msgp, 5 );
	break;

      default:
	printf( "Bad software revision number %d\n", revision );
    }

} /* printOneMsg() */


Void    exPCO2(ArgosMsg *pMsg, DataItem *pItem,Flt64 date){
  
  Nat32	i;     /*PCO2_10BIT, PCO2_12BIT*/
  Nat32 msbs;  /*PCO2_10BIT*/
  Int32	co2;   /*PCO2_10BIT, PCO2_12BIT, PCO2_CAL*/  
  Reg Byte *p; /*PCO2_12BIT*/
  Nat32 val;   /*PCO2_CAL*/

  switch(pItem->id){
  case PCO2_10BIT:
    switch(gRevision){
    case REV0:
    case REV1:
    case REV2:
    case REV3:
    case REV4:
      msbs = getIntelword((Byte *)&pMsg->data[pItem->offset+8] );/* ch_pco2_msbs */

      for ( i = 0; i < pItem->size; i++ ){
	co2 = pMsg->data[pItem->offset+i]  | (((msbs >> (i + i)) & 3) << 8);
	if ( co2 & 0x200 )
	  co2 |= ~0x3ff;				/* Sign extend	*/
	print_sensor( CO2_FILE, "%8.4f  %4d\n",  date + CHEM_TIME(i), co2 );
      }
      break;
    default:
      break;
    }/*switch(gRevision)*/
    break;
  case PCO2_12BIT:
    switch(gRevision){
    case REV0:
    case REV1:
    case REV2:
    case REV3:
    case REV4:
      p = pMsg->data[pItem->offset];

      for ( i = 0; i < pItem->size; i+=2 ){
	co2 = *p++ << 4;
	co2 |= ((*p >> 4) & 0x0f);
	if ( co2 & 0x800 )
	  co2 |= 0xfffff000;				/* Sign extend	*/
	print_sensor( CO2_FILE, "%8.4f  %4d\n",  date + CHEM_TIME(i), co2 );
	
	co2 = ((*p++ & 0x0f) << 8);
	co2 |= *p++;
	if ( co2 & 0x800 )
	  co2 |= 0xfffff000;				/* Sign extend	*/
	print_sensor( CO2_FILE, "%8.4f  %4d\n",  date + CHEM_TIME(i+1), co2 );
      }
    default:
      break;
    }/*switch(gRevision)*/
    break;
  case PCO2_CAL:
    switch(gRevision){
    case REV0:
    case REV1:
    case REV2:
    case REV3:
    case REV4:
    val = getIntelword( (Byte *)&pMsg->data[pItem->offset] );
    co2 = val & 0x3ff;
    if ( co2 & 0x200 )
	co2 |= ~0x3ff;					/* Sign extend	*/

    print_sensor( CO2_CAL_FILE, "%8.4f  %3d  %02d00\n", date, co2, ((val >> 10) & 7) * 3 );
    default:
      break;
    }/*switch(gRevision)*/
    break;
  default:
    break;
  }/*switch()*/

}/*exPCO2()*/

Void    exOSMO_NO3(ArgosMsg *pMsg, DataItem *pItem,Flt64 date){}/*exOSMO_NO3*/

Void    exPRR
Void    exCTD
Void    exISUS
Void    exSAT
Void    exWETSTAR
Void    exMCP
Void    exSHUTTER
Void    exHS2


/* dummy func for data items that don't need to be processed */
Void null_func(ArgosMsg *pMsg, Nat16 mapIndex){
  return;
}/*null_func*/

/*printMsg()*/
Void printMsg(ArgosInMsg *pMsg){
  Nat32	msgType;
  Flt64	fdate;
  DataItem *pDataItem;
  Nat16 item;
  
  if ( gREVISION == 1 ){
    msgType = pMsg->msg.rawb.bdata[0];
    if ( (msgType & 0x0f) != ((msgType >> 4) & 0x0f) )
      return;
    if ( !checkSum(&pMsg->msg.bmsg) )
      return;
    msgType = msgType & 0x0f;/*msgTypesRev1[msgType & 0x0f];*/
  }
  else{
    msgType = (pMsg->msg.bmsg.bdata[1] >> 4) & 0x0f;
    if ( !checkCRC(&pMsg->msg.wmsg) )
      return;
  }
  
  if ( !decode_all && !decode_msg[msgType] )
    return;
  
  fdate = fileDate( pMsg->first_msg );

  gMessageType=msgType;

  pDataItem=NULL;
  item=0;
  do{
    pDataItem=MsgItemMap[msgType][item];

    if(pDataItem==NULL)
      break;

    if( pDataItem != null_func )
      pDataItem->pExtract(pMsg->amsg,pDataItem);

    item++;
  }while(pDataItem!=NULL);
  
  return;

}/*printMsg()*/

/*createDataItem()*/
DataItem *creatDataItem(int id,Nat16 offset, Nat16 size, void (*exfunc(ArgosMsg *msg))){
  DataItem *pDI;

  if((pDI=(DataItem *)malloc(sizeof(DataItem)))==NULL)
    return NULL;

  pDI->id=id;
  pDI->offset=offset;
  pDI->size=size;
  pDI->pExtract=exfunc;

  return pDI;

}/*createDataItem()*/

/* initMessageItemMap()*/
void initMessageItemMap(){
  Nat16 msg_type,rev;
  for(msg_type=0;msg_type<MAX_MSG_TYPES;msg_type++){
    for(rev=FIRST_REVISION;rev<MAX_REVISION;rev++){
      switch(msg_type){

      case ARGOS0:
	switch(rev){
	case REV0:
	case REV1:
	case REV2:

	  MsgItemMap[msg_type][0]=createDataItem(PCO2_10BIT, 0,10,exPCO2);
	  MsgItemMap[msg_type][1]=createDataItem(OSMO_NO3,  10,10,exOSMO_NO3);
	  MsgItemMap[msg_type][2]=NULL;
	  break;
	case REV3:
	  MsgItemMap[msg_type][0]=createDataItem(PCO2_12BIT,           0,12,exPCO2);
	  MsgItemMap[msg_type][1]=createDataItem(0M_1200_PRR620_PITCH,12, 2,exPRR);
	  MsgItemMap[msg_type][2]=createDataItem(0M_1200_PRR620_ROLL, 14, 2,exPRR);
	  MsgItemMap[msg_type][3]=createDataItem(0M_1200_PRR600_TEMP, 16, 2,exPRR);
	  MsgItemMap[msg_type][4]=createDataItem(0M_1200_PRR600_DEPTH,18, 2,exPRR);
	  MsgItemMap[msg_type][5]=createDataItem(0M_1230_PRR620_PITCH,20, 2,exPRR);
	  MsgItemMap[msg_type][6]=createDataItem(0M_1230_PRR620_ROLL, 22, 1,exPRR);
	  MsgItemMap[msg_type][7]=createDataItem(0M_1230_PRR600_TEMP, 24, 2,exPRR);
	  MsgItemMap[msg_type][8]=createDataItem(0M_1230_PRR600_DEPTH,26, 2,exPRR);
	  MsgItemMap[msg_type][9]=NULL;
	  break;
	case REV4:
	  MsgItemMap[msg_type][0]=createDataItem(PCO2_12BIT,           0,12,exPCO2);
	  MsgItemMap[msg_type][1]=createDataItem(PCO2_CAL,            12, 2,exPCO2);
	  MsgItemMap[msg_type][2]=createDataItem(0M_1200_PRR600_TEMP, 14, 2,exPRR);
	  MsgItemMap[msg_type][3]=createDataItem(0M_1200_PRR600_DEPTH,16, 2,exPRR);
	  MsgItemMap[msg_type][4]=createDataItem(0M_1200_CTD_COND,    18, 2,exCTD);
	  MsgItemMap[msg_type][5]=createDataItem(0M_1200_CTD_TEMP,    20, 2,exCTD);
	  MsgItemMap[msg_type][6]=createDataItem(0M_ISUS_REC0_NO3,    22, 7,exISUS);
	  MsgItemMap[msg_type][7]=NULL;
	  break;
	default:
	  break;
	}/* switch(rev)*/
	break;/* ARGOS0 */

      case ARGOS1:
	switch(rev){
	case REV0:
	case REV1:
	case REV2:
	case REV3:
	case REV4:
	  MsgItemMap[msg_type][0]=createDataItem(0M_1200_PRR620_ED,0,14,exPRR);
	  MsgItemMap[msg_type][1]=createDataItem(0M_1200_SATA_LU, 14,14,exSAT);
	  MsgItemMap[msg_type][2]=createDataItem(0M_PRR_ERRCNT0,  28, 2,exPRR);
	  MsgItemMap[msg_type][3]=NO_ITEM;
	default:
	  break;
	}/* switch(rev)*/
	break;/* ARGOS1 */

      case ARGOS2:
	switch(rev){
	case REV0:
	case REV1:
	case REV2:
	case REV3:
	  MsgItemMap[msg_type][0]=createDataItem(0M_1230_PRR620_ED,0,14,exPRR);
	  MsgItemMap[msg_type][1]=createDataItem(0M_1230_SATA_LU, 14,14,exSAT);
	  MsgItemMap[msg_type][2]=createDataItem(0M_PRR_ERRCNT1,  28, 2,exPRR);
	  MsgItemMap[msg_type][3]=NULL;
	  break;
	case REV4:
	  MsgItemMap[msg_type][0]=createDataItem(0M_1000_PRR620_ED, 0,14,exPRR);
	  MsgItemMap[msg_type][1]=createDataItem(0M_ISUS_REC1_NO3, 14,14,exISUS);
	  MsgItemMap[msg_type][2]=createDataItem(0M_PRR_ERRCNT1,   28, 2,exPRR);
	  MsgItemMap[msg_type][3]=NULL;
	default:
	  break;
	}/* switch(rev)*/
	break;/* ARGOS2 */

      case ARGOS3:
	switch(rev){
	case REV0:
	case REV1:
	  MsgItemMap[msg_type][0]=createDataItem(PCO2_CAL,        0, 2,exPCO2);
	  MsgItemMap[msg_type][1]=createDataItem(OASIS_DAT,       2, 4,exOASIS);
	  MsgItemMap[msg_type][2]=createDataItem(10M_1200_MCP_ED, 6, 2,exMCP);
	  MsgItemMap[msg_type][3]=createDataItem(30M_1200_MCP_ED, 8, 2,exMCP);
	  MsgItemMap[msg_type][4]=createDataItem(20M_OSMO_NO,     3,10,exOSMO_NO3);
	  MsgItemMap[msg_type][5]=NULL;
	  break;
	case REV2:
	  MsgItemMap[msg_type][0]=createDataItem(PCO2_CAL,          0, 2,exPCO2);
	  MsgItemMap[msg_type][1]=createDataItem(OASIS_DAT,         2, 4,exOASIS);
	  MsgItemMap[msg_type][2]=createDataItem(10M_1200_MCP_ED,   6, 2,exMCP);
	  MsgItemMap[msg_type][3]=createDataItem(30M_1200_MCP_ED,   8, 2,exMCP);
	  MsgItemMap[msg_type][4]=createDataItem(0M_WETSTAR_FLUOR, 10, 8,exWETSTAR);
	  MsgItemMap[msg_type][5]=createDataItem(20M_WETSTAR_FLUOR,18,12,exWETSTAR);
	  MsgItemMap[msg_type][6]=NULL;
	  break;
	case REV3:
	  MsgItemMap[msg_type][0]=createDataItem(PCO2_CAL,          0, 2,exPCO2);
	  MsgItemMap[msg_type][1]=createDataItem(OASIS_DAT,         2, 4,exOASIS);
	  MsgItemMap[msg_type][2]=createDataItem(0M_WETSTAR_FLUOR,  6,12,exWETSTAR);
	  MsgItemMap[msg_type][3]=createDataItem(20M_WETSTAR_FLUOR,18,12,exWETSTAR);
	  MsgItemMap[msg_type][4]=NULL;
	  break;
	case REV4:
	  MsgItemMap[msg_type][0]=createDataItem(OASIS_DAT,         0, 4,exOASIS);
	  MsgItemMap[msg_type][1]=createDataItem(0M_WETSTAR_FLUOR,  4,12,exWETSTAR);
	  MsgItemMap[msg_type][2]=createDataItem(20M_WETSTAR_FLUOR,16,12,exWETSTAR);
	  MsgItemMap[msg_type][3]=NULL;
	  break;
	default:
	  break;
	}/* switch(rev)*/
	break;/* ARGOS3 */

      case ARGOS4:
	switch(rev){
	case REV0:
	case REV1:
	case REV2:
	case REV3:
	  MsgItemMap[msg_type][0]=createDataItem(20M_1200_PRR600_ED, 0,14,exPRR);
	  MsgItemMap[msg_type][1]=createDataItem(20M_1200_PRR600_LU,14,14,exPRR);
	  MsgItemMap[msg_type][2]=createDataItem(10M_1200_MCP_ED,   28, 2,exMCP);
	  MsgItemMap[msg_type][3]=NULL;
	  break;
	case REV4:
	  MsgItemMap[msg_type][0]=createDataItem(20M_1200_PRR600_ED, 0,14,exPRR);
	  MsgItemMap[msg_type][1]=createDataItem(20M_1200_PRR600_LU,14,14,exPRR);
	  MsgItemMap[msg_type][2]=NULL;
	  break;
	default:
	  break;
	}/* switch(rev)*/
	break;/* ARGOS4 */

      case ARGOS5:
	switch(rev){
	case REV0:
	case REV1:
	case REV2:
	case REV3:
	  MsgItemMap[msg_type][0]=createDataItem(20M_1230_PRR600_ED, 0,14,exPRR);
	  MsgItemMap[msg_type][1]=createDataItem(20M_1230_PRR600_LU,14,14,exPRR);
	  MsgItemMap[msg_type][2]=createDataItem(30M_1200_MCP_ED    28, 2,exMCP);
	  MsgItemMap[msg_type][3]=NULL;
	  break;
	case REV4:
	  MsgItemMap[msg_type][0]=createDataItem(20M_1000_PRR600_ED, 0,14,exPRR);
	  MsgItemMap[msg_type][1]=createDataItem(20M_1000_PRR600_LU,14,14,exPRR);
	  MsgItemMap[msg_type][2]=NULL;
	  break;
	default:
	  break;
	}/* switch(rev)*/
	break;/* ARGOS5 */

      case ARGOS6:
	switch(rev){
	case REV0:
	case REV1:
	case REV2:
	case REV3:
	  MsgItemMap[msg_type][0]=createDataItem(0M_SHUTTER_ATTEMPTS, 0, 2,exSHUTTER);
	  MsgItemMap[msg_type][1]=createDataItem(0M_SHUTTER_OPENS,    2, 1,exSHUTTER);
	  MsgItemMap[msg_type][2]=createDataItem(0M_SHUTTER_CLOSES,   3, 1,exSHUTTER);
	  MsgItemMap[msg_type][3]=createDataItem(0M_SHUTTER_TOTALERRS,4, 1,exSHUTTER);
	  MsgItemMap[msg_type][4]=createDataItem(0M_SHUTTER_ERRTYPES, 5, 1,exSHUTTER);
	  MsgItemMap[msg_type][5]=createDataItem(OASIS_ERRS,          6, 2,exOASIS);
	  MsgItemMap[msg_type][6]=createDataItem(0M_0000_SATB_LU,     8,14,exSAT);
	  MsgItemMap[msg_type][7]=createDataItem(0M_1200_CTD_COND,   22, 2,exCTD);
	  MsgItemMap[msg_type][8]=createDataItem(0M_1200_CTD_TEMP,   24, 2,exCTD);
	  MsgItemMap[msg_type][9]=NULL;
	  break;
	case REV4:
	  MsgItemMap[msg_type][0]=createDataItem(0M_SHUTTER_ATTEMPTS, 0,2,exSHUTTER);
	  MsgItemMap[msg_type][1]=createDataItem(0M_SHUTTER_OPENS,    2,1,exSHUTTER);
	  MsgItemMap[msg_type][2]=createDataItem(0M_SHUTTER_CLOSES,   3,1,exSHUTTER);
	  MsgItemMap[msg_type][3]=createDataItem(0M_SHUTTER_TOTALERRS,4,1,exSHUTTER);
	  MsgItemMap[msg_type][4]=createDataItem(0M_SHUTTER_ERRTYPES, 5,1,exSHUTTER);
	  MsgItemMap[msg_type][5]=createDataItem(OASIS_ERRS,          6,2,exOASIS);
	  MsgItemMap[msg_type][6]=createDataItem(0M_0000_SATB_LU,     8,14,exSAT);
	  MsgItemMap[msg_type][7]=NULL;
	  break;
	default:
	  break;
	}/* switch(rev)*/
	break;/* ARGOS6 */

      case ARGOS7:
	switch(rev){
	case REV0:
	case REV1:
	case REV2:
	case REV3:
	case REV4:
	  MsgItemMap[msg_type][0]=createDataItem(0M_0000_PRR620_ED,0,14,exPRR);
	  MsgItemMap[msg_type][1]=createDataItem(0M_0000_SATA_LU, 14,14,exSAT);
	  MsgItemMap[msg_type][2]=NULL;
	  break;
	default:
	  break;
	}/* switch(rev)*/
	break;/* ARGOS7 */

      case ARGOS8:
	switch(rev){
	case REV0:
	case REV1:
	case REV2:
	case REV3:
	case REV4:
	  MsgItemMap[msg_type][0]=createDataItem(20M_0000_PRR600_ED, 0,14,exPRR);
	  MsgItemMap[msg_type][1]=createDataItem(20M_0000_PRR600_LU,14,14,exPRR);
	  MsgItemMap[msg_type][2]=NULL;
	  break;
	default:
	  break;
	}/* switch(rev)*/
	break;/* ARGOS8 */

      case ARGOS9:
	switch(rev){
	case REV0:
	case REV1:
	case REV2:
	case REV3:
	  MsgItemMap[msg_type][0]=createDataItem(0M_1200_SATB_LU, 0,14,exSAT);
	  MsgItemMap[msg_type][1]=createDataItem(0M_1230_SATB_LU,14,14,exSAT);
	  MsgItemMap[msg_type][2]=NULL;
	  break;
	case REV4:
	  MsgItemMap[msg_type][0]=createDataItem(0M_1200_SATB_LU,0,14,exSAT);
	  MsgItemMap[msg_type][1]=createDataItem(0M_HS2_SCATTER,14,10,exHS2);
	  MsgItemMap[msg_type][2]=NO_ITEM;
	  break;
	default:
	  break;
	}/* switch(rev)*/
	break;/* ARGOS9 */

      default:
	break;
      }/*switch (msg_type)*/
    }
  }
}
/* initMessageItemMap()*/

/************************************************************************/
/* Function    : decodeFile						*/
/* Purpose     : Decode one ARGOS message file				*/
/* Inputs      : File name						*/
/* Outputs     : None							*/
/************************************************************************/
	Void
decodeFile( char *filename )
{
    Reg Nat32	i;
    Reg FILE	*fp;

    if ( (fp = fopen(filename, "rb")) == (FILE *)NULL )
	fprintf(stderr, "Cannot open %s\n", filename);
    else
    {
	getMessages( fp, buffer );

	for ( i = 0; i < msgCnt; i++ )
	    printOneMsg( &msgs[i] );
    }

    close_sensor_files();

} /* printMessages() */
