/****************************************************************************/
/* Copyright 1999 MBARI                                                     */
/****************************************************************************/
/* $Header: tstring.c,v 3.1 2001/05/04 09:11:27 oasisa Exp $		    */
/* Summary  : Seacat tstring decode routines for decode.c, extract.c	    */
/* Filename : tstring.c							    */
/* Authors  : Todd Anderson, Robert Herlien (rah)			    */
/* Project  : OASIS Mooring						    */
/* $Revision: 3.1 $							    */
/* Created  : 01/21/99							    */
/****************************************************************************/
/* Modification History:						    */
/* 21jan99 rah - created from mods to extract done by Todd Anderson	    */
/* $Log:	tstring.c,v $
 * Revision 3.1  2001/05/04  09:11:27  09:11:27  oasisa (Oasis users)
 * better parsing to filter garbage characters
 * 
 * Revision 3.0  99/05/12  10:11:33  10:11:33  bobh (Bob Herlien)
 * Added tstring, misc changes
 * 
*/
/****************************************************************************/

#include <stdio.h>			/* Standard I/O			    */
#include <mbariTypes.h>			/* MBARI type definitions	    */
#include <mbariConst.h>			/* MBARI constants		    */
#include <decode.h>			/* OASIS controller definitions	    */
#include <time.h>			/* Time				    */
#include <string.h>			/* for strcmp()			    */


/********************************/
/*	External Functions	*/
/********************************/

Extern Void	print_sensor( Int sensor, char *fmt, ... );
Extern Void	print_error( char *fmt, ... );
Extern double	ctd_salinity( double c, double t, double p );


/************************/
/*  External Data	*/
/************************/

Extern double	dtime;			/* Time in double precision format  */


/********************************/
/*	Module Local Data	*/
/********************************/

MLocal char	*month_names[] = 
{ "jan", "feb", "mar", "apr", "may", "jun", "jul", "aug", "sep", "oct",
  "nov", "dec"
};

MLocal Nat16    jdays[12] =             /* Days since Jan. 1 in non-leap yr */
{   0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334
};


/************************************************************************/
/* Function    : read_tstring_cfg	   				*/
/* Purpose     : Read TSTRING configuration file			*/
/* Inputs      : Ptr to tstring cfg file name ptr			*/
/* Outputs     : OK or ERROR						*/
/************************************************************************/
	Status
read_tstring_cfg( char *fileName, TStringCfg *tCfgp )
{
    Reg FILE		*fp;
    Reg Nat32		calNum;
    char		pflag;
    char        	linebuf[256];

    bzero( tCfgp, sizeof(TStringCfg) );

    if ( (fp = fopen(fileName, "rb")) == (FILE *)NULL )
	return( ERROR );

    for ( calNum = 0; (calNum < MAX_TSTRING) && 
	  (fgets(linebuf, sizeof(linebuf), fp) != NULL); )
    {
	if ( sscanf(linebuf, " %d %d %c", &tCfgp->pod[calNum].ser_no, 
		    &tCfgp->pod[calNum].depth, &pflag) >= 3 )
	{
	    tCfgp->pod[calNum].press_flag = (toupper(pflag) == 'Y') ? 1 : 0;
	    calNum++;
	}
    }

    tCfgp->numValidPods = calNum;
    fclose( fp );
    return( OK );

} /* read_tstring_cfg() */


/************************************************************************/
/* Function    : getMonth						*/
/* Purpose     : Get month (1-12) from 3 character month name		*/
/* Inputs      : 3 character name of month				*/
/* Outputs     : 0 - 11 for valid month, or ERROR			*/
/************************************************************************/
	Int32
getMonth( char *mon )
{
    Reg Int32	month;

    for ( month = 0; month < 12; month++ )
	if ( strncasecmp(month_names[month], mon, 3) == 0 )
	    return( month );

    return( ERROR );

} /* getMonth() */


/************************************************************************/
/* Function    : julday                                                 */
/* Purpose     : Calculate Julian Day given year, month, day            */
/* Inputs      : year (0000 - 9999), month (jan - dec), day (1 - 31),   */
/*               hour(0 - 23), min(0 - 59), sec(0 - 59)                 */
/* Outputs     : Julian day and fraction of day                         */
/* Comment     : Returns Julian day in range 1.000:366.000                      */
/************************************************************************/
        double
julday( Int yr, char *mon, Int day, Int hour, Int min, Int sec)
{
    Reg Int32	month, leap, jday;

    if ( (month = getMonth(mon)) == ERROR )
	return( 0.0 );

    leap = (((yr % 4) == 0) && (month > 1)) ? 1 : 0;

    return( (double)jdays[month] + (double)(day + leap) +
	    (((double)sec/60.0 + (double)min)/60.0 + (double)hour)/24.0 );

} /* julday() */


/************************************************************************/
/* Function    : findPod						*/
/* Purpose     : Find a pod in cfg array with matching serial number	*/
/* Inputs      : Serial number, cfg ptr					*/
/* Outputs     : Array index (0..MAX_TSTRING-1) or ERROR		*/
/************************************************************************/
	Int32
findPod( Int32 serNum, TStringCfg *tCfgp )
{
    Reg Int32	podNum;

    for( podNum = 0; podNum < MAX_TSTRING; podNum++)
	if ( tCfgp->pod[podNum].ser_no == serNum )
	    return( podNum );

    return( ERROR );

} /* findPod() */


/************************************************************************/
/* Function    : print_tstring						*/
/* Purpose     : Print data from Inductive Modem Temperature String	*/
/* Inputs      : Sensor, Length						*/
/* Outputs     : None							*/
/************************************************************************/
	Void
print_tstring( Int sensor, Byte *buffer, Int len, TStringCfg *tCfgp )
{
    Reg char	*datap, *linep;
    Reg Int32	cnt, i;
    Tstring     tstring_read, tstring[MAX_TSTRING];
    Int 	ser_no, day, year, hr, min, sec,recnum;
    char	month[8];
    double      salinity;

    
    bzero( (char *)tstring, sizeof(tstring) );
    for ( i = 0; i < MAX_TSTRING; i++ )
    {
	tstring[i].temp =  tstring[i].cond = tstring[i].press = 0.0;
	tstring[i].jdate = 0.0;
	tstring[i].recnum = 0;
    }

    for ( datap = buffer, cnt = 0; *datap && (cnt < len); )
    {
	for ( linep = datap; *datap && (cnt < len); datap++, cnt++ )
	    if ( (*datap == '\r') || (*datap == '\n') )
	    {
		*datap++ = '\0';
		cnt++;
		break;
	    }else if(*datap<' ' || *datap > '~'){
	      /*printf("(%x)",*datap);*/
	      *datap++ = '\0';
	      cnt++;
	      break;
	    }

	bzero( (char *)&tstring_read, sizeof(tstring_read) );

	if(sscanf(linep, " %d, %lf, %lf, %d %s %d, %d:%d:%d,%d",
		  &ser_no, &tstring_read.temp, &tstring_read.cond, 
		  &day, month, &year, &hr, &min, &sec,&tstring_read.recnum ) == 10)
	{
	    tstring_read.valid = 1;
	    tstring_read.press = 0.0;
	}
	else if(sscanf(linep, " %d, %lf, %lf, %lf, %d %s %d, %d:%d:%d,%d", 
		       &ser_no, &tstring_read.temp, 
		       &tstring_read.cond, &tstring_read.press, 
		       &day, month, &year, &hr, &min, &sec,&tstring_read.recnum) == 11)
	{ 
	    tstring_read.valid = 1;
	}else if(sscanf(linep, " %d, %lf, %lf, %d %s %d, %d:%d:%d",
		  &ser_no, &tstring_read.temp, &tstring_read.cond, 
		  &day, month, &year, &hr, &min, &sec) == 9)
	{
	    tstring_read.valid = 1;
	    tstring_read.press = 0.0;
	}
	else if(sscanf(linep, " %d, %lf, %lf, %lf, %d %s %d, %d:%d:%d", 
		       &ser_no, &tstring_read.temp, 
		       &tstring_read.cond, &tstring_read.press, 
		       &day, month, &year, &hr, &min, &sec) == 10)
	{ 
	    tstring_read.valid = 1;
	}

	if ( (tstring_read.valid) && ((i = findPod(ser_no, tCfgp)) != ERROR) )
	{
	    tstring[i] = tstring_read;
	    tstring[i].jdate = julday(year, month, day, hr, min, sec);
	}
    }

    print_sensor( sensor, "%9.5f", dtime );

    for(i = 0; i < tCfgp->numValidPods; i++ )
    {
	if ( tstring[i].valid )
	{
	    if ( tCfgp->pod[i].press_flag )
		salinity = ctd_salinity(tstring[i].cond, tstring[i].temp,
					tstring[i].press); 
	    else
		salinity = ctd_salinity(tstring[i].cond, tstring[i].temp,
					(double)tCfgp->pod[i].depth); 

	    print_sensor(sensor, " %3d %3d %8.4f %9.5f %7.4f",
			 tCfgp->pod[i].ser_no, tCfgp->pod[i].depth, 
			 tstring[i].temp, tstring[i].cond, salinity);
	}
	else
	    print_sensor(sensor, " %3d %3d %8.4f %9.5f %7.4f",
			 tCfgp->pod[i].ser_no, tCfgp->pod[i].depth,
			 0.0, 0.0, 0.0);

	if ( tCfgp->pod[i].press_flag )
		print_sensor(sensor, " %8.3f", tstring[i].press);

	print_sensor(sensor, " %9.5f", tstring[i].jdate);

	if(tstring[i].recnum > 0)
	  print_sensor(sensor, " %7d", tstring[i].recnum);
	else
	  print_sensor(sensor, " %07d", tstring[i].recnum);

    }

    print_sensor( sensor, "\n" );

} /* print_tstring() */
