/************************************************************************/
/* Copyright 1996 MBARI							*/
/************************************************************************/
/* Summary  : Program to analyze ARGOS receiver data			*/
/* Filename : argos.c							*/
/* Author   : Bob Herlien (rah)						*/
/* Project  : OASIS							*/
/* Version  : 1.0							*/
/* Created  : 07/15/96							*/
/************************************************************************/
/* Modification History:						*/
/* 15jul96 rah - created						*/
/************************************************************************/

#define ARGOS_REV	2
#include <stdio.h>			/* Unix standard I/O		*/
#include <mbari/types.h>		/* MBARI standard types		*/
#include <mbari/const.h>		/* MBARI standard constants	*/
#include <time.h>			/* Unix time library		*/
#include <decode.h>			/* OASIS data definitions	*/
#include <string.h>			/* Unix string library		*/
#include <argos.h>			/* ARGOS buffer definitions	*/

#define NUM_MSGS	1000		/* Max messages to store & compare */
#define FLUOR_CHAN	0		/* Analog channel for Wetstar fluor*/
#define TEMP_CHAN	2		/* Analog channel for temperature  */
#define OBATT_CHAN	3		/* Analog channel for OASIS battery*/
#define ABATT_CHAN	6		/* Analog channel for ARGOS battery*/

typedef enum				/********************************/
{					/* Return value from getXXXMsg	*/
    MSG_OK,				/* Successful conversion	*/
    BAD_MSG,				/* Bad message data		*/
    END_OF_FILE				/* End of input file		*/
} MsgRtn;				/********************************/

typedef enum				/********************************/
{					/* Type of input		*/
    SVC_ARGOS,				/* From Serivce Argos		*/
    RCVR_NORM,				/* From local receiver, normal mode*/
    RCVR_DIAG				/* Local receiver, diagnostic mode*/
} InputMode;				/********************************/


/********************************/
/*	External Functions	*/
/********************************/

Extern char	*get_can_name( char *dataname );
Extern Status	read_cfg( char **cfgname );
Extern double	decode_prr_chan( Int16 ival, SpecChanCal *sccp );
Extern Flt64	decode_ac9_temp( Int32 count );


/********************************/
/*	External Data		*/
/********************************/

Extern Analog	analog[ANALOG_CHANS];	/* Analog calibration		    */
Extern SpecCal  spec_cal[];		/* Structs to hold Spect calibrations*/


/********************************/
/*	Global Data		*/
/********************************/

Global Int	itime;			/* Just to satisfy file.c, not used */


/********************************/
/*	Module Local Data	*/
/********************************/

MLocal InputMode	inputMode = SVC_ARGOS;
MLocal MBool		printChksumErrors = FALSE;
MLocal MBool		smallConfig = FALSE;
MLocal char		*cfgp = NULL;	/* Ptr to name of OASIS config file */

MLocal ArgosInMsg	msgs[NUM_MSGS];
MLocal Nat32		numMsgs = 0;
MLocal Nat32		goodMsgs = 0;
MLocal Nat32		badMsgs = 0;
MLocal char		*msgNames[] =
{ "Chemical", "Noon Surface Spectro", "1030 Surface Spectro",
  "Misc", "Noon 20m Spectro", "AC-9 Part 1", "AC-9 Part 2" };

MLocal char		*ac9ChanNames[] =
{ "a650", "a676", "a715", "c510", "c555", "c630", "a412", "a440", "a488",
  "c650", "c676", "c715", "a510", "a555", "a630", "c412", "c440", "c488" };


/********************************/
/*	Forward Declarations	*/
/********************************/

MBool	processCommandLine ( Int argc, char **argv );
Void	getMessages( Void );
Void	printMessages( Void );


/************************************************************************/
/* Function    : main							*/
/* Purpose     : Main routine						*/
/* Inputs      : argc, argv						*/
/* Outputs     : Integer, 0 for success, 1 for failure			*/
/************************************************************************/
	Int32
main( Int32 argc, char **argv )
{
    char	*cp;

    if ( !processCommandLine(argc, argv) )
	exit( 1 );

    get_can_name( "argos" );
    cp = cfgp;

    if ( read_cfg(&cp) != OK )
	    printf("Can't read configuration file \"%s\"\n", cp);
    else
    {
	getMessages();
	printMessages();
    }

} /* main() */


/************************************************************************/
/* Function : use_msg							*/
/* Purpose  : Print Usage Message					*/
/* Inputs   : Name of program						*/
/* Outputs  : None							*/
/************************************************************************/
	Void
use_msg( char *s )
{
    fprintf( stderr, "Usage: %s [-b] [-c cfg_file] [-d] [-n]\n" );
    fprintf( stderr, "-b prints messages with checksum errors\n" );
    fprintf( stderr, "-d is for a local receiver in diagnostic mode\n" );
    fprintf( stderr, "-n is for a local receiver in normal mode\n" );
    fprintf( stderr, "default is for data from Service Argos\n");

} /* use_msg() */


/************************************************************************/
/* Function : processCommandLine					*/
/* Purpose  : Read the arguments from the command line			*/
/* Inputs   : argc, argv from main() routine				*/
/* Outputs  : TRUE if arguments OK, else FALSE				*/
/************************************************************************/
	MBool
processCommandLine ( Int argc, char **argv )
{
    Int		i;

    while ( (i = getopt(argc, argv, "bc:dns")) != EOF )
	switch( i )
	{
	  case 'b':
	    printChksumErrors = TRUE;
	    break;

	  case 'c':
	    cfgp = optarg;
	    break;

	  case 'd':
	    inputMode = RCVR_DIAG;
	    break;

	  case 'n':
	    inputMode = RCVR_NORM;
	    break;

	  case 's':
	    smallConfig = TRUE;
	    break;

	  default:
	    use_msg( argv[0] );
	    return( FALSE );
	}

    return( TRUE );

} /* processCommandLine() */


/************************************************************************/
/* Function    : getLine						*/
/* Purpose     : Get one line of ARGOS message				*/
/* Inputs      : Buffer ptr						*/
/* Outputs     : MSG_OK or END_OF_FILE					*/
/************************************************************************/
	MsgRtn
getLine( char *buffer )
{
    buffer[0] = '\0';

    while( TRUE )
    {
	if ( gets(buffer) == NULL )
	    return( END_OF_FILE );
	if ( strlen(buffer) > 0 )
	    return( MSG_OK );
    }

} /* getLine() */


/************************************************************************/
/* Function    : getSvcArgosMsg						*/
/* Purpose     : Get an ARGOS message in Service Argos format		*/
/* Inputs      : Buffer ptr, message ptr, time ptr, number of msgs	*/
/* Outputs     : Message return code					*/
/************************************************************************/
	MsgRtn
getSvcArgosMsg( char *buffer, WordMsg *msgp, struct tm *tmPtr, Int32 *nmsgs )
{
    Nat32		i, j, dat[4];

    if ( getLine(buffer) == END_OF_FILE )
	return( END_OF_FILE );

    if ( sscanf(buffer, " %d-%d-%d %d:%d:%d %d %x %x %x %x",
		&tmPtr->tm_year, &tmPtr->tm_mon, &tmPtr->tm_mday,
		&tmPtr->tm_hour, &tmPtr->tm_min, &tmPtr->tm_sec,
		nmsgs, &dat[0], &dat[1], &dat[2], &dat[3]) < 11 )
	return( BAD_MSG );

    tmPtr->tm_year -= 1900;
    tmPtr->tm_mon -= 1;

    for ( i = 0; i < 4; i++ )
	msgp->wd_data[i] = (Nat16)dat[i];

    for ( i = 4; i < ARGOS_WORDS; i += 4 )
    {
	if ( getLine(buffer) == END_OF_FILE )
	    return( END_OF_FILE );

	if ( sscanf(buffer, " %x %x %x %x", 
		    &dat[0], &dat[1], &dat[2], &dat[3]) < 4 )
	    return( BAD_MSG );

	for ( j = 0; j < 4; j++ )
	    msgp->wd_data[i + j] = (Nat16)dat[j];
    }

    return( MSG_OK );

} /* getSvcArgosMsg() */


/************************************************************************/
/* Function    : getNormalMsg						*/
/* Purpose     : Get an ARGOS message in normal format for local rcvr	*/
/* Inputs      : Buffer ptr, message ptr for return message		*/
/* Outputs     : Message return code					*/
/************************************************************************/
	MsgRtn
getNormalMsg( char *buffer, ByteMsg *msgp, struct tm *tmPtr )
{
    Nat32		i, j, dat[8];

    if ( getLine(buffer) == END_OF_FILE )
	return( END_OF_FILE );

    if ( sscanf(buffer, " %d-%d-%d %d:%d:%d", &tmPtr->tm_mon,
		    &tmPtr->tm_mday, &tmPtr->tm_year, &tmPtr->tm_hour,
		    &tmPtr->tm_min, &tmPtr->tm_sec) < 6 )
	return( BAD_MSG );

    tmPtr->tm_mon -= 1;

    for ( i = 0; i < ARGOS_LEN; i += 8 )
    {
	if ( getLine(buffer) == END_OF_FILE )
	    return( END_OF_FILE );

	if ( sscanf(buffer, " %d %d %d %d %d %d %d %d",
		    &dat[0], &dat[1], &dat[2], &dat[3],
		    &dat[4], &dat[5], &dat[6], &dat[7]) < 8 )
	    return( BAD_MSG );

	for ( j = 0; j < 8; j++ )
	    msgp->by_data[i + j] = (Byte)dat[j];
    }

    return( MSG_OK );

} /* getNormalMsg() */


/************************************************************************/
/* Function    : getDiagnosticMsg					*/
/* Purpose     : Get an ARGOS message in diagnostic format for local rcvr*/
/* Inputs      : Buffer ptr, message ptr for return message		*/
/* Outputs     : Message return code					*/
/************************************************************************/
	MsgRtn
getDiagnosticMsg( char *buffer, ByteMsg *msgp, struct tm *tmPtr )
{
    Nat32		i, j, dat[8];

    if ( getLine(buffer) == END_OF_FILE )
	return( END_OF_FILE );

    if ( sscanf(buffer, " %d-%d-%d %d:%d:%d", &tmPtr->tm_mon,
		    &tmPtr->tm_mday, &tmPtr->tm_year, &tmPtr->tm_hour,
		    &tmPtr->tm_min, &tmPtr->tm_sec) < 6 )
	return( BAD_MSG );

    tmPtr->tm_mon -= 1;

    if ( getLine(buffer) == END_OF_FILE )
	return( END_OF_FILE );

    if ( sscanf(buffer, " %*x %*x %*x %*x %*x %*x %x %x",
		     &dat[0], &dat[1]) < 2)
	    return( BAD_MSG );

    msgp->by_data[0] = (Byte)dat[0];
    msgp->by_data[1] = (Byte)dat[1];

    for ( i = 2; i < ARGOS_LEN - 6; i += 8 )
    {
	if ( getLine(buffer) == END_OF_FILE )
	    return( END_OF_FILE );

	if ( sscanf(buffer, " %x %x %x %x %x %x %x %x",
		    &dat[0], &dat[1], &dat[2], &dat[3],
		    &dat[4], &dat[5], &dat[6], &dat[7]) < 8 )
	    return( BAD_MSG );

	for ( j = 0; j < 8; j++ )
	    msgp->by_data[i + j] = (Byte)dat[j];
    }

    if ( getLine(buffer) == END_OF_FILE )
	return( END_OF_FILE );

    if ( sscanf(buffer, " %x %x %x %x %x %x",
		    &dat[0], &dat[1], &dat[2], &dat[3], &dat[4], &dat[5]) < 6 )
	    return( BAD_MSG );

    for ( j = 0; j < 6; j++ )
	msgp->by_data[i + j] = (Byte)dat[j];

    getLine( buffer );

    return( MSG_OK );

} /* getDiagnosticMsg() */


/************************************************************************/
/* Function    : msgMatch						*/
/* Purpose     : See if two ARGOS messages are the same			*/
/* Inputs      : Two Message ptrs					*/
/* Outputs     : TRUE if match, else FALSE				*/
/************************************************************************/
	MBool
msgMatch( ByteMsg *msgp1, ByteMsg *msgp2 )
{
    Int32	i;

    for ( i = 0; i < ARGOS_LEN; i++ )
	if ( msgp1->by_data[i] != msgp2->by_data[i] )
	    return( FALSE );

    return( TRUE );

} /* msgMatch() */


/************************************************************************/
/* Function    : findMatchingMessage					*/
/* Purpose     : Find a matching ARGOS message from those already received*/
/* Inputs      : Message ptr						*/
/* Outputs     : Message number of matching message, or ERROR		*/
/************************************************************************/
	Int32
findMatchingMessage( ByteMsg *msgp )
{
    Int32	i;

    for ( i = 0; i < numMsgs; i++ )
	if ( msgMatch(msgp, &msgs[i].msg.rawb) )
	    return( i );

    return( ERROR );

} /* findMatchingMessage() */


/************************************************************************/
/* Function    : getMessages						*/
/* Purpose     : Get all incoming ARGOS messages			*/
/* Inputs      : None							*/
/* Outputs     : None							*/
/************************************************************************/
	Void
getMessages( Void )
{
    char		buffer[128];
    struct tm		msgTime, *tmPtr;
    time_t		msg_time_t;
    MsgRtn		state;
    ArgosUnion		msg;
    Int32		msgNum, nmsgs;
    MBool		didError;

    numMsgs = 0;
    didError = FALSE;
    bzero( (void *)msgs, sizeof(msgs) );
    time( &msg_time_t );
    tmPtr = localtime( &msg_time_t );
    bcopy( (void *)tmPtr, (void *)&msgTime, sizeof(struct tm) );
    state = MSG_OK;
    itime = (1000 * tmPtr->tm_year) + tmPtr->tm_yday + 1;

    while ( state != END_OF_FILE )
    {
	nmsgs = 1;

	switch( inputMode )
	{
	  case SVC_ARGOS:
	    state = getSvcArgosMsg( buffer, &msg.raww, &msgTime, &nmsgs );
	    break;

	  case RCVR_NORM:
	    state = getNormalMsg( buffer, &msg.rawb, &msgTime );
	    break;

	  case RCVR_DIAG:
	    state = getDiagnosticMsg( buffer, &msg.rawb, &msgTime );
	    break;

	  default:
	    return;
	}

	if ( state == MSG_OK )
	{
	    msg_time_t = mktime( &msgTime );

	    if ( (msgNum = findMatchingMessage(&msg.rawb)) != ERROR )
	    {
		if ( msg_time_t < msgs[msgNum].first_msg )
		    msgs[msgNum].first_msg = msg_time_t;
		if ( msg_time_t > msgs[msgNum].last_msg )
		    msgs[msgNum].last_msg = msg_time_t;
		msgs[msgNum].msg_cnt += nmsgs;
	    }
	    else if ( numMsgs < (NUM_MSGS - 1) )
	    {
		msgs[numMsgs].first_msg = msg_time_t;
		msgs[numMsgs].last_msg = msg_time_t;
		msgs[numMsgs].msg_cnt = 1;
		bcopy( (void *)&msg, (void *)&msgs[numMsgs].msg.rawb,
		       sizeof(ArgosUnion) );
		numMsgs += nmsgs;
	    }
	    else if ( !didError )
	    {
		fprintf( stderr, "Too many different incoming messages\n" );
		didError = TRUE;
	    }
	}
    }

} /* getMessages() */


/************************************************************************/
/* Function    : printTime						*/
/* Purpose     : Print time to screen					*/
/* Inputs      : Time							*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printTime( time_t timeToPrint )
{
    struct tm	*tmPtr;

    tmPtr = localtime( &timeToPrint );
    printf( "%02d/%02d %02d:%02d:%02d", tmPtr->tm_mon + 1,
	    tmPtr->tm_mday, tmPtr->tm_hour, tmPtr->tm_min, tmPtr->tm_sec );

} /* printTime() */


/************************************************************************/
/* Function    : printNO3						*/
/* Purpose     : Print an NO3 buffer					*/
/* Inputs      : NO3 message ptr					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printNO3( No3Msg *no3p )
{
    Nat32	i, val1, val2;
    Reg Byte	*p;


    printf("NO3   ");	
    p = no3p->no3;

    for ( i = 0; i < CHEM_SAMPLES/2; i++ )
    {
	val1 = *p++ << 4;
	val1 |= ((*p >> 4) & 0x0f);
	val2 = ((*p++ & 0x0f) << 8);
	val2 |= *p++;
	printf( "%03d   %03d   ", val1, val2 );
    }

    printf("\nTemp ");
    for ( i = 0; i < CHEM_SAMPLES; i++ )
	printf("%5.2f ", (Flt32)(no3p->temp[i])/20.0 + 19.0);

    printf( "\n" );

} /* printNO3() */


/************************************************************************/
/* Function    : printSpec						*/
/* Purpose     : Print a PRR Spectro message				*/
/* Inputs      : Spectro message ptr, Address tags for Ed, Lu chans	*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printSpec( SpecMsg *msgp, Int32 edAdr, Int32 luAdr, Int32 luChan )
{
    Reg Int32	i;
    Reg Nat16	ival;

    for ( i = 0; i < ASPEC_CHANS; i++ )
    {
	ival = getIntelword( (Byte *)&msgp->sp_ed[i] );
	printf("%-5.5s %9.5f  ", spec_cal[SPECPRR_CAL].spc_cal[edAdr][i].name,
	      decode_prr_chan(ival, &spec_cal[SPECPRR_CAL].spc_cal[edAdr][i]));

	if ( (i % 4) == 3 )
	    printf("\n");
    }

    for ( i = 0; i < ASPEC_CHANS; i++ )
    {
	if ( (i % 4) == 0 )
	    printf("\n");

	ival = getIntelword( (Byte *)&msgp->sp_lu[i] );
	printf("%-5.5s %9.5f  ",
	       spec_cal[SPECPRR_CAL].spc_cal[luAdr][i+luChan].name,
	       decode_prr_chan(ival, 
			     &spec_cal[SPECPRR_CAL].spc_cal[luAdr][i+luChan]));
    }

} /* printSpec() */


/************************************************************************/
/* Function    : crcOneWord						*/
/* Purpose     : Check CRC-12 on one 16 bit word			*/
/* Inputs      : Initial CRC, word to check				*/
/* Outputs     : Resulting CRC						*/
/************************************************************************/
	Nat16
crcOneWord( Nat16 initCRC, Nat16 wordToCheck )
{
    Reg Nat32	bitCnt;
    Reg Nat16	crc;

    crc = initCRC;

    for ( bitCnt = 0; bitCnt < 16; bitCnt++ )
    {
	crc <<= 1;
	if ( wordToCheck & (0x8000 >> bitCnt) )
	    crc |= 1;
	if ( crc & CRC_MSB )
	    crc ^= CRC12;
    }

    return( crc );

} /* crcOneWord() */


/************************************************************************/
/* Function    : checkCRC						*/
/* Purpose     : Check CRC-12 on incoming message			*/
/* Inputs      : Word Message Pointer					*/
/* Outputs     : TRUE if CRC is OK, else FALSE				*/
/* Comment     : OASIS uses the standard CRC-12 polynomial 0x180f	*/
/*		 However, it puts the checksum and type information in	*/
/*		 the first word of the message, which it checks LAST	*/
/************************************************************************/
	MBool
checkCRC( WordMsg *msgp )
{
    Reg Nat32	wordCnt;
    Reg Nat16	crc;

    for ( wordCnt = 1, crc = 0; wordCnt < ARGOS_WORDS; wordCnt++ )
	crc = crcOneWord( crc, getIntelword((Byte *)&msgp->wd_data[wordCnt]) );

    return( crcOneWord(crc, getIntelword((Byte *)&msgp->wd_data[0])) == 0 );

} /* checkCRC() */


/************************************************************************/
/* Function    : printOneMsg						*/
/* Purpose     : Print one received ARGOS messages			*/
/* Inputs      : Message Pointer					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printOneMsg( ArgosInMsg *msgp )
{
    Nat32	i, msgType, val, msbs;
    Int32	ac9cnt;
    Flt64	flt1, flt2, flt3;
    Int16	co2;
    Nat16	mcp10, mcp30, fluor;
    MBool	isGood;
    
    if ( (isGood = checkCRC(&msgp->msg.raww)) )
	goodMsgs += msgp->msg_cnt;
    else
    {
	badMsgs += msgp->msg_cnt;
	if ( !printChksumErrors )
	    return;
    }

    msgType = (msgp->msg.rawb.by_data[1] >> 4) & 0x0f;

    printf( "%s Msg rcvd %d times from ", msgNames[msgType], msgp->msg_cnt );

    printTime( msgp->first_msg );
    printf( " to " );
    printTime( msgp->last_msg );
    printf( "\n" );

    if ( !isGood )
	printf("Checksum Error!\n");

    for ( i = 0; i < ARGOS_LEN; i++ )
    {
	printf("%02x ", msgp->msg.rawb.by_data[i]);
	if ( (i % 16) == 15 )
	    printf( "\n" );
    }

    switch( msgType )
    {
      case CHEM:
	printf("CO2   ");
	msbs = getIntelword( (Byte *)&msgp->msg.chem.ch_pco2_msbs );

	for ( i = 0; i < CHEM_SAMPLES; i++ )
	{
	    co2 = msgp->msg.chem.ch_pco2[i] | (((msbs >> (i + i)) & 3) << 8);
	    if ( co2 & 0x200 )
		co2 |= 0xfc00;				/* Sign extend	*/
	    printf( "%03d   ", co2 );
	}
	printf( "\n" );
	printNO3( &msgp->msg.chem.ch_no3 );
	break;

      case SPEC_0M_NOON:
	printSpec( &msgp->msg.spec0m, SPEC_0M_ED, SPEC_0M_LU,
		   SPEC_NOON_LU_CHAN );
	fluor = getIntelword( (Byte *)&msgp->msg.spec0m.sp_fluor );
	flt1 = ((analog[FLUOR_CHAN].a * (Flt32)(fluor)) +
		 analog[FLUOR_CHAN].b) * analog[FLUOR_CHAN].c;
	printf( "Fluor %9.5f\n", flt1 );
	break;

      case SPEC_0M_AM:
	printSpec( &msgp->msg.spec0m_1030, SPEC_0M_ED, SPEC_0M_LU,
		   SPEC_AM_LU_CHAN );
	fluor = getIntelword( (Byte *)&msgp->msg.spec0m.sp_fluor );
	flt1 = ((analog[FLUOR_CHAN].a * (Flt32)(fluor)) +
		 analog[FLUOR_CHAN].b) * analog[FLUOR_CHAN].c;
	printf( "Fluor %9.5f\n", flt1 );
	break;

      case SPEC_20M:
	printSpec( &msgp->msg.spec20m, SPEC_20M_ED, SPEC_20M_LU, 0 );
	fluor = getIntelword( (Byte *)&msgp->msg.spec0m.sp_fluor );
	printf("Fluor %9.5f\n",
	       decode_prr_chan(fluor, &spec_cal[SPECPRR_CAL].spc_cal[2][12]));
	break;

      case AC9_1:
	printf("AC-9  ");
	for ( i = 0; i < AC9_1_CHANS; i++ )
	{
	    val = getIntelword( (Byte *)&msgp->msg.ac9.ac_data[i] );
	    printf( "%s %5.3f  ", ac9ChanNames[i], (Flt32)val / 1000.0 );
	    if ( (i % 6) == 5 )
		printf( "\n      " );
	}
	printf( "\n" );
	break;

      case AC9_2:
	printf("AC-9  ");
	for ( i = 0; i < AC9_2_CHANS; i++ )
	{
	    val = getIntelword( (Byte *)&msgp->msg.ac92.ac_data[i] );
	    printf( "%s %5.3f  ", ac9ChanNames[i + AC9_1_CHANS],
		    (Flt32)val / 1000.0 );
	}

	ac9cnt = getIntelword( (Byte *)&msgp->msg.ac92.ac_temp );
	flt1 = decode_ac9_temp( ac9cnt );
	printf("Temp  %5.3f", flt1);

	printf("\nAC-9 Chl   ");
	for ( i = 0; i < CHEM_SAMPLES; i++ )
	{
	    val = getIntelword( (Byte *)&msgp->msg.ac92.ac_chl[i] );
	    printf( "%5.3f ", (Flt32)val / 1000.0 );
	}

	printf( "\n" );
	break;

      case MISC:
	val = getIntelword( (Byte *)&msgp->msg.misc.ms_pco2 );
	co2 = val & 0x3ff;
	if ( co2 * 0x200 )			/* Sign extend	*/
	    co2 |= 0xfc00;
	printf( "CO2 Calibr %3d at %02d00 local\n", co2,
	        ((val >> 10) & 7) * 3 );
	    
	val = getIntellong( (Byte *)&msgp->msg.misc.ms_oasis );
	flt1 = ((analog[TEMP_CHAN].a * (Flt32)(val & 0x3ff)) +
		  analog[TEMP_CHAN].b) * analog[TEMP_CHAN].c;
	flt2 = ((analog[OBATT_CHAN].a * (Flt32)((val >> 10) & 0x3ff)) +
		  analog[OBATT_CHAN].b) * analog[OBATT_CHAN].c;
	flt3 = ((analog[ABATT_CHAN].a * (Flt32)((val >> 20) & 0x3ff)) +
		  analog[ABATT_CHAN].b) * analog[ABATT_CHAN].c;

	printf("OASIS temp %5.2f C  Battery %5.2f V  Argos Batt %5.2f V\n",
	       flt1, flt2, flt3);

	if ( !smallConfig )
	{
	    mcp10 = getIntelword( (Byte *)&msgp->msg.misc.ms_mcp10 );
	    mcp30 = getIntelword( (Byte *)&msgp->msg.misc.ms_mcp30 );
	    printf("10m MCP %9.5f  30m MCP %9.5f\n",
		decode_prr_chan(mcp10, &spec_cal[SPECPRR_CAL].spc_cal[2][10]),
		decode_prr_chan(mcp30, &spec_cal[SPECPRR_CAL].spc_cal[2][11]));

	    printNO3( &msgp->msg.misc.ms_no3 );
	}
	break;

      default:
	printf("Bad Message Type\n");
    }

    printf( "\n" );

} /* printOneMsg() */


/************************************************************************/
/* Function    : printMessages						*/
/* Purpose     : Print the received ARGOS messages			*/
/* Inputs      : None							*/
/* Outputs     : None							*/
/************************************************************************/
	Void
printMessages( Void )
{
    Reg Nat32	i;

    for ( i = 0; i < numMsgs; i++ )
	printOneMsg( &msgs[i] );

    printf(" %d valid messages,  %d checksum errors\n", goodMsgs, badMsgs );

} /* printMessages() */

