/************************************************************************/
/* Copyright 1996 MBARI							*/
/************************************************************************/
/* $Header: argos.h,v 3.0 99/05/12 10:11:22 bobh Exp $			*/
/* Summary  : Definitions for ARGOS messages				*/
/* Filename : argos.h							*/
/* Author   : Robert Herlien (rah)					*/
/* Project  : OASIS Mooring						*/
/* $Revision: 3.0 $							    */
/* Created  : 06/28/96							*/
/************************************************************************/
/* Modification History:						*/
/* 28jun96 rah, created							*/
/* $Log:	argos.h,v $
 * Revision 3.0  99/05/12  10:11:22  10:11:22  bobh (Bob Herlien)
 * Added tstring, misc changes
 * 
 * Revision 2.9  98/08/24  13:45:49  13:45:49  bobh (Bob Herlien)
 * Archiving sources after M2/M3 & Eqpac deployments of 1998
 * 
 * Revision 2.8  98/03/17  11:11:35  11:11:35  bobh (Bob Herlien)
 * Archiving sources prior to porting to DOS/Windows
 * 
 * Revision 1.1  97/10/27  09:53:18  09:53:18  bobh (Bob Herlien)
 * Initial revision
 * 
*/
/************************************************************************/

#ifndef INCargosh
#define INCargosh		1

#define	ARGOS_LEN		32	/* Length of ARGOS data		    */
#define	ARGOS_WORDS	(ARGOS_LEN/2)	/* Length of data in words	    */

#define	CHEM_SAMPLES		 8	/* 8 samples/day for CO2, NO3, etc  */
#define CHEM4_SIZE		12	/* 12 bits * 8 samples = 12 bytes   */
#define NO3_SIZE		12	/* 12 bits * 8 samples = 12 bytes   */
#define ASPEC_CHANS		 7	/* Number of channels of spectro data*/
#define AC9_1_CHANS		15	/* Num chans in first AC-9 buff	    */
#define AC9_2_CHANS		 3	/* Num chans in second AC-9 buff    */
#define SPEC_0M_ED		 0	/* Address tag for surface Spec Ed  */
#define SPEC_0M_LU		 1	/* Address tag for surface Spec Lu  */
#define SPEC_20M_ED		 2	/* Address tag for 20m Spec Ed	    */
#define SPEC_20M_LU		 3	/* Address tag for 20m Spec Lu	    */
#define SPEC_NOON_LU_CHAN	 0	/* Channel for noon surface Spec Lu */
#define SPEC_AM_LU_CHAN		 8	/* Channel for 1030am surface Spec Lu*/
#define SPEC_1230_LU_CHAN	 8	/* Channel for 1230pm surface Spec Lu*/
#define SPEC_1230_LU_CHAN_REV4	 0	/* Channel for 1230pm surface Spec Lu*/
#define SAT1_ED			 1	/* Address tag for Satlantic 1 Ed   */
#define SAT1_LU			 1	/* Address tag for Satlantic 1 Lu   */
#define SAT1_ED_CHAN		 8	/* Channel for Satlantic 1 Ed	    */
#define SAT1_LU_CHAN		 8	/* Channel for Satlantic 1 Lu	    */
#define FLUOR_SIZE		12	/* 12 bits * 8 samples = 12 bytes   */
#define FLUOR20_WORDS		 6	/* 0, 6, 9, 12, 15, 1745	    */

/* Software & message type revision numbers:	*/
/* Rev 1 - Uses Rev 1&2 message types (below), messages had simple checksum */
/* Rev 2 - Same messages as Rev 1 but renumbered, messages had CRC	    */
/* Rev 3 - Delete AC-9, include Dark and shutter messages (as rev 3 below)  */
/* Rev 4 - Delete NO3, substitute PRR parms, add 10th msg type for Satlantic*/

#define DFLT_REV		3	/* Default revision of msg types    */


/****************************************/
/* ARGOS Message Buffer Identifiers	*/
/****************************************/

/* Revision 1 & 2 messages (rev 2 numbers, rev 1 had MISC last)	*/
#define CHEM		0		/* pCO2 and NO3 buffer		    */
#define SPEC_0M_NOON	1		/* Surface PRR, noon sample	    */
#define SPEC_0M_AM	2		/* Surface PRR, 1030 AM sample	    */
#define MISC		3		/* Misc data buffer		    */
#define SPEC_20M	4		/* 20 meter PRR, noon sample	    */
#define AC9_1		5		/* First part of AC-9 data	    */
#define AC9_2		6		/* Second part of AC-9 data	    */
#define NUM_MSG_TYPES_REV2 7		/* Number of message types	    */

/* Revision 3 messages different from above	*/
#define SPEC_0M_1230	2		/* Surface PRR, 1230 PM sample	    */
#define SPEC_20M_1230	5		/* 20 meter PRR, 1230 PM sample     */
#define SHUTTR		6		/* Shutter/Satlantic Dark	    */
#define SPEC_0M_DARK	7		/* Surface PRR, Midnight sample	    */
#define SPEC_20M_DARK	8		/* 20 meter PRR, Midnight sample    */
#define NUM_MSG_TYPES_REV3 9		/* Number of message types	    */

/* Additional Revision 4 message	*/
#define SAT1		9		/* Satlantic 1 message		    */
#define NUM_MSG_TYPES_REV4 10		/* Number of message types	    */

#define NUM_MSG_TYPES	NUM_MSG_TYPES_REV4


/****************************************/
/* Definitions for CRC-12 checksum	*/
/****************************************/

#define CRC_MSB		0x1000		/* MSB of CRC checksum word	    */
#define CRC12		0x180f		/* CRC-12 check polynomial	    */


/****************************************/
/* ARGOS Message Buffer Types		*/
/****************************************/

typedef struct				/************************************/
{					/* Chemical Buffer - pCO2 & NO3     */
    Byte	no3[NO3_SIZE];		/* Surface NO3 samples		    */
    Byte	temp[CHEM_SAMPLES];	/* Surface NO3 temperatures	    */
} No3Msg;				/************************************/

typedef struct				/************************************/
{					/* Chemical Buffer - pCO2 & NO3     */
    Word	ch_type;		/* Message type and checksum	    */
    Byte	ch_pco2[CHEM_SAMPLES];	/* 8 LSBs for each pCO2 sample	    */
    Word	ch_pco2_msbs;		/* 2 MSBs for each pCO2 sample	    */
    No3Msg	ch_no3;			/* Surface NO3 and temperatures	    */
} ChemMsg;				/************************************/

typedef struct				/************************************/
{					/* Wetstar fluorometer message	    */
    Byte	fl_0m[CHEM_SAMPLES];	/* Surface Wetstar samples	    */
    Byte	fl_20m[FLUOR_SIZE];	/* 20m Wetstar samples		    */
} FluorMsg;				/************************************/

typedef struct				/************************************/
{					/* Spectral Data Buffer		    */
    Word	sp_type;		/* Message type and checksum	    */
    Nat16	sp_ed[ASPEC_CHANS];	/* Downwelling (incident) channels  */
    Nat16	sp_lu[ASPEC_CHANS];	/* Upwelling (reflected) channels   */
    Nat16	sp_misc;		/* Fluorometer (or misc)	    */
} SpecMsg;				/************************************/

typedef struct				/************************************/
{					/* AC-9 Message Buffer 1	    */
    Word	ac_type;		/* Message type and checksum	    */
    Nat16	ac_data[AC9_1_CHANS];	/* First set of AC-9 data	    */
} Ac9Msg;				/************************************/

typedef struct				/************************************/
{					/* AC-9 Message Buffer 2	    */
    Word	ac_type;		/* Message type and checksum	    */
    Nat16	ac_data[AC9_2_CHANS];	/* Second set of AC-9 data	    */
    Nat16	ac_temp;		/* AC-9 Temperature		    */
    Nat16	ac_chl[CHEM_SAMPLES];	/* AC-9 Chlorophyll from 8 chem times*/
    Nat16	ac_pump;		/* Pump flow			    */
    Nat16	ac_fill[2];		/* No data			    */
} Ac92Msg;				/************************************/

typedef struct				/************************************/
{					/* Miscellaneous Message Buffer     */
    Word	ms_type;		/* Message type and checksum	    */
    Nat16	ms_pco2;		/* pCO2 calibration and sample time */
    Nat32	ms_oasis;		/* OASIS batt, temp, ARGOS batt	    */
    Nat16	ms_mcp10;		/* 10m MCP from 20m PRR		    */
    Nat16	ms_mcp30;		/* 30m MCP from 20m PRR		    */
    No3Msg	ms_no3;			/* 20m NO3 and temperatures	    */
} MiscMsg;				/************************************/

typedef struct				/************************************/
{					/* Misc Message Buffer for Rev3     */
    Word	ms_type;		/* Message type and checksum	    */
    Nat16	ms_pco2;		/* pCO2 calibration and sample time */
    Nat32	ms_oasis;		/* OASIS batt, temp, ARGOS batt	    */
    Nat16	ms_mcp10;		/* 10m MCP from 20m PRR		    */
    Nat16	ms_mcp30;		/* 30m MCP from 20m PRR		    */
    FluorMsg	ms_wetstar;		/* Wetstar values		    */
} Misc3Msg;				/************************************/

typedef struct				/************************************/
{					/* Shutter Message		    */
    Word	sh_type;		/* Message type and checksum	    */
    Nat16	sh_attempts;		/* Total attempts		    */
    Byte	sh_opens;		/* Total open operations	    */
    Byte	sh_closes;		/* Total close operations	    */
    Byte	sh_toterrs;		/* Non-zero returns		    */
    Byte	sh_errs[3];		/* Errors by type		    */
    Word	sh_oasisErrs;		/* OASIS error flags		    */
    Nat16	sh_satDark[ASPEC_CHANS]; /* Dark Satlantic 0 values	    */
    Nat16	sh_ctd_temp;		/* MicroCat temp at noon	    */
    Nat16	sh_ctd_cond;		/* MicroCat conductivity at noon    */
    Nat16	sh_fill;		/* Unused			    */
} ShutterMsg;				/************************************/

typedef struct				/************************************/
{					/* Chemical Buffer for Rev 4	    */
    Word	ch_type;		/* Message type and checksum	    */
    Byte	ch_pco2[CHEM4_SIZE];	/* pCO2 samples			    */
    Nat16	ch_prr_pitch;		/* Pitch from noon PRR sample	    */
    Nat16	ch_prr_roll;		/* Roll from noon PRR sample	    */
    Nat16	ch_prr_temp;		/* Temp from noon 20m PRR sample    */
    Nat16	ch_prr_depth;		/* Depth from noon 20m PRR sample   */
    Nat16	ch_prr_pitch1230;	/* Pitch from 1230 PRR sample	    */
    Nat16	ch_prr_roll1230;	/* Roll from 1230 PRR sample	    */
    Nat16	ch_prr_temp1230;	/* Temp from 1230 20m PRR sample    */
    Nat16	ch_prr_depth1230;	/* Depth from 1230 20m PRR sample   */
    Nat16	ch_fill;		/* Unused			    */
} Chem4Msg;				/************************************/

typedef struct				/************************************/
{					/* Miscellaneous Message Buffer     */
    Word	ms_type;		/* Message type and checksum	    */
    Nat16	ms_pco2;		/* pCO2 calibration and sample time */
    Nat32	ms_oasis;		/* OASIS batt, temp, ARGOS batt	    */
    Byte	fl_0m[FLUOR_SIZE];	/* Surface Wetstar samples	    */
    Nat16	fl_20m[FLUOR20_WORDS];	/* 20m Wetstar samples		    */
} Misc4Msg;				/************************************/

typedef struct				/************************************/
{					/* ARGOS message as array of bytes  */
    Byte	by_data[ARGOS_LEN];	/* Data				    */
} ByteMsg;				/************************************/

typedef struct				/************************************/
{					/* ARGOS message as array of words  */
    Nat16	wd_data[ARGOS_WORDS];	/* Data				    */
} WordMsg;				/************************************/

typedef union				/************************************/
{					/* Union of all ARGOS Message types */
    ChemMsg	chem;			/* Chemistry buffer		    */
    Chem4Msg	chem4;			/* Rev 4 Chem buffer		    */
    SpecMsg	spec0m;			/* Noon surface spectral buffer	    */
    SpecMsg	spec0m_1030;		/* 1030 AM surface spectral buffer  */
    SpecMsg	spec0m_1230;		/* 1030 AM surface spectral buffer  */
    SpecMsg	spec20m;		/* Noon 20 meter spectral buffer    */
    SpecMsg	spec20m_1230;		/* 1230 PM 20 meter spectral buffer */
    Ac9Msg	ac9;			/* First AC-9 buffer		    */
    Ac92Msg	ac92;			/* Second AC-9 buffer		    */
    MiscMsg	misc;			/* Miscellaneous buffer		    */
    Misc3Msg	misc3;			/* Rev 3 Miscellaneous buffer	    */
    Misc4Msg	misc4;			/* Rev 4 Miscellaneous buffer	    */
    ShutterMsg	shutter;		/* Shutter message		    */
    SpecMsg	spec0mDark;		/* Midnight Surface spectral buffer */
    SpecMsg	spec20mDark;		/* Midnight 20 meter spectral buffer*/
    SpecMsg	sat1;			/* Satlantic 1 message		    */
    ByteMsg	rawb;			/* Raw byte format		    */
    WordMsg	raww;			/* Raw word format		    */
} ArgosUnion;				/************************************/

typedef struct				/************************************/
{					/* Struct to hold incoming msgs	    */
    time_t	first_msg;		/* First time msg was sent	    */
    time_t	last_msg;		/* Last time msg was sent	    */
    Nat32	msg_cnt;		/* Number of times msg was sent	    */
    Int32	printed;		/* Boolean to say we printed it	    */
    ArgosUnion	msg;			/* Message			    */
} ArgosInMsg;				/************************************/


/****************************************/
/* Output file identifiers		*/
/****************************************/

typedef enum				/************************************/
{					/* Output file index		    */
    ERROR_FILE = 0,			/* Errors			    */
    CO2_FILE,				/* CO2				    */
    CO2_CAL_FILE,			/* CO2 calibration records	    */
    NO3_0_FILE,				/* Surface Nitrate		    */
    NO3_20_FILE,			/* 20m Nitrate			    */
    SPEC_NOON_FILE,			/* Noon surface spectro		    */
    SPEC_1030_FILE,			/* 1030AM surface spectro	    */
    SPEC_20_FILE,			/* Noon 20m spectro		    */
    MCP_FILE,				/* Noon 20m MCP			    */
    AC9_1_FILE,				/* AC-9, first half		    */
    AC9_2_FILE,				/* AC-9, second half		    */
    TEMP0_FILE,				/* Surface temperature from NO3	    */
    TEMP20_FILE,			/* 20m temperature from NO3	    */
    OASIS_FILE,				/* OASIS battery		    */
    ARGOS_FILE,				/* Number of hits on ARGOS xmission */
    SPEC_0M_1230_FILE,			/* 1230PM surface spectro	    */
    SPEC_20M_1230_FILE,			/* 1230PM 20 meter spectro	    */
    SPEC_0M_DARK_FILE,			/* Midnight surface spectro	    */
    SPEC_20M_DARK_FILE,			/* Midnight 20 meter spectro	    */
    WETSTAR_FILE,			/* Wetstar			    */
    SHUTTER_FILE,			/* PRR Shutter			    */
    SAT1_DARK_FILE,			/* Satlantic 1 Dark file	    */
    SAT1_FILE,				/* Satlantic 1 file		    */
    SPEC_MISC_FILE,			/* Pitch, roll, temp, depth from spec*/
} argosOutFileType;			/************************************/

#define ARGFILES	(SAT1_DARK_FILE+1)

#endif	/* INCargosh */
