/****************************************************************************/
/* Copyright 1995 MBARI                                                     */
/****************************************************************************/
/* $Header: satlant.c,v 4.4 2001/06/19 12:15:17 oasisa Exp $		    */
/* Summary  : Driver Routines for Satlantic Spectroradiometer on OASIS	    */
/* Filename : satlant.c							    */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 4.4 $							    */
/* Created  : 02/15/95 from sensors.c					    */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* 15feb95 rah - created from sensors.c					    */
/* $Log:	satlant.c,v $
 * Revision 4.4  2001/06/19  12:15:17  12:15:17  oasisa (Oasis users)
 * New Repository; 6/19/2001 (klh)
 * 
 * Revision 1.1  2001/06/19  11:44:39  11:44:39  oasisa (Oasis users)
 * Initial revision
 * 
 * Revision 3.4  96/06/18  15:24:25  15:24:25  bobh (Bob Herlien)
 * June '96 deployment of M1
 * 
 * Revision 3.1  95/03/09  19:31:01  19:31:01  hebo (Bob Herlien)
 * March '95 Deployment of M1A
 * 
 * Revision 3.0  95/02/21  18:42:46  18:42:46  hebo (Bob Herlien)
 * February '95 Deployment
 * 
*/
/****************************************************************************/

#include <types.h>			/* MBARI type definitions	    */
#include <const.h>			/* MBARI constants		    */
#include <oasis.h>			/* OASIS controller definitions	    */
#include <io.h>				/* OASIS I/O definitions	    */
#include <log.h>			/* Log record definitions	    */
#include <80C196.h>			/* 80196 Register mapping           */
#include <task.h>			/* OASIS Multitasking definitions   */

#define SATLANT_BUFSIZE	128		/* Line buffer size for Satlantic   */
#define MAX_SATLANT_WRDS 20		/* Max number words to log for Satlan*/

/********************************/
/*	External Functions	*/
/********************************/

Extern Void	drvLogWords( Driver *dp, Nat16 *samplep, Int16 wrds );
Extern Void	drvLogError( Driver *dp, Errno err );
Extern char	*drvSerPortAndMalloc( Driver *dp, Nat16 size );
Extern Void	drvSerReleaseAndFree( Driver *dp, char *buffer );
Extern Void	drv_pwroff( Driver *dp );
Extern Int16	xgets_tmout( char *s, Int16 len, Nat16 tmout );
Extern MBool	getnum( char **s, Int16 *result, Nat16 radix );
Extern Void	bzero( void *s, int n );
Extern char	*tmpMalloc( Nat16 size );
Extern Void	tmpFree( char *ptr );


/************************************************************************/
/* Function    : get_satlantic						*/
/* Purpose     : Make one attempt to get Satlantic Spectro data		*/
/* Inputs      : Buffer, ptr to place to store result, timeout		*/
/* Outputs     : Number of words received from Satlantic		*/
/************************************************************************/
	Int16
get_satlantic( char *buff, Nat16 *result, Nat16 tmout )
{
    char	*p;
    Reg Nat16	i;

    while ( (i = xgets_tmout(buff, SATLANT_BUFSIZE, tmout)) == 0 )
	   ;					/* If CR or LF, try again */

    if ( i < 0 )
	return( 0 );

    for ( i = 0, p = buff; getnum(&p, (Int16 *)(result + i), 10); i++ )
	;
    
    return( i );
    
} /* get_satlantic() */


/************************************************************************/
/* Function    : satlantic_drv						*/
/* Purpose     : Satlantic Spectroradiometer driver			*/
/* Inputs      : Driver Pointer						*/
/* Outputs     : None							*/
/************************************************************************/
	Void
satlantic_drv( Driver *dp )
{
    Nat16	i, samples, avgs;
    char	*satlant_buf;
    Nat32	*accum;
    Nat16	*samplep;
    Reg Nat16	j;

    samples = dp->drv_parms[PARM0];		/* Get number words in sample*/

    satlant_buf = drvSerPortAndMalloc(dp, SATLANT_BUFSIZE +
			    (samples * (sizeof(Nat16) + sizeof(Nat32))));
    if ( satlant_buf == NULL )			/* Alloc buff, return if none*/
	return;

    accum = (Nat32 *)(satlant_buf + SATLANT_BUFSIZE);
    samplep = (Nat16 *)(accum + samples);
    bzero( accum, samples * sizeof(Nat32) );

    avgs = 0;				/* Init number samples to average*/

    if ( get_satlantic(satlant_buf, samplep, dp->drv_parms[PARM2]) > 0 )
    {					/* Get & throw away 1st sample	*/

	for ( i = dp->drv_parms[PARM1], avgs = 0; i; i-- )
	    if ( get_satlantic(satlant_buf, samplep, dp->drv_parms[TIMEOUT])
		 >= samples )
	    {				/* Get & average PARM1 samples	*/
		avgs++;
		for ( j = 0; j < samples; j++ )
		    accum[j] += samplep[j];
	    }
    }

    if ( avgs > 0 )			/* If got some samples,		*/
    {					/*   compute the means		*/
	for ( j = 0; j < samples; j++ )
	    samplep[j] = accum[j] / avgs;
					/*   and log them		*/
	drvLogWords( dp, samplep, samples );
    }
    else				/* Else, log the error		*/
	drvLogError( dp, NO_DATA );
	
    drv_pwroff( dp );			/* Turn off power		*/
    drvSerReleaseAndFree( dp, satlant_buf ); /* Release serial port & buf*/

} /* satlantic_drv() */
