/****************************************************************************/
/* Copyright 1991 MBARI                                                     */
/****************************************************************************/
/* $Header: oasis.c,v 4.4 2001/06/19 12:14:48 oasisa Exp $			    */
/* Summary  : Main Routine for OASIS Mooring Controller                     */
/* Filename : oasis.c                                                       */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 4.4 $ (Note changed manually.  Still need to RCS)		    */
/* Created  : 02/06/91                                                      */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* 06feb91 rah - created						    */
/* $Log:	oasis.c,v $
 * Revision 4.4  2001/06/19  12:14:48  12:14:48  oasisa (Oasis users)
 * New Repository; 6/19/2001 (klh)
 * 
 * Revision 1.1  2001/06/19  11:44:19  11:44:19  oasisa (Oasis users)
 * Initial revision
 * 
 * Revision 4.3  99/06/16  10:21:31  10:21:31  bobh (Bob Herlien)
 * Mar/May '99 Deployments of M3/M2
 * 
 * Revision 4.2  98/09/09  10:48:00  10:48:00  bobh (Bob Herlien)
 * Sept/Oct '98 deployments of M1, Eqpac 1 & 2
 * 
 * Revision 4.1  98/05/12  09:35:06  09:35:06  bobh (Bob Herlien)
 * June '98 turnaround for EqPac
 * 
 * Revision 4.0  98/03/09  11:43:40  11:43:40  bobh (Bob Herlien)
 * M3 Deployment of March '98, new Sat-Pac driver
 * 
 * Revision 3.9  97/10/28  13:58:51  13:58:51  bobh (Bob Herlien)
 * EqPac Deployment of Nov 1997
 * 
 * Revision 3.8  97/09/12  10:50:24  10:50:24  bobh (Bob Herlien)
 * Redeploy M1
 * 
 * Revision 3.7  97/07/23  11:24:28  11:24:28  bobh (Bob Herlien)
 * July '97 M1 deployment, new shutter code
 * 
 * Revision 3.6  96/10/30  13:59:18  13:59:18  bobh (Bob Herlien)
 * Release for EqPac, M2 ReplacTest Replacement
 * 
 * Revision 3.5  96/07/19  09:53:20  09:53:20  bobh (Bob Herlien)
 * July '96 Deployment of M2 with ARGOS code
 * 
 * Revision 3.4  96/06/18  15:23:44  15:23:44  bobh (Bob Herlien)
 * June '96 deployment of M1
 * 
 * Revision 3.3  95/04/13  13:45:53  13:45:53  hebo (Bob Herlien)
 * Drifter Deployment for Coop (flip) cruise
 * 
 * Revision 3.2  95/04/11  14:02:28  14:02:28  hebo (Bob Herlien)
 * Drifter Deployment on IronEx
 * 
 * Revision 3.1  95/03/09  19:30:06  19:30:06  hebo (Bob Herlien)
 * March '95 Deployment of M1A
 * 
 * Revision 3.0  95/02/21  18:42:09  18:42:09  hebo (Bob Herlien)
 * February '95 Deployment
 * 
 * Revision 2.6  94/09/16  14:34:17  14:34:17  hebo (Bob Herlien)
 * Change CTD to use "TS" for opto-isolated CTD interface
 * 
 * Revision 2.5  94/01/10  11:31:38  11:31:38  hebo (Bob Herlien)
 * Fixed bug in "getlog", changed PRR format
 * 
 * Revision 2.4  93/10/29  11:12:10  11:12:10  hebo (Bob Herlien)
 * November 1993 Deployment
 * 
 * Revision 2.3  93/10/12  08:29:15  08:29:15  hebo (Bob Herlien)
 * Oct '93 Deployment of M2
 * 
 * Revision 2.2  93/04/06  11:54:28  11:54:28  hebo (Bob Herlien)
 * Minor change to GPS for 5-channel Magellan
 * 
 * Revision 2.1  93/01/08  10:42:04  10:42:04  hebo (Bob Herlien)
 * Fixed bug with 24 hr radio silence.  M2 swap due to failed electronics.
 * 
 * Revision 2.0  92/08/26  10:37:34  10:37:34  hebo (Bob Herlien 408-647-3748)
 * August 1992 Deployment
 * 
 * Revision 1.3  92/03/03  16:41:13  16:41:13  hebo (Bob Herlien 408-647-3748)
 * New defaults, restart check, perm power stuff, analog command
 * 
*/
/****************************************************************************/

#include <types.h>			/* MBARI type definitions	    */
#include <const.h>			/* MBARI constants		    */
#include <oasis.h>			/* OASIS controller definitions	    */
#include <io.h>				/* OASIS I/O definitions	    */
#include <80C196.h>			/* 80196 Register mapping           */
#include <task.h>			/* OASIS task dispatcher	    */
#include <custom.h>			/* PWR_PERM definition		    */


/********************************/
/*	External Functions	*/
/********************************/

Extern Void	io_init(Void);		/* Initialize I/O module	    */
Extern Void	io_term(Void);		/* Power down I/O module	    */
Extern Void	ioCheckXoffStatus(Void); /* See if XOFFs have timed out	   */
Extern MBool	chkRadioTmout( Void );	/* Check Radio Timeout 		    */
Extern Void	mallocInit(Void);	/* Initialize memory module	    */
Extern Void	mallocColdInit(Void);	/* Cold start of memory module	    */
Extern MBool	clk_init(Void);		/* Init real-time clock		    */
Extern MBool	clk_read(Void);		/* Read real-time clock		    */
Extern Void	clk_alarm(Nat16 altime);/* Set RTC alarm		    */
Extern Void	clk_pwrdown(Void);	/* Power down			    */
Extern Void	drvr_init(Void);	/* Initialize driver module	    */
Extern Nat16	drvr_sample(Void);	/* Call sample() func of all drvrs  */
Extern Void	drvrColdInit(Void);	/* Cold init of drivers		    */
Extern Void	logColdInit(Void);	/* Cold init of log routines	    */
Extern Void	parmColdInit(Void);	/* Cold init of global parameters   */
Extern Void	usrColdInit(Void);	/* Cold init of user i/f	    */
Extern Void	logError( Word vect );	/* Log OASIS system error	    */
Extern Void	logOasisStatus( Word onoff ); /* Log OASIS system status    */
Extern Void	usr_drv( Driver *dp );	/* User interface driver	    */
Extern Driver	*drvr_find( char *name ); /* Find driver by name	    */
#ifdef DISK
Extern Void	diskInit( Void );	 /* Initialize disk routine	    */
#endif
#ifdef ARGOS
Extern Void	swapArgosBufs( Void );	 /* Swap the argos buffers	    */
Extern Void	argosOasisError( Void ); /* Log OASIS error		    */
#endif

#ifdef BLINKY
extern volatile unsigned char ledon;
extern volatile unsigned char ledoff;
#endif

/********************************/
/*	External Data		*/
/********************************/

Extern Reg Word		error;		/* Error vector			    */
Extern Reg Nat16	tick;		/* 10 ms ticker			    */
Extern Reg TimeOfDay	tod;		/* Current time in TimeOfDay format */
Extern Reg DateTime	dttm;		/* Current date & time in DateTime  */
Extern Reg Byte		localHour;	/* Hour of day (0 - 23) in local time*/
Extern Nat16		pwr_perm;	/* Devices to leave powered on	    */
Extern Int16		wdcnt;		/* Countdown counter for watchdog   */
Extern Nat16		radio_tmout;	/* Timeout for radio reset	    */
Extern LstHead		drv_list;	/* Driver list header		    */
Extern Nat16		deploymentMode;	/* Non-zero if in deployment mode   */
Extern Nat16		logStatus;	/* Log on power up/down		    */


/********************************/
/*	Global Data		*/
/********************************/

Global Reg MBool	ext_wake;	/* TRUE if woke by /wake pin	    */
Global Nat16		sys_init;	/* Set to MAGIC at system init	    */
Global TimeOfDay	radio_time;	/* Last time user connected on radio*/
					/*  (set in userif.c, used here)    */

const char * const	signon = "\nOASIS $Revision: 4.4 $\n";


/********************************/
/*	Module Local Data	*/
/********************************/

MLocal Nat16		sys_pwrdown;	/* Set to MAGIC to show pwrd down OK*/
MLocal TimeOfDay	alarm_time;	/* Time to sync for setting alarm   */
MLocal Byte		didSync;	/* Did midnight sync		    */


/************************************************************************/
/* Function    : chkRadioTmout						*/
/* Purpose     : Check for (nominally) 24 hour timeout on radio connections*/
/* Inputs      : None							*/
/* Outputs     : TRUE if timed out, FALSE if not			*/
/************************************************************************/
	MBool
chkRadioTmout( Void )
{
    if ( radio_tmout && (tod > (radio_time + (radio_tmout * 3600L))) )
    {					/* If no radio traffic for	*/
	error |= COMM_ERR;		/*  24 hrs, set error vector	*/
	return( TRUE );			/*  and return TRUE		*/
    }

    return( FALSE );

} /* chkRadioTmout() */


    
/************************************************************************/
/* Function    : oasis_init						*/
/* Purpose     : Initialize OASIS controller				*/
/* Inputs      : None							*/
/* Outputs     : None							*/
/************************************************************************/
	Void
oasis_init( Void )
{
    Reg Word	errSave;
    
    error = 0;				/* Init error vector		*/

    if ( sys_init != MAGIC )		/* If RAM checksum bad, startup	*/
    {					/*  clears sys_init		*/
	error = RAM_ERR;		/* Show RAM error		*/
	pwr_perm = PWR_PERM;		/* Power default devices	*/
	sys_init = MAGIC;		/* Initialize flag		*/
    }

    io_init();				/* Init I/O system, starts watchdog*/
    mallocInit();			/* Init temp RAM		*/

    if ( !clk_init() )			/* Initialize the real-time clock*/
	error |= CLOCK_ERR;		/* If bad, show so		*/

    if ( sys_pwrdown != MAGIC )		/* If not shut down properly,	*/
	error |= RESTART_ERR;		/*   force a restart		*/

    sys_pwrdown = 0;			/* Show we're NOT powered down	*/
/*    clk_read();*/				/* for tod and drvrColdInit()	*/

#ifdef RADIO
    chkRadioTmout();			/* If no radio for 24 hrs, set error*/
#endif

    if ( error & (RAM_ERR | LOG_ERR | COMM_ERR) )
    {					/* If critical error, restart drvrs*/
	errSave = error;
	mallocColdInit();		/* Reinitialize all RAM		*/
	usrColdInit();			/* Initialize user i/f		*/
	parmColdInit();			/* Reset default parameters	*/
	drvrColdInit();			/* Cold init drvr module	*/
	radio_time = tod;		/* Init radio timeout		*/
	ext_wake = TRUE;		/* For debugging, but OK otherwise*/

	if ( errSave & RAM_ERR )
	{				/* If RAM checksum was bad	*/
	    logColdInit();		/*   reinitialize log memory	*/
#ifdef ARGOS
	    swapArgosBufs();		/* Swap the argos buffers	*/
	    deploymentMode = TRUE;	/* Turn on deployment mode	*/
	    didSync = TRUE;		/* So we don't turn right back off*/
#endif
	}
	error = errSave;
    }

    task_init();			/* Initialize task module	*/
					/* Main function becomes Init task*/
    drvr_init();			/* Init driver module		*/
#ifdef DISK
    diskInit();				/* Init disk module		*/
#endif

    if ( tod < alarm_time )		/* If we woke up before alarm,	*/
	ext_wake = TRUE;		/* it must be due to wakeup button*/

} /* oasis_init() */


/************************************************************************/
/* Function    : _main							*/
/* Purpose     : Main entry point					*/
/* Inputs      : None							*/
/* Outputs     : None							*/
/* Comments    : Never returns						*/
/************************************************************************/
	Void
_main( Void )
{
    Nat16	next_alarm;		/* Seconds 'til next alarm	   */
    Int16	button;			/* Counter for /EXTWAKE button	   */
    Nat16	loopCntr;		/* Loop counter			   */
    Reg Driver	*dp;			/* Driver ptr			   */
    Driver	*usrdp;			/* Driver ptr for user i/f	   */
    Nat16	xoffTick;		/* For XOFF timing		   */
    MBool       led=FALSE;              /* Blink Light State		   */

    oasis_init();			/* Initialize			   */

#ifdef LOG_STATUS
    if ( logStatus & LOG_PWRUP )	/* Log power up event if enabled   */
	logOasisStatus( 1 );
#endif

#ifdef SYNC_AT_MIDNIGHT
    if ( localHour != 0 )		/* See if we just passed midnight  */
	didSync = FALSE;		/* If no, reset didSync var	   */
    else if ( !didSync )		/* If yes, did we already sync?	   */
    {
	didSync = TRUE;			/* Do just once per day		   */
#ifdef ARGOS
	swapArgosBufs();		/* Swap the argos buffers	   */
	deploymentMode = 0;		/* Turn off deployment mode	   */
#endif
	for (dp = (Driver *)drv_list.lst_head; dp != DRV_NULL;
	     dp = dp->drv_next)
	    if ( dp->drv_parms[INTERVAL] )
	    {
		dp->drv_wakeup = tod;
		dp->drv_flags |= (DO_SYNC | DO_ARGOS);
	    }
    }
#endif

    button = loopCntr = 0;
    xoffTick = tick;
    usrdp = drvr_find( "UserIF" );

    do					/* Main loop - until drivers done  */
    {					/* Loop executes at 10 Hz	   */
      if ( (++loopCntr % 5) == 0 ){
	    clk_read();			/* Get current time twice per second*/
#ifdef BLINKY
	    if(led==FALSE){
	      ledon=1;
              led=TRUE;
	    }else{
	      ledoff=1;
              led=FALSE;
	    }
#endif /*BLINKY*/
      }
	if ( ioport2 & EXTWAKE_BIT )
	{				/* If user attention button is down*/
	    ext_wake = TRUE;		/*   run user interface		   */
	    if ( ++button >= 20 )	/* See if he's been holding it for */
		asm rst;		/*   2 seconds.  If so, restart.   */
	}
	else button = 0;

	if ( chkRadioTmout() )		/* If 24 hr radio timeout, reset   */
	    asm rst;

	if ( ext_wake && (usrdp != DRV_NULL) )
	    usrdp->drv_wakeup = tod;	/* This is split from EXTWAKE_BIT  */
					/*  test due to switch bounce	   */
	next_alarm = drvr_sample();	/* Start any drvr tasks not running*/
					/*  and get time to next wakeup    */
	task_delay(TICKS_PER_SECOND/10); /* Wait to see if drvrs running   */
	wdcnt = WATCH_COUNT;		/* Keep us ticking		   */

	if ( (tick - xoffTick) > TICKS_PER_SECOND )
	{				/* XOFF timeout code		   */
	    xoffTick += TICKS_PER_SECOND;
	    ioCheckXoffStatus();	/* See if XOFFs have timed out	   */
	}

    } while ( next_alarm < 4 );		/* Repeat until all drivers done   */

    if ( error )			/* If any generic errors,	   */
    {
	logError( error | ERR_LOG_BIT ); /*   log them			   */
#ifdef ARGOS
	argosOasisError();		/* Tell ARGOS module		   */
#endif
    }
    
#ifdef LOG_STATUS
    if ( logStatus & LOG_PWRDOWN )	/* Log power down event if enabled */
	logOasisStatus( 0 );
#endif

    alarm_time = tod + next_alarm;	/* Remember when we set alarm	   */
    clk_alarm( next_alarm );		/* Set alarm for next sample	   */
    io_term();				/* Tell I/O system about pwr down  */
    sys_pwrdown = MAGIC;		/* Set powerdown variable	   */
    clk_pwrdown();			/* DONE.  Power down the system	   */

} /* _main() */
