/****************************************************************************/
/* Copyright 1998 MBARI                                                     */
/****************************************************************************/
/* $Header: metsys.c,v 4.4 2001/06/19 12:14:38 oasisa Exp $                     */
/* Summary  : Driver Routines for MetSystem on OASIS moorings       */
/* Filename : metsys.c                                                      */
/* Author   : Danelle E. Cline                                              */
/* Project  : OASIS M2 Mooring                                              */
/* $Revision: 4.4 $                                                         */
/* Created  : 02/10/98                                              */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:                                                                                                    */
/* 10feb98  Danelle C.- created                                                                                         */
/* $Log:	metsys.c,v $
 * Revision 4.4  2001/06/19  12:14:38  12:14:38  oasisa (Oasis users)
 * New Repository; 6/19/2001 (klh)
 * 
 * Revision 1.1  2001/06/19  11:44:12  11:44:12  oasisa (Oasis users)
 * Initial revision
 * 
 * Revision 4.3  99/06/16  10:21:42  10:21:42  bobh (Bob Herlien)
 * Mar/May '99 Deployments of M3/M2
 * 
 * Revision 4.2  98/09/09  10:48:12  10:48:12  bobh (Bob Herlien)
 * Sept/Oct '98 deployments of M1, Eqpac 1 & 2
 * */
/****************************************************************************/
#include <metsys.h>
#include <types.h>                      /* MBARI type definitions           */
#include <const.h>                      /* MBARI constants                  */
#include <oasis.h>                      /* OASIS controller definitions     */
#include <io.h>                         /* OASIS I/O definitions            */
#include <log.h>                        /* Log record definitions           */
#include <custom.h>                     /* CTD_TS definition                */
#include <80C196.h>                     /* 80196 Register mapping           */
#include <task.h>                       /* OASIS Multitasking definitions   */
#include <stdio.h>                      /* Standard I/O functions           */
#include <ctype.h>


/*************************************************************************/
/*                                      External Functions               */
/*************************************************************************/
Extern Void     drvLog( Driver *dp, Byte *samplep, Int16 len );
Extern Void     drvLogError( Driver *dp, Errno err );
Extern char     *drvSerPortAndMalloc( Driver *dp, Nat16 size );
Extern Void	drvSerReleaseAndFree( Driver *dp, char *buffer );
Extern Void     xputs( const char *s );
Extern Int16    xgetc_tmout( Nat16 tmout  );
Extern Int16    xgetn_tmout( char *s, Int16 len, Nat16 tmout );
Extern char     *find_str( char *src, char *tgt );
Extern Void     tmpFree( char *ptr );
Extern MBool    delimit( Reg char c );  
Extern Void     bzero( void *s, int n );        
Extern Void     bcopy(const Byte *src, Byte *dst, Nat16 len);
Extern  Void    xprintf( const char *format, ... );     
Extern Void     xflush_ser( Nat16 timeout );
Extern MBool    getnum( char **s, Int16 *result, Nat16 radix );
Extern Void     drv_pwron( Driver *dp );
Extern Void     drv_pwroff( Driver *dp );
Extern char     *skipField( Reg char *p, Nat16 nfields );


/************************************************************************/
/* Function    : MetSysDrv                                                      */
/* Purpose     : veryifies record checksum valid        */
/* Inputs      : data buffer and length                                 */
/* Outputs     : OK or ERROR                                            */
/************************************************************************/
Void MetSysDrv(Driver *dp)
{
    char *metsysbuf;
    Nat16 curmptr;
    Nat16 tmpmptr;
    Nat16 nrec;
    Nat16 ntrys;
    Nat16 nunread;
    Nat16 len;
    MBool status;
    Nat16 resyncRecords;

    metsysbuf = NULL;
    resyncRecords = dp->drv_parms[PARM0];
	
    if((metsysbuf = drvSerPortAndMalloc(dp, BUFF_SIZE)) == NULL)
	return;

    /*initialize*/
    bzero(metsysbuf,BUFF_SIZE);     
    status = ERROR;
    curmptr = 0;
    tmpmptr =0;
    nrec = 0;
    ntrys =0;
    nunread = 0;
    len = 0;
			
    /*wakup logger */
    if(WakeUp(dp) == OK)
    {       
	while(status == ERROR && ntrys++ < 5)
	{
	    xflush_ser(0);
	    xputs(STATUS_CMD);      
	    len = xgetn_tmout(metsysbuf, BUFF_SIZE, dp->drv_parms[TIMEOUT]);
	    if((status = VerifyCheckSum(metsysbuf, len))==ERROR)
		dp->drv_cnt++; /*checksum error*/               
	}
	
	if(status == OK)
	{       
	/*Get num unread records, & update curmptr*/
	    nunread = GetUnreadRecords(metsysbuf, &curmptr, &dp->drv_parms[PARM1]);
	
	    tmpmptr = curmptr; 

	    if(nunread > 0)
	    {
	    /*if number of unread records in range (0 - dp->drv_parms[PARM0]*/                      
		if(nunread < resyncRecords )
		    xprintf("%d%s",dp->drv_parms[PARM1], MOVEMPTR_CMD);  
		else
		    xprintf("%d%s",resyncRecords,BACKUP_CMD);       
	
		ntrys = 0;
		xflush_ser(TICKS_PER_SECOND);

		do
		{
		    if((nrec = ReadData(dp, metsysbuf, &curmptr)) > 0)
		    {                                               
			drvLog(dp, (Byte *)metsysbuf, nrec*REC_SIZE);   
			dp->drv_parms[PARM1] = curmptr; /*save last mptr */
		    }
		    else
			ntrys++;        
		}while( (curmptr < tmpmptr) && (ntrys < resyncRecords) );

	    }/*end if status record recv'd correctly */

	}/*end if nunread > 0*/
	else
	    drvLogError(dp, CKSUM_ERR);
		    
	if(nrec==0)
	    drvLogError(dp, SYNC_ERR);
	
	MetSleep();                             
    } /*end if metsys woken */              
    else
	drvLogError(dp, NOT_AVAIL_ERR);
	
    drvSerReleaseAndFree( dp, metsysbuf );       /* Release serial port          */
    drv_pwroff(dp);     
}


/************************************************************************/
/* Function    : VerifyCheckSum                                                                                 */
/* Purpose     : veryifies if record checksum valid                                             */
/* Inputs      : data buffer and length                                                                 */
/* Outputs     : OK or ERROR                                                                                    */
/************************************************************************/
MBool VerifyCheckSum(char * buff, Nat16 len)
{
    char            *c;
    Reg Nat16       calcchecksum;
    Nat16           checksum, curlen;
	
    calcchecksum = 0;
    c=buff;

    for( ; len > 0; len--, c++ )
    {       
	if(*c == '*')
	    return( ERROR );

	calcchecksum += (Nat16)*c;
	if(*c == 'C') /*found checksum */
	{
	    c++;
	    if(getnum(&c,(Int16 *)&checksum,10) != ERROR)
	    {
		calcchecksum %= MAXCHECKSUM;
		if(calcchecksum == checksum)
		    return OK;

		return ERROR;
	    }
	}

    }/*end for */           

    return ERROR;
}


/************************************************************************/
/* Function    : GetUnreadRecords                                                                               */
/* Purpose     : Updates new read mptr                                                                  */
/* Inputs      : data buffer, old/new memptrs                                                   */
/* Outputs     : number of unrecords                                                                    */
/************************************************************************/
Nat16 GetUnreadRecords(char *buff, Nat16 *newmptr, Nat16 *oldmptr)
{
    Nat16 i;
    char *c;

    i=0;
    *newmptr = 0;

    for(c=buff;c!=NULL;c++)
    {               
	if(*c == 'L')/*found memory location delimiter */
	{                       
	    c+=2;/*skip over the sign */                                            
	    getnum(&c,(Int16 *)newmptr,10);
	    if(*newmptr >= *oldmptr)
		return((*newmptr - *oldmptr + 1)/NUM_REC);                        
	    else
		return((*newmptr - *oldmptr + 1 + NUM_MEM_LOC)/NUM_REC);
	}       
    }
    return 0;
}


/************************************************************************/
/* Function    : WakeUp                                                                                                 */
/* Purpose     : wakeup metsys                                                                                  */
/* Inputs      : data buffer & driver struct                                                    */
/* Outputs     : OK is awaken, else ERROR                                                               */
/************************************************************************/
MBool WakeUp(Driver *dp)
{       
    Nat16 i;
    Reg Int16 c;
    i = 0;

    for(i = 0;i < 10 ;i++)
    {
	xputs("\r");
	while ( (c = xgetc_tmout(dp->drv_parms[TIMEOUT])) != ERROR )
	    if ( c == '*' )         /* Look for prompt              */
		return( OK );

    } /*end for */
    return ERROR;
}


/************************************************************************/
/* Function    : Sleep                                                                                                  */
/* Purpose     : put metsys to sleep                                                                    */
/* Inputs      : none                                                                                                   */
/* Outputs     : none                                                                                                   */
/************************************************************************/
Void MetSleep()
{
    xputs("E\r");   
}


/************************************************************************/
/* Function    : ParseData                                                                                              */
/* Purpose     : parses data records and places in buff                             */
/* Inputs      : data buff and curReadPtr                                                               */
/* Outputs     : Returns num records found                                                              */
/************************************************************************/
Nat16 ParseData(char * buff, Nat16 *curReadPtr)
{
    char    *c;        
    Nat16 i;

    c = buff;
    i=0;

/*find the first record id */
    if( (c = find_str(buff,"01+") ) !=  NULL) 
    {
	c-=3;
	while(c != NULL)
	{
	    if(*c != '\n' || *c != '\r')/*skip over newline &carriage rtrns*/
		buff[i] = *c;			
 
	    if(*c == 'L')/*found the new memptr */
	    {       
		c+=2;
		if(getnum(&c,(Int16 *)curReadPtr,10)!= ERROR)
		    return i/REC_SIZE;
	    }
	    c++;
	    i++;
	}/*end while */
    }
	
    return 0;           
}


/************************************************************************/
/* Function    : ReadData                                                                                               */
/* Purpose     : Dumps and parses data records from logger                          */
/* Inputs      : data buff and length, curReadPtr, nrecords to read             */
/* Outputs     : Returns  num data records read                                                 */
/************************************************************************/
Nat16 ReadData(Driver *dp, char *buff, Nat16 *mptr)
{
    Nat16 n, len;
    Nat16 ntrys;
    Nat16 ptr;

    n = 0;
    ntrys = 0;
    ptr = 0;
    len = 0;

    xputs(DUMP_CMD);        /*send dump 1 record command */

    if((len = xgetn_tmout(buff, BUFF_SIZE, dp->drv_parms[TIMEOUT])) > 0)
    {       
	if(VerifyCheckSum(buff, len) == OK)
	    n = ParseData(buff, &ptr);              
	else
	{                       
	    xprintf("1%s",BACKUP_CMD);      
	    xflush_ser(TICKS_PER_SECOND);
	    dp->drv_cnt++;/*checksum error*/
	}
    }

    if(n > 0)
	*mptr = ptr;

    return n;
}
