/****************************************************************************/
/* Copyright 1999 MBARI                                                     */
/****************************************************************************/
/* $Header: hydrodas.c,v 4.5 2001/10/17 10:28:51 oasisa Exp $		    */
/* Summary  : Driver Routines for Hobilabs HydroDAS on OASIS mooring	    */
/* Filename : hydrodas.c						    */
/* Author   : Kent Headley (klh)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 4.5 $							    */
/* Created  : 07/18/00							    */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* 18jul00 klh - created						    */
/* $Log:	hydrodas.c,v $
 * Revision 4.5  2001/10/17  10:28:51  10:28:51  oasisa (Oasis users)
 * internal separation of HR channels
 * 
 * Revision 4.4  2001/06/19  12:14:08  12:14:08  oasisa (Oasis users)
 * New Repository; 6/19/2001 (klh)
 * 
 * Revision 1.1  2001/06/19  11:43:56  11:43:56  oasisa (Oasis users)
 * Initial revision
 * 
*/
/****************************************************************************/

#include <types.h>			/* MBARI type definitions	    */
#include <const.h>			/* MBARI constants		    */
#include <oasis.h>			/* OASIS controller definitions	    */
#include <io.h>				/* OASIS I/O definitions	    */
#include <log.h>			/* Log record definitions	    */
#include <custom.h>			/* CTD_TS definition		    */
#include <80C196.h>			/* 80196 Register mapping           */
#include <task.h>			/* OASIS Multitasking definitions   */
#include <stdio.h>			/* Standard I/O functions	    */
#include <string.h>			/* String library functions	    */
#include <ctype.h>			/* Standard character types	    */
#include "hobi.h"

#define CTRL_C		3		/* Control-C character		    */
#define HOBI_BUFSIZE	1500		/* Specified max is 1340	    */
#define HOBI_WAKETIME	1
#define HOBI_FLUSHTIME	(TICKS_PER_SECOND/2)
#define NUM_HOBI_FILES 9

/********************************/
/*	External Functions	*/
/********************************/

Extern Void	drvLog( Driver *dp, Byte *samplep, Int16 len );
Extern Void	drvLogError( Driver *dp, Errno err );
Extern Void	drv_pwroff( Driver *dp );
Extern char	*drvSerPortAndMalloc( Driver *dp, Nat16 size );
Extern Void	drvSerReleaseAndFree( Driver *dp, char *buffer );
Extern Void	xputc( Int16 c );
Extern Void	xputs( const char *s );
Extern Void	xprintf( const char *format, ... );
Extern Int16	xgets_tmout( char *s, Int16 len, Nat16 tmout );
Extern Int16	xgetn_tmout( char *s, Int16 len, Nat16 tmout );
Extern Void	xflush_ser( Nat16 tmout );
Extern Void	logPutRec( LogRecHdr *hdrp, Byte *dp );
Extern Void     bzero(void *s, int n);
Extern Int16    xgetc_tmout( Nat16 tmout  );
Extern char*    find_str( char *src, char *tgt );

/********************************/
/*	External Functions	*/
/********************************/

Extern Reg TimeOfDay	tod;		/* Current time in TimeOfDay format */
Extern volatile Reg Nat16 tick;		/* 10 ms ticker			    */


/************************************************************************/
/* Function    : wakeup_hydrodas					*/
/* Purpose     : Wakeup HydroDAS Controller				*/
/* Inputs      : Driver ptr, buffer					*/
/* Outputs     : TRUE if woke, else FALSE				*/
/************************************************************************/
	MBool
wakeup_hydrodas( Driver *dp, char *buffer )
{
    Int16	i;
    Nat16	st_tick;
    
    for ( i = dp->drv_parms[PARM0]; i; i-- )
    {
	xputc( CTRL_C );
	st_tick = tick;
	if ( xgets_tmout(buffer, HOBI_BUFSIZE, HOBI_WAKETIME) > 0 )
	{
	    deblank(buffer);
	    if ( find_str(buffer, "Ready") != NULL ){
		return( TRUE );
	    }
	}

	xflush_ser( TICKS_PER_SECOND - (tick - st_tick) );
    }
    
    return( FALSE );

} /* wakeup_hydrodas() */


/************************************************************************/
/* Function    : hydrodas_drv						*/
/* Purpose     : Driver for HOBI Labs HydroDAS Controller		*/
/* Inputs      : Driver Pointer						*/
/* Outputs     : None							*/
/* PARMS       : 0: Wakeup retries					*/
/*		 1: Unused						*/
/*		 2: Unused						*/
/************************************************************************/
	Void
hydrodas_drv( Driver *dp )
{
    Reg Int16	bufLen,dataLen;
    char	*hobiBuf;
    Int16	cnt;
    Nat16	tmout;
    LogRecHdr	rh;
    Nat16 i;
    Nat16	secs, st_tick;
    char *hobiFiles[]={"AB.BIN","CB.BIN","HS2.BIN","HR41.BIN","HR42.BIN","HR43.BIN","HR44.BIN","HR21.BIN","HR22.BIN"};
    Byte hobiTypes[]={HOBI_AB,HOBI_CB,HOBI_HS2,HOBI_HR41,HOBI_HR42,HOBI_HR43,HOBI_HR44,HOBI_HR21,HOBI_HR22};
    Nat16 hobiIDMask[]={fAB,fCB,fHS2,fHR41,fHR42,fHR43,fHR44,fHR21,fHR22};

    if ( (hobiBuf = drvSerPortAndMalloc(dp, HOBI_BUFSIZE)) == NULL ){
	drvLogError( dp, MALLOC_ERR );		/* No malloc space	*/
	return;
    }

    if (dp->drv_flags & DO_INIT){
      xputc(0x02);				/* To do on power up	*/
      xputc(0x13);
      xputs("M1.scr");			/* Load data collection script */
      xputc(0x00);
      xputc(0x02);
      xputc(0x1C);/*0x8B or 0x1C*/
      xputs("200000");				/* put HDAS to sleep	*/
      xputc(0x00);
      dp->drv_flags &= ~DO_INIT;                /* Turn off INIT flag   */
   }

    if ( wakeup_hydrodas(dp, hobiBuf) ){	/* Got mem and wakeup	*/
	xflush_ser( HOBI_FLUSHTIME );
	tmout = dp->drv_parms[TIMEOUT];

	for (i=0;i<NUM_HOBI_FILES;i++){
	  if(hobiIDMask[i] & dp->drv_parms[PARM1]){
	    xputc(0x02);				/* Get & Log One File */
	    xputc(0x30);
	    xputs(hobiFiles[i]);
	    xputc(0x00);
	    rh.log_len = xgetn_tmout(hobiBuf, HOBI_BUFSIZE, tmout);
	    rh.log_type = hobiTypes[i];
	    logPutRec( &rh, (Byte *)hobiBuf );
	    task_delay(TICKS_PER_SECOND/4); /* Be a good citizen */
	  }
	}

	/* Need this until firmware patch...put back to sleep */
        xputc(0x02);				/* Run data collection script */
        xputc(0x15);
        xputc(0x00);

    }else
	drvLogError( dp, SYNC_ERR );		/* Else no wakeup	*/
    
    /*drvLog(dp,(Byte *)hobiBuf,sizeof(hobiBuf));*/
    drvSerReleaseAndFree( dp, hobiBuf );	/* Release serial port	*/
    drv_pwroff( dp );				/* Turn off power	*/

} /* hydrodas_drv() */

