/****************************************************************************/
/* Copyright 2000 MBARI                                                     */
/****************************************************************************/
/* $Header: gashound.c,v 4.4 2001/06/19 12:13:47 oasisa Exp $*/
/* Summary  : Driver Routines for pCO2 Anaylyzer on OASIS mooring	    */
/*	      Gashound is based on the Tattletale8 (tt8) data acquisition   */
/*	      system.							    */
/* Filename : gashound.c						    */
/* Author   : Kent Headley (klh)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 4.4 $							    */
/* Created  : 06/06/00 from no3.c					    */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* 06jun00 klh - created from no3.c					    */
/* 
 *$Log:	gashound.c,v $
 * Revision 4.4  2001/06/19  12:13:47  12:13:47  oasisa (Oasis users)
 * New Repository; 6/19/2001 (klh)
 * 
 * Revision 1.1  2001/06/19  11:43:35  11:43:35  oasisa (Oasis users)
 * Initial revision
 * 
*/
/****************************************************************************/

#include <types.h>			/* MBARI type definitions	    */
#include <const.h>			/* MBARI constants		    */
#include <oasis.h>			/* OASIS controller definitions	    */
#include <io.h>				/* OASIS I/O definitions	    */
#include <log.h>			/* Log record definitions	    */
#include <custom.h>			/* Deployment-specific definitions  */
#include <task.h>			/* OASIS Multitasking definitions   */
#include <argos.h>			/* ARGOS definitions		    */
#include <stdio.h>			/* Standard I/O functions	    */
#include <ctype.h>			/* Standard ctype.h defs	    */

#define CO2_BUFSIZE	512		/* Size of buffer for pCO2	    */
#define CO2_LINE_MAX	100
/*#define USE_WAKESTR*/                 /* Different ways of waking the     */
#define USE_LONG_BREAK                  /* instrument			*/
#define RETRY_SOON


/********************************/
/*	External Functions	*/
/********************************/
Extern Void	drvLog( Driver *dp, Byte *samplep, Int16 len );
Extern char	*drvSerPortAndMalloc( Driver *dp, Nat16 size );
Extern Void	drvSerReleaseAndFree( Driver *dp, char *buffer );
Extern Void	xputs( const char *s );
Extern Void	xflush_ser( Nat16 tmout );
Extern Int16	xgets_tmout( char *s, Int16 len, Nat16 tmout );
Extern char 	*find_str( char *src, char *tgt );
Extern Void	tmpFree( char *ptr );
Extern Void     ser_break(Nat16 port, Int16 count );

/********************************/
/*	External Data		*/
/********************************/
Extern Reg TimeOfDay	tod;		/* Current time in TimeOfDay format */


/********************************/
/*	Module Local Data	*/
/********************************/



/************************************************************************/
/* Function    : wake_gashound						*/
/* Purpose     : Wake up the pCO2 analyzer				*/
/* Inputs      : Driver ptr, wakeup string, buffer to use		*/
/* Outputs     : TRUE if woke up pCO2, else FALSE			*/
/************************************************************************/
	MBool
wakeup_gashound( Driver *dp, char *wakestr, char *buffer )
{
    Nat16	i,j;
 
#ifdef USE_WAKESTR
    for ( i = dp->drv_parms[PARM0]; i; i-- )
    {
	xputs(wakestr);			/* Wake up pCO2 sensor */

	while ( xgets_tmout(buffer, 30, dp->drv_parms[TIMEOUT]) >= 0 )
	    if ( find_str(buffer, "okay") != NULL )
	    {					/* Look for ACK message	    */
		xflush_ser( TICKS_PER_SECOND );	/* Flush serial buffer	    */
		return( TRUE );			/* Got it, return TRUE	    */
	    }
   }

   return( FALSE );
#endif /*USE_WAKESTR*/

#ifdef USE_LONG_BREAK
    for ( i = dp->drv_parms[PARM0]; i; i-- )
    {
	ser_break(dp->drv_parms[SER_PORT], 10*TICKS_PER_SECOND);

	if ( find_str(buffer, "ready") != NULL )
	{
	  ;/*ignore ready*/
	}
	xputs(wakestr);			/* Wake up pCO2 sensor */

	while ( xgets_tmout(buffer, 30, dp->drv_parms[TIMEOUT]) >= 0 )

	    if ( find_str(buffer, "okay") != NULL )
	    {					/* Look for ACK message	    */
		xflush_ser( TICKS_PER_SECOND );	/* Flush serial buffer	    */
		return( TRUE );			/* Got it, return TRUE	    */
	    }
   }
   return( FALSE );
#endif /*USE_LONG_BREAK*/   

} /* wakeup_gashound() */

/************************************************************************/
/* Function    : gashound_wake						*/
/* Purpose     : CO2 serial wakeup function				*/ 
/* Inputs      : Driver ptr, Boolean (TRUE for on, FALSE for off)	*/
/* Outputs     : TRUE							*/
/************************************************************************/
	MBool
gashound_wake( Driver *dp, MBool on, char *buffer )
{
  if ( on )				/* Wake instrument		*/
	wakeup_gashound( dp, "wake\r", buffer );
    else
	xputs("bye\r");			/* Put instrument to sleep	*/

    return( TRUE );

} /* gashound_wake() */


/************************************************************************/
/* Function    : getCo2Sample						*/
/* Purpose     : Get one pCO2 data sample				*/
/* Inputs      : Driver Pointer, buffer ptr				*/
/* Outputs     : Number of bytes gotten, 0 for end of record		*/
/************************************************************************/
	Int16
getCo2Sample( Driver *dp, char *bufp )
{
    Reg Int16		curLen;
    Reg Nat16		val;

    do
    {
	curLen = xgets_tmout(bufp, CO2_LINE_MAX, dp->drv_parms[TIMEOUT]);
	if ( curLen < 0 )
	    return( 0 );
    } while ( curLen == 0 );


    return( curLen );

} /* getCo2Sample() */


/************************************************************************/
/* Function    : gashound_drv						*/
/* Purpose     : pCO2 Sensor driver					*/
/* Inputs      : Driver Pointer						*/
/* Outputs     : None							*/
/* Parms       :							*/
/*		 PARM[0]: Wakeup Retry Count				*/
/*		 PARM[1]: Seconds until retry when no sync		*/
/*		 PARM[2]: Max Failure Retries				*/
/* Comments    : Driver for pCO2 sensor based on TattleTale8		*/
/*		 (tt8,TxBasic) data acquisition system. The tt8 works	*/
/*		 independently and talks to oasis via a serial		*/
/*		 interface. It stays in	 a deep sleep, waking once an	*/
/*		 hour to sample, then stays awake until contacted by	*/
/*		 the oasis driver. Should the lose sync, it will begin	*/
/*               requesting data every three minutes until the tt8	*/
/*		 responds. When in sync, the oasis driver makes a data	*/
/*		 request once per sample cycle to a waiting tt8, which	*/
/************************************************************************/
	Void
gashound_drv( Driver *dp )
{
    Nat16	bufLen;
    char	*gashound_buf;		/* Buffer for pCO2 data		*/
    Reg Nat16	curLen;
    TimeOfDay   entryTime;
  

    entryTime = tod;			/* Save tod to sync next time	*/


    if (dp->drv_flags & DO_SYNC)	/* OnSync, reset failure count  */
      dp->drv_cnt = 0;
					/* Do nothing until SYNC if too */
					/* many failures		*/
    if (dp->drv_cnt > dp->drv_parms[PARM2] )
      return;

    if ( (gashound_buf = drvSerPortAndMalloc(dp, CO2_BUFSIZE)) == NULL )
	return;


    if ( wakeup_gashound(dp, "wake\r", gashound_buf) )
    {					/* Wake up CO2			*/
	
       bufLen = curLen = 0;		/* Is this necessary?		*/
       xputs( "data\r" );		/* Send data request string	*/

       while( (curLen = getCo2Sample(dp, gashound_buf + bufLen)) > 0 )
       {
          bufLen += curLen;
          gashound_buf[bufLen++] = '\n';
          if ( bufLen >= (CO2_BUFSIZE - CO2_LINE_MAX) )
          {
        	  drvLog( dp, (Byte *)gashound_buf, bufLen );
		  bufLen = 0;
          }
       }
       xputs( "bye\r" );		/* Go back to sleep NOW		*/
       if ( bufLen > 0 )
         drvLog( dp, (Byte *)gashound_buf, bufLen );

					/* This wakeup time worked,	*/
					/* sync to it for next time	*/
       dp->drv_wakeup = entryTime + dp->drv_parms[INTERVAL];
			
       dp->drv_cnt = 0;			/* Reset Failure Counter	*/

    }else
    {					/* If Wakeup unsuccessful, run  */
					/* again in a few (PARM1) minutes */
      dp->drv_cnt++;			/* Increment Failure Count	*/

      if(dp->drv_cnt < 2)               /* Retry once after first failure */
        dp->drv_wakeup = tod + dp->drv_parms[PARM1];

    }
 
    drvSerReleaseAndFree( dp, gashound_buf ); /* Release serial port & buf*/

} /* gashound_drv() */
