/****************************************************************************/
/* Copyright 1995 MBARI                                                     */
/****************************************************************************/
/* $Header: atlas.c,v 4.4 2001/06/19 12:12:38 oasisa Exp $			    */
/* Summary  : Driver Routines for ATLAS controller on OASIS mooring	    */
/* Filename : atlas.c							    */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 4.4 $							    */
/* Created  : 02/15/95 from sensors.c					    */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* 15feb95 rah - created from sensors.c					    */
/* $Log:	atlas.c,v $
 * Revision 4.4  2001/06/19  12:12:38  12:12:38  oasisa (Oasis users)
 * New Repository; 6/19/2001 (klh)
 * 
 * Revision 1.1  2001/06/19  11:42:40  11:42:40  oasisa (Oasis users)
 * Initial revision
 * 
 * Revision 3.4  96/06/18  15:24:21  15:24:21  bobh (Bob Herlien)
 * June '96 deployment of M1
 * 
 * Revision 3.3  95/04/13  13:46:55  13:46:55  hebo (Bob Herlien)
 * Drifter Deployment for Coop (flip) cruise
 * 
 * Revision 3.1  95/03/10  15:56:36  15:56:36  hebo (Bob Herlien)
 * March '95 Deployment of M1
 * 
 * Revision 3.0  95/02/21  18:42:43  18:42:43  hebo (Bob Herlien)
 * February '95 Deployment
 * 
*/
/****************************************************************************/

#include <types.h>			/* MBARI type definitions	    */
#include <const.h>			/* MBARI constants		    */
#include <oasis.h>			/* OASIS controller definitions	    */
#include <io.h>				/* OASIS I/O definitions	    */
#include <log.h>			/* Log record definitions	    */
#include <custom.h>			/* ATLAS_BIT definition		    */
#include <80C196.h>			/* 80196 Register mapping           */
#include <task.h>			/* OASIS Multitasking definitions   */
#include <stdio.h>			/* Standard I/O functions	    */
#include <string.h>			/* String library functions	    */

#define ATLAS_BUFSIZE	128		/* Line buffer size for ATLAS	    */
#define ATLAS_DBUFSIZE	60		/* Space for 30 words ATLAS result  */
#define CTRLG		0x07		/* ASCII ctrl-G			    */
#define CTRLT		0x14		/* ASCII ctrl-T			    */
#define NAK		0x15		/* ASCII NAK			    */
#define ATLAS_NAK_TIME	180		/* Seconds to wait if get NAK	    */


/********************************/
/*	External Functions	*/
/********************************/

Extern Void	drvLogWords( Driver *dp, Nat16 *samplep, Int16 wrds );
Extern char	*drvSerPortAndMalloc( Driver *dp, Nat16 size );
Extern Void	drvSerReleaseAndFree( Driver *dp, char *buffer );
Extern Void	xputc( Int16 c );
Extern Int16	xgets_tmout( char *s, Int16 len, Nat16 tmout );
Extern Void	newline( Void );
Extern MBool	getnum( char **s, Int16 *result, Nat16 radix );
Extern char	*tmpMalloc( Nat16 size );
Extern Void	tmpFree( char *ptr );


/********************************/
/*	External Data		*/
/********************************/

Extern Reg Byte		io_wakebits;	/* Copy of wakeup bits		    */
Extern Reg TimeOfDay	tod;		/* Current time in TimeOfDay format */


/************************************************************************/
/* Function    : atlas_wake						*/
/* Purpose     : ATLAS serial wakeup function				*/ 
/* Inputs      : Driver ptr, Boolean (TRUE for on, FALSE for off)	*/
/* Outputs     : TRUE							*/
/************************************************************************/
	MBool
atlas_wake( Driver *dp, MBool on )
{
    if (on)
	io_wakebits |= ATLAS_BIT;	/* Turn on DTR to ATLAS		*/
    else
	io_wakebits &= ~ATLAS_BIT;	/* Turn off DTR to ATLAS	*/

    wakeport = io_wakebits;		/* Send local copy to hardware	*/
    return( TRUE );

} /* atlas_wake() */


/************************************************************************/
/* Function    : get_atlas						*/
/* Purpose     : Make one attempt to get ATLAS data			*/
/* Inputs      : Driver Pointer, buffer, ptr to place to store result	*/
/* Outputs     : Number of words received from ATLAS, or ERROR if NAK	*/
/************************************************************************/
	Int16
get_atlas( Driver *dp, char *buff, Nat16 *result )
{
    Nat16	i;
    Reg char	*p;
    char	*q;

    newline();				/* ATLAS may eat first char	*/
    task_delay( TICKS_PER_SECOND/2 );
    xputc( CTRLT );			/* Send ctrl-t			*/
    
    for ( i = 0; i < dp->drv_parms[PARM0]; )
    {
	if ( xgets_tmout(buff, ATLAS_BUFSIZE, dp->drv_parms[TIMEOUT]) < 0 )
	    break;

	for ( p = buff; *p; p++ )
	    if ( *p == NAK )
		return( ERROR );
	    else if ( *p == CTRLG )
		*p = ' ';

	for ( q = buff; getnum(&q, (Int16 *)&result[i], 16); )
	    i++;
    }
    
    return( i );
    
} /* get_atlas() */


/************************************************************************/
/* Function    : atlas_drv						*/
/* Purpose     : ATLAS driver						*/
/* Inputs      : Driver Pointer						*/
/* Outputs     : None							*/
/************************************************************************/
	Void
atlas_drv( Driver *dp )
{
    Reg Int16	words;
    Int16	i;
    char	*atlas_buf;
    Nat16	*samples;

    if ((atlas_buf = 
	 drvSerPortAndMalloc(dp, ATLAS_BUFSIZE + ATLAS_DBUFSIZE)) != NULL)
    {					/* Malloc buffer, error if no mem*/
	samples = (Nat16 *)(atlas_buf + ATLAS_BUFSIZE);

	atlas_wake( dp, TRUE );		/* Turn on ATLAS serial port	 */
					   
	for ( i = dp->drv_parms[PARM1]; i; i-- )
	{
	    words = get_atlas(dp, atlas_buf, samples);

	    if ( words == ERROR )
	    {
		dp->drv_wakeup = tod + ATLAS_NAK_TIME;
		break;
	    }

	    if ( words >= dp->drv_parms[PARM0] )
		break;
	}

	drvLogWords( dp, samples, words );
	drvSerReleaseAndFree( dp, atlas_buf );
    }

    atlas_wake( dp, FALSE );		/* Turn off ATLAS serial port	*/

} /* atlas_drv() */
