/****************************************************************************/
/* Copyright 1996 MBARI                                                     */
/****************************************************************************/
/* $Header: argping.c,v 4.4 2001/06/19 12:12:34 oasisa Exp $		    */
/* Summary  : Driver Routines for Argos satellite interface on OASIS mooring*/
/* Filename : argosBeacon.c						    */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 4.4 $							    */
/* Created  : 04/22/96							    */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* 15jun98 rah - created from argos.c					    */
/* $Log:	argping.c,v $
 * Revision 4.4  2001/06/19  12:12:34  12:12:34  oasisa (Oasis users)
 * New Repository; 6/19/2001 (klh)
 * 
 * Revision 1.1  2001/06/19  11:42:36  11:42:36  oasisa (Oasis users)
 * Initial revision
 * 
*/
/****************************************************************************/

#include <types.h>			/* MBARI type definitions	    */
#include <const.h>			/* MBARI constants		    */
#include <oasis.h>			/* OASIS controller definitions	    */
#include <custom.h>			/* ARGOS_BIT definition		    */
#include <io.h>				/* OASIS I/O definitions	    */
#include <task.h>			/* OASIS Multitasking definitions   */

#define ACK		0x06
#define ST_XMIT1_CMD	0x30

#define RANDOM_DIVISOR	3121		/* Yields random nmbr between 0 & 20*/
					/* For 9-21, make this 180	    */

/********************************/
/*	External Data		*/
/********************************/

Extern Reg Byte		io_wakebits;	/* Copy of wakeup bits		    */
Extern Reg Word		error;		/* Error vector			    */
Extern Reg DateTime	dttm;		/* Current date & time in DateTime  */


/********************************/
/*	External Functions	*/
/********************************/

Extern Void	drv_ser_port( Driver *dp );
Extern Void	drv_ser_release( Driver *dp );
Extern Void	xputc( Int16 c );
Extern Void	xputn( const char *s, Int16 len );
Extern Int16	xgetc_tmout( Nat16 tmout );
Extern Void	xflush_ser( Nat16 tmout  );
Extern Void	bzero( void *s, int n );	/* Zero memory		    */
Extern Nat16	random( Void );



/************************************************************************/
/* Function    : wakeup_argos						*/
/* Purpose     : Wake up ARGOS PTT					*/ 
/* Inputs      : None							*/
/* Outputs     : None							*/
/************************************************************************/
	Void
wakeup_argos_p( Void )
{
    io_wakebits |= ARGOS_BIT;			/* Wake up the PTT	*/
    wakeport = io_wakebits;			/* Send to hardware	*/
    task_delay( 4 );				/* Leave it on 30-40 ms	*/
    io_wakebits &= ~ARGOS_BIT;			/* Turn off wakeup bit	*/
    wakeport = io_wakebits;			/* Send to hardware	*/
    xflush_ser( 1 );				/* Throw away any input	 */

} /* wakeup_argos_p() */


/************************************************************************/
/* Function    : ping_ptt						*/
/* Purpose     : Wake up PTT, send one buffer				*/
/* Inputs      : Driver ptr						*/
/* Outputs     : TRUE if successful, else FALSE				*/
/************************************************************************/
	MBool
ping_ptt( Driver *dp )
{
    Reg Int16	c;				/* Scratch char buffer	*/
    
    wakeup_argos_p();				/* Wake up argos PTT	*/
    xputc( ST_XMIT1_CMD | 0 );

    for ( c = 0; c < 32; c++ )
	xputc( 0 );

    c = xgetc_tmout( dp->drv_parms[TIMEOUT] );	/* Get response		 */
    return( c == ACK );				/* Check for ACK	 */

} /* ping_ptt() */


/************************************************************************/
/* Function    : argosPing						*/
/* Purpose     : Driver to send dummy message to Argos PTT		*/
/* Inputs      : Driver Pointer						*/
/* Outputs     : None							*/
/************************************************************************/
	Void
argosPing( Driver *dp )
{
    Nat16	i;

    drv_ser_port( dp );			/* Setup serial port		*/

    for ( i = dp->drv_parms[PARM0]; (i > 0) && (!ping_ptt(dp)); i-- )
	delay_secs( dp->drv_parms[TIMEOUT] );

    drv_ser_release( dp );		/* Release serial port		*/

    if ( i <= 0 )			/* If retried out, set error	*/
	error |= ARGOS_ERR;

    dp->drv_wakeup += ((Int16)(random()/RANDOM_DIVISOR) - 10);

} /* argosPing() */
