/****************************************************************************/
/* Copyright 1996 MBARI                                                     */
/****************************************************************************/
/* $Header: argos.c,v 4.4 2001/06/19 12:12:10 oasisa Exp $			    */
/* Summary  : Driver Routines for Argos satellite interface on OASIS mooring*/
/* Filename : argos.c							    */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 4.4 $							    */
/* Created  : 04/22/96							    */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* 22apr96 rah - created from nrl.c					    */
/* $Log:	argos.c,v $
 * Revision 4.4  2001/06/19  12:12:10  12:12:10  oasisa (Oasis users)
 * New Repository; 6/19/2001 (klh)
 * 
 * Revision 1.1  2001/06/19  11:42:22  11:42:22  oasisa (Oasis users)
 * Initial revision
 * 
 * Revision 4.2  98/09/09  10:48:05  10:48:05  bobh (Bob Herlien)
 * Sept/Oct '98 deployments of M1, Eqpac 1 & 2
 * 
 * Revision 4.1  98/05/12  09:35:10  09:35:10  bobh (Bob Herlien)
 * June '98 turnaround for EqPac
 * 
 * Revision 4.0  98/03/09  11:44:43  11:44:43  bobh (Bob Herlien)
 * M3 Deployment of March '98, new Sat-Pac driver
 * 
 * Revision 3.9  97/10/28  13:59:52  13:59:52  bobh (Bob Herlien)
 * EqPac Deployment of Nov 1997
 * 
 * Revision 3.7  97/07/23  11:18:13  11:18:13  bobh (Bob Herlien)
 * July '97 M1 deployment, new shutter code
 * 
 * Revision 3.6  96/10/30  14:00:18  14:00:18  bobh (Bob Herlien)
 * Release for EqPac, M2 Test Replacement
 * 
 * Revision 3.5  96/07/19  09:53:36  09:53:36  bobh (Bob Herlien)
 * July '96 Deployment of M2 with ARGOS code
 * 
 * Revision 3.4  96/06/18  15:24:27  15:24:27  bobh (Bob Herlien)
 * June '96 deployment of M1
 * 
*/
/****************************************************************************/

#include <types.h>			/* MBARI type definitions	    */
#include <const.h>			/* MBARI constants		    */
#include <oasis.h>			/* OASIS controller definitions	    */
#include <custom.h>			/* ARGOS_BIT definition		    */
#include <argos.h>			/* ARGOS buffer definitions	    */
#include <io.h>				/* OASIS I/O definitions	    */
#include <log.h>			/* Log record definitions	    */
#include <task.h>			/* OASIS Multitasking definitions   */
#include <ac9.h>			/* AC-9 definitions		    */
#include <specprr.h>			/* PRR spectro definitions	    */
#include <string.h>			/* for strlen()			    */
#include <hobi.h>			/* Hobi Labs HS2 definitions	    */
#ifdef TINYSHUT
#include <tinyshut.h>			/* Reduced-size shutter driver	    */
#else
#include <shut22.h>                     /* shutter definitions              */
#endif

#define ACK		0x06
#define NACK		0x15
#define NULL_CMD	0xb0
#define STORE1_CMD	0x10		/* PTT index num OR'd into low nibble*/
#define STORE2_CMD	0x20
#define ST_XMIT1_CMD	0x30
#define ST_XMIT2_CMD	0x40
#define XMIT1_CMD	0x50
#define XMIT2_CMD	0x60

#define RANDOM_DIVISOR	3121		/* Yields random nmbr between 0 & 20*/
#define NOON_SAMPLE	4
#define NO3_TEMP_OFFSET	380		/* For WARMWATER, temp range 19 - 31*/

					/* For 9-21, make this 180	    */
#define OASIS_ANALOG_DONE 0x8000
#define CTD_NOON_DONE	0x4000
#define PRR_NOON_DONE	0x8000
#define PRR_1230_DONE	0x4000
#define PRR_1000_DONE	0x2000
#define PRR_MIDNITE_DONE 0x1
#define WETSTAR_1745	0x400


#ifndef NO3Sample
typedef struct				/************************************/
{					/* Result of one NO3 sample	    */
    Nat16	no3;			/* Nitrate value		    */
    Nat16	temp;			/* Temperature, centidegrees C	    */
} NO3Sample;				/************************************/
#endif


/********************************/
/*	External Data		*/
/********************************/

Extern Reg Byte		io_wakebits;	/* Copy of wakeup bits		    */
Extern Reg Word		error;		/* Error vector			    */
Extern Reg DateTime	dttm;		/* Current date & time in DateTime  */
Extern Reg Byte		localHour;	/* Hour of day (0 - 23) in local time*/
Extern Nat16		deploymentMode;	/* In deployment mode mode	    */


/********************************/
/*	External Functions	*/
/********************************/

Extern Void	drv_ser_port( Driver *dp );
Extern Void	drv_ser_release( Driver *dp );
Extern Void	drv_pwroff( Driver *dp );
Extern Void	xputc( Int16 c );
Extern Void	xputn( const char *s, Int16 len );
Extern Int16	xgetc_tmout( Nat16 tmout );
Extern Void	xflush_ser( Nat16 tmout  );
Extern Void	bcopy( const Byte *src, Byte *dst, Nat16 len );
Extern Void	bzero( void *s, int n );	/* Zero memory		    */
Extern Nat16	swap( Nat16 w );
Extern Nat16	random( Void );
Extern Void	calcArgosChksums( ArgosUnion *dp, Nat16 numBufs );
Extern char     *find_str( char *src, char *tgt );
Extern Int16    getByte( char *p, Nat16 radix );

/********************************/
/*	Module Local Data	*/
/********************************/

MLocal ArgosUnion	newData;	/* Data we're still collecting	*/
MLocal ArgosUnion	sendData;	/* Data we're sending		*/

MLocal Nat16		bufToSend;	/* Buffer to send		*/
MLocal Word		prrState;	/* Bits 0-7 for Wetstar, 13-15 PRR*/
MLocal Word		analogState;
MLocal Word		co2State;

MLocal const Byte	sendOrder[] = {0,5,1,6,2,7,3,8,4,9};


/************************************************************************/
/* Function    : copyArgosBufs						*/
/* Purpose     : Copy newData into sendData buffer, calc chksums	*/
/* Inputs      : None							*/
/* Outputs     : None							*/
/************************************************************************/
	Void
copyArgosBufs( Void )
{
    bcopy( (Byte *)&newData, (Byte *)&sendData, sizeof(ArgosUnion) );
    calcArgosChksums( &sendData, NUM_ARGOS_BUFS );

} /* copyArgosBufs() */


/************************************************************************/
/* Function    : swapArgosBufs						*/
/* Purpose     : Copy newData into sendData buffer, clear old data	*/
/* Inputs      : None							*/
/* Outputs     : None							*/
/************************************************************************/
	Void
swapArgosBufs( Void )
{
    copyArgosBufs();
    bzero( (void *)&newData, sizeof(ArgosUnion) );
    bufToSend = 0;
    prrState = 0;
    analogState = 0;
    co2State = 0;

} /* swapArgosBufs() */


/************************************************************************/
/* Function    : wakeup_argos						*/
/* Purpose     : Wake up ARGOS PTT					*/ 
/* Inputs      : None							*/
/* Outputs     : None							*/
/************************************************************************/
	Void
wakeup_argos( Void )
{
    io_wakebits |= ARGOS_BIT;			/* Wake up the PTT	*/
    wakeport = io_wakebits;			/* Send to hardware	*/
    task_delay( 4 );				/* Leave it on 30-40 ms	*/
    io_wakebits &= ~ARGOS_BIT;			/* Turn off wakeup bit	*/
    wakeport = io_wakebits;			/* Send to hardware	*/
    xflush_ser( 1 );				/* Throw away any input	 */

} /* wakeup_argos() */


/************************************************************************/
/* Function    : argos_wake						*/
/* Purpose     : ARGOS serial wakeup function				*/ 
/* Inputs      : Driver ptr, Boolean (TRUE for on, FALSE for off)	*/
/* Outputs     : TRUE							*/
/************************************************************************/
	MBool
argos_wake( Driver *dp, MBool on, char *buffer )
{
    if ( on )
	wakeup_argos();

    return( TRUE );

} /* argos_wake() */


/************************************************************************/
/* Function    : send_ptt						*/
/* Purpose     : Wake up PTT, send one buffer				*/
/* Inputs      : Driver ptr						*/
/* Outputs     : TRUE if successful, else FALSE				*/
/************************************************************************/
	MBool
send_ptt( Driver *dp )
{
    Reg Int16	ch;				/* Scratch char buffer	*/
    Reg Nat16	argosId, numArgosIds;

    wakeup_argos();				/* Wake up argos PTT	*/

    argosId = dp->drv_parms[PARM1];
    numArgosIds = dp->drv_parms[PARM2];

    if ( numArgosIds > 1 )
	argosId += (bufToSend % numArgosIds);

    xputc( ST_XMIT1_CMD | argosId );
    xputn( (char *)&(sendData.arg_data[bufToSend]), ARGOS_LEN );

    ch = xgetc_tmout( dp->drv_parms[TIMEOUT] );	/* Get response		 */
    return( ch == ACK );			/* Check for ACK	 */

} /* send_ptt() */


/************************************************************************/
/* Function    : argos_drv						*/
/* Purpose     : Driver to send data to Argos PTT			*/
/* Inputs      : Driver Pointer						*/
/* Outputs     : None							*/
/************************************************************************/
	Void
argos_drv( Driver *dp )
{
    Nat16	i;

    if ( deploymentMode )
	copyArgosBufs();

    drv_ser_port( dp );			/* Setup serial port		*/

    for ( i = dp->drv_parms[PARM0]; (i > 0) && (!send_ptt(dp)); i-- )
	delay_secs( dp->drv_parms[TIMEOUT] );

    drv_ser_release( dp );		/* Release serial port		*/

    if ( i <= 0 )			/* If retried out, set error	*/
	error |= ARGOS_ERR;

    bufToSend++;
    bufToSend %= NUM_ARGOS_BUFS;
    
    dp->drv_wakeup += ((Int16)(random()/RANDOM_DIVISOR) - 10);

} /* argos_drv() */


/************************************************************************/
/* Function    : sample8Num						*/
/* Purpose     : Determine sample number for items sampled 8 times/day	*/
/* Inputs      : None							*/
/* Outputs     : Sample number						*/
/* Comment     : Sample 0 is at midnight local, sample 1 at 0200, etc	*/
/************************************************************************/
	Nat16
sample8Num( Void )
{
    return( (localHour/3) & 7 );

} /* sample8Num() */


/************************************************************************/
/* Function    : argos12bitSample					*/
/* Purpose     : Put 12 bit sample into byte array			*/
/* Inputs      : Data, sample state vector, byte array ptr		*/
/* Outputs     : None							*/
/************************************************************************/
	Void
argos12bitSample( Nat16 sample, Word *sampleState, Byte *dp )
{
    Reg Byte	*dataPtr;
    Reg Nat16	sampleNum;

    sampleNum = sample8Num();

    if ( *sampleState & (1 << sampleNum) )		/* 1 bit means sample*/
	return;						/*  already taken    */

    *sampleState |= (1 << sampleNum);			/* Show sample taken */

    dataPtr = dp + sampleNum + (sampleNum >> 1);	/* Point to entry    */

    if ( sampleNum & 1 )
    {
	*dataPtr++ = (*dataPtr & 0xf0) | ((sample >> 8) & 0x0f);
	*dataPtr = (Byte)sample;
    }
    else
    {
	*dataPtr++ = (Byte)(sample >> 4);
	*dataPtr = (*dataPtr & 0x0f) | ((sample << 4) & 0xf0);
    }

} /* argos12bitSample() */


/************************************************************************/
/* Function    : checkAnalogSample					*/
/* Purpose     : Check if Analog or CTD sample needs to be done		*/
/* Inputs      : Bit to check for					*/
/* Outputs     : TRUE to do sample, else FALSE				*/
/* Side Effect : Sets bit in analogState				*/
/************************************************************************/
	MBool
checkAnalogSample( Word stateBit )
{
    if ( (deploymentMode || (localHour >= 12)) &&
	 ((analogState & stateBit) == 0) )
    {
	analogState |= stateBit;
	return( TRUE );
    }

    return( FALSE );

} /* checkAnalogSample() */


/************************************************************************/
/* Function    : argosAnalogSample					*/
/* Purpose     : Put Analog sample into ARGOS buffer			*/
/* Inputs      : Driver pointer, A/D sample pointer			*/
/* Outputs     : None							*/
/************************************************************************/
	Void
argosAnalogSample( Driver *dp, Nat16 *ad_sample )
{
    Reg Nat16	sampleNum;

    switch( dp->drv_parms[SAMPLE_TYPE] )
    {
      case FLUOR:
	argos12bitSample( (*ad_sample << 2), &analogState,
			  newData.arg_struct.misc.fl_0m );
	break;
	
      case OASIS_CAN:
	if ( checkAnalogSample(OASIS_ANALOG_DONE) )
	{
#ifdef FLOW_SENSOR
	    newData.arg_struct.shutter.sh_flow = ad_sample[0];
	    newData.arg_struct.misc.ms_oasis = 
		(ad_sample[1] & 0x3ff) | ((Nat32)(ad_sample[2] & 0x3ff) << 10)
		    | ((Nat32)(ad_sample[5] & 0x3ff) << 20);
#else
	    newData.arg_struct.misc.ms_oasis = 
		(ad_sample[0] & 0x3ff) | ((Nat32)(ad_sample[1] & 0x3ff) << 10)
		    | ((Nat32)(ad_sample[4] & 0x3ff) << 20);
#endif
	}
	break;
    }

} /* argosAnalogSample() */


/************************************************************************/
/* Function    : argosCo2Sample						*/
/* Purpose     : Put pCO2 sample into ARGOS buffer			*/
/* Inputs      : Sample, Boolean to denote calibration			*/
/* Outputs     : None							*/
/************************************************************************/
#ifdef ARGOS_CO2
	Void
argosCo2Sample( Int16 sample, MBool calibr )
{
    if ( calibr )
	newData.arg_struct.misc.ms_pco2 = 
	    ((sample8Num() << 12) | (sample & 0xfff));
    else
	argos12bitSample( sample, &co2State,
			  newData.arg_struct.chem.ch_pco2 );

} /* argosCo2Sample() */
#endif/*ARGOS_CO2*/

/************************************************************************/
/* Function    : checkPrrSample						*/
/* Purpose     : Check if PRR sample needs to be done			*/
/* Inputs      : Sample time in minutes since midnight, bit to check for*/
/* Outputs     : TRUE to do sample, else FALSE				*/
/* Side Effect : Sets bit in prrState					*/
/************************************************************************/
#ifdef ARGOS_SPECPRR
	MBool
checkPrrSample( Nat16 minTime, Word stateBit )
{
    if ( (deploymentMode || ((60 * localHour) + dttm.dt_min >= minTime)) && 
	 ((prrState & stateBit) == 0) )
    {
	prrState |= stateBit;
	return( TRUE );
    }

    return( FALSE );

} /* checkPrrSample() */


/************************************************************************/
/* Function    : argosPrrSample						*/
/* Purpose     : Put PRR Spectro sample into ARGOS buffer, if time	*/
/* Inputs      : Ptr to PRR data, number of address tags		*/
/* Outputs     : None							*/
/************************************************************************/
	Void
argosPrrSample( PrrData *prrdp, Nat16 totChans, Nat16 errs )
{
    Reg Nat16	wetstar20m, sampleNum;

    newData.arg_struct.spec0m.sp_misc += errs;
#if (ARGOS_REV==4)
    newData.arg_struct.spec0m_1230.sp_misc += totChans;
#endif
#if (ARGOS_REV==5)
    newData.arg_struct.spec0m_1000.sp_misc += totChans;
#endif

    wetstar20m = prrdp[2].pd_data[11];
    
    if ( checkPrrSample(0, PRR_MIDNITE_DONE) )
    {				/* Time to send midnight sample		*/

	bcopy( (Byte *)&prrdp[0].pd_data[0],
	       (Byte *)&newData.arg_struct.spec0mDark.sp_ed[0],
	       SPEC_CHANS * sizeof(Int16) );
	bcopy( (Byte *)&prrdp[1].pd_data[0],
	       (Byte *)&newData.arg_struct.spec0mDark.sp_lu[0],
	       SPEC_CHANS * sizeof(Int16) );
	bcopy( (Byte *)&prrdp[1].pd_data[8],
	       (Byte *)&newData.arg_struct.shutter.sh_sat1Dark[0],
	       SPEC_CHANS * sizeof(Int16) );

	bcopy( (Byte *)&prrdp[2].pd_data[0],
	       (Byte *)&newData.arg_struct.spec20mDark.sp_ed[0],
	       SPEC_CHANS * sizeof(Int16) );
	bcopy( (Byte *)&prrdp[3].pd_data[0],
	       (Byte *)&newData.arg_struct.spec20mDark.sp_lu[0],
	       SPEC_CHANS * sizeof(Int16) );
	newData.arg_struct.misc.fl_20m[0] = wetstar20m;
    }

    else if ( checkPrrSample(720, PRR_NOON_DONE) )
    {				/* Time to send 12 noon sample		*/

	bcopy( (Byte *)&prrdp[0].pd_data[0],
	       (Byte *)&newData.arg_struct.spec0m.sp_ed[0],
	       SPEC_CHANS * sizeof(Int16) );
	bcopy( (Byte *)&prrdp[1].pd_data[0],
	       (Byte *)&newData.arg_struct.spec0m.sp_lu[0],
	       SPEC_CHANS * sizeof(Int16) );
	bcopy( (Byte *)&prrdp[1].pd_data[8],
	       (Byte *)&newData.arg_struct.sat1.sp_ed[0],
	       SPEC_CHANS * sizeof(Int16) );

	bcopy( (Byte *)&prrdp[2].pd_data[0],
	       (Byte *)&newData.arg_struct.spec20m.sp_ed[0],
	       SPEC_CHANS * sizeof(Int16) );
	bcopy( (Byte *)&prrdp[3].pd_data[0],
	       (Byte *)&newData.arg_struct.spec20m.sp_lu[0],
	       SPEC_CHANS * sizeof(Int16) );

	newData.arg_struct.spec20m.sp_misc = prrdp[2].pd_data[10];
#if (ARGOS_REV==4)
	newData.arg_struct.spec20m_1230.sp_misc = prrdp[2].pd_data[14];
	newData.arg_struct.chem.ch_prr_pitch = prrdp[0].pd_data[10];
	newData.arg_struct.chem.ch_prr_roll  = prrdp[0].pd_data[11];
#endif
#if (ARGOS_REV==5)
	newData.arg_struct.spec20m_1000.sp_misc = prrdp[2].pd_data[14];
#endif
	newData.arg_struct.chem.ch_prr_temp  = prrdp[2].pd_data[8];
	newData.arg_struct.chem.ch_prr_depth = prrdp[2].pd_data[9];
    }

#if (ARGOS_REV==4)
    else if ( checkPrrSample(745, PRR_1230_DONE) )
    {				/* Time to send 12:30 PM sample		*/

	bcopy( (Byte *)&prrdp[0].pd_data[0],
	       (Byte *)&newData.arg_struct.spec0m_1230.sp_ed[0],
	       SPEC_CHANS * sizeof(Int16) );
	bcopy( (Byte *)&prrdp[1].pd_data[0],
	       (Byte *)&newData.arg_struct.spec0m_1230.sp_lu[0],
	       SPEC_CHANS * sizeof(Int16) );
	bcopy( (Byte *)&prrdp[1].pd_data[8],
	       (Byte *)&newData.arg_struct.sat1.sp_lu[0],
	       SPEC_CHANS * sizeof(Int16) );

	bcopy( (Byte *)&prrdp[2].pd_data[0],
	       (Byte *)&newData.arg_struct.spec20m_1230.sp_ed[0],
	       SPEC_CHANS * sizeof(Int16) );
	bcopy( (Byte *)&prrdp[3].pd_data[0],
	       (Byte *)&newData.arg_struct.spec20m_1230.sp_lu[0],
	       SPEC_CHANS * sizeof(Int16) );

	newData.arg_struct.chem.ch_prr_pitch1230 = prrdp[0].pd_data[10];
	newData.arg_struct.chem.ch_prr_roll1230  = prrdp[0].pd_data[11];
	newData.arg_struct.chem.ch_prr_temp1230  = prrdp[2].pd_data[8];
	newData.arg_struct.chem.ch_prr_depth1230 = prrdp[2].pd_data[9];
    }
#endif
#if (ARGOS_REV==5)
    else if ( checkPrrSample(595, PRR_1000_DONE) )
    {				/* Time to send 1000 AM sample		*/

	bcopy( (Byte *)&prrdp[0].pd_data[0],
	       (Byte *)&newData.arg_struct.spec0m_1000.sp_ed[0],
	       SPEC_CHANS * sizeof(Int16) );
	/*This should no longer be done in REV5
	bcopy( (Byte *)&prrdp[1].pd_data[0],
	       (Byte *)&newData.arg_struct.spec0m_1000.sp_lu[0],
	       SPEC_CHANS * sizeof(Int16) );
	*/
	bcopy( (Byte *)&prrdp[1].pd_data[8],
	       (Byte *)&newData.arg_struct.sat1.sp_lu[0],
	       SPEC_CHANS * sizeof(Int16) );

	bcopy( (Byte *)&prrdp[2].pd_data[0],
	       (Byte *)&newData.arg_struct.spec20m_1000.sp_ed[0],
	       SPEC_CHANS * sizeof(Int16) );
	bcopy( (Byte *)&prrdp[3].pd_data[0],
	       (Byte *)&newData.arg_struct.spec20m_1000.sp_lu[0],
	       SPEC_CHANS * sizeof(Int16) );

    }
#endif
    else if ( checkPrrSample(1065, WETSTAR_1745) )
	newData.arg_struct.misc.fl_20m[FLUOR20_WORDS-1] = wetstar20m;
    else
    {
	sampleNum = sample8Num();
	if ( sampleNum && ((prrState & (1 << sampleNum)) == 0) )
	{
	    newData.arg_struct.misc.fl_20m[sampleNum-1] = wetstar20m;
	    prrState |= (1 << sampleNum);
	}
    }
    
} /* argosPrrSample() */
#endif /*ARGOS_PRR*/

/************************************************************************/
/* Function    : argosOasisError					*/
/* Purpose     : OR oasis error into argos message			*/
/* Inputs      : None							*/
/* Outputs     : None							*/
/************************************************************************/
	Void
argosOasisError( Void )
{
    newData.arg_struct.shutter.sh_oasisErrs |= error;

} /* argosOasisError() */


/************************************************************************/
/* Function    : argosShutter						*/
/* Purpose     : Save shutter information in ARGOS buffer		*/
/* Inputs      : Shutter_data ptr					*/
/* Outputs     : None							*/
/************************************************************************/
#ifdef ARGOS_SHUTTER
	Void
argosShutter( Shutter_data *sdp )
{
    Reg Nat16	errType;
    
    newData.arg_struct.shutter.sh_attempts += sdp->TryCount;

    if ( sdp->Open )
	newData.arg_struct.shutter.sh_opens++;
    else
	newData.arg_struct.shutter.sh_closes++;
	
    if ( sdp->ErrCode )
    {
	newData.arg_struct.shutter.sh_toterrs++;

	errType = sdp->ErrCode - 1;
	if ( errType > 2 )
	    errType = 2;

	newData.arg_struct.shutter.sh_errs[errType]++;
    }
    
} /* argosShutter() */
#endif /*ARGOS_SHUTTER*/

/************************************************************************/
/* Function    : argosCTDSample						*/
/* Purpose     : Put Noon CTD sample into ARGOS buffer			*/
/* Inputs      : CTD Temperature, conductivity				*/
/* Outputs     : None							*/
/************************************************************************/
#ifdef ARGOS_CTD
	Void
argosCTDSample( Nat16 temp, Nat16 conduct )
{
    if ( checkAnalogSample(CTD_NOON_DONE) )
    {
	newData.arg_struct.shutter.sh_ctd_temp = temp;
	newData.arg_struct.shutter.sh_ctd_cond = conduct;
    }

} /* argosCTDSample() */
#endif/*ARGOS_CTD*/

/************************************************************************/
/* Function    : argosHS2Sample						*/
/* Purpose     : Put Hobi HS2 sample into ARGOS buffer			*/
/* Inputs      : Driver ptr, buffer len, Data ptr (ASCII)		*/
/* Outputs     : None							*/
/************************************************************************/
#ifdef ARGOS_HS2
	Void
argosHS2Sample( Driver *dp, Nat16 len, Byte *data )
{
  char *ip;
  Byte hs2buf[HS2BUF_SIZE];
  Int16 b;

  /* look for error messages */
  if((ip=find_str((char *)data,"SYNC_ERR"))==NULL){
    bcopy( (Byte *)&hs2buf[0],
	 (Byte *)"SYNC",
	 4 );
    return;
  }

  /* Find first 'D' record */
  if((ip=find_str((char *)data,"*D"))==NULL)
    return;/* data not found */
  ip -= 2;

  /* check record size */
  if( strlen(ip) < HS2_RECORD_SIZE )
     return;/* incomplete record */

  /* convert ASCII to hex */
  bzero(hs2buf,HS2BUF_SIZE);
  hs2buf[BOS_SNORM1]= (Byte)getByte(&ip[AOS_SNORM1],16);
  hs2buf[BOS_SNORM1+1]= (Byte)getByte(&ip[AOS_SNORM1+2],16);
  hs2buf[BOS_SNORM2]= (Byte)getByte(&ip[AOS_SNORM2],16);
  hs2buf[BOS_SNORM2+1]= (Byte)getByte(&ip[AOS_SNORM2+2],16);
  hs2buf[BOS_SNORM3]= (Byte)getByte(&ip[AOS_SNORM3],16);
  hs2buf[BOS_SNORM3+1]= (Byte)getByte(&ip[AOS_SNORM3+2],16);
  hs2buf[BOS_GAIN1]= (Byte)getByte(&ip[AOS_GAIN1],16);
  hs2buf[BOS_GAIN2]=  (hs2buf[BOS_GAIN1]<<4)+(Byte)getByte(&ip[AOS_GAIN2],16);
  hs2buf[BOS_GAIN3]= (Byte)getByte(&ip[AOS_GAIN3],16);
  hs2buf[BOS_TEMPRAW]= (Byte)getByte(&ip[AOS_TEMPRAW],16);

  /* store in argos buffer */
   bcopy( (Byte *)&hs2buf[0],
	 (Byte *)&newData.arg_struct.sat1.sp_lu[0],
	 SPEC_CHANS*sizeof(Int16) );

  return;
} /* argosHS2Sample() */
#endif /*ARGOS_HS2*/

/************************************************************************/
/* Function    : argosIsusSample	       				*/
/* Purpose     : Put Isus NO3 sample into ARGOS buffer			*/
/* Inputs      : Driver ptr, Number of samples, Data ptr		*/
/* Outputs     : None							*/
/************************************************************************/
#ifdef ARGOS_ISUS
	Void
argosIsusSample( Driver *dp, Nat16 len, Byte *data )
{
  Nat16 sz[2];
  Int16 ck1sz,ck2sz;

  if ( (dp->drv_parms[SAMPLE_TYPE] != ISUS) || (len <= 0))
	return;

  sz[0]=SPEC_CHANS*sizeof(Int16);
  sz[1]=CH_UNUSED_SIZE*sizeof(Int16);

  ck1sz=(len<=sz[0])?len:sz[0];
  ck2sz=( (len-ck1sz)<=sz[1])?(len-ck1sz):sz[1];
  
  if(ck1sz>0)
  bcopy( (Byte *)&data[0],
	 (Byte *)&newData.arg_struct.spec0m_1000.sp_lu[0],
	 ck1sz );

  if(ck2sz>0)
  bcopy( (Byte *)&data[ck1sz],
	 (Byte *)&newData.arg_struct.chem.ch_unused[0],
	 ck2sz );
   return;
  
} /* argosIsusSample() */
#endif /*ARGOS_ISUS*/


/************************************************************************/
/* Function    : argosNo3Sample						*/
/* Purpose     : Put NO3 sample into ARGOS buffer			*/
/* Inputs      : Driver ptr, Number of samples, Data ptr		*/
/* Outputs     : None							*/
/************************************************************************/
#ifdef ARGOS_OSMO
/* From deployed/m3/argos.c

	Void
argosNo3Sample( Driver *dp, Nat16 numSamples, NO3Sample *data )
{
    Reg No3Msg	*no3Ptr;
    Reg Nat16	sample, sampleNum;

    if ( dp->drv_parms[SAMPLE_TYPE] != NO3 )
	return;

    for ( sample = 0; sample < numSamples; sample++ )
    {
	sampleNum = (sample + ((localHour + 1) % 24)/3) & 7;

	no3Ptr->temp[sampleNum] = 
	    (Byte)((data[sample].temp+2)/5 - NO3_TEMP_OFFSET);

	argos12bitSample(data[sample].no3, sampleNum,
			 newData.arg_struct.chem.ch_no3.no3);
    }

}*/ /* argosNo3Sample() */
#endif /*ARGOS_OSMO*/








