/****************************************************************************/
/* Copyright 1995 MBARI                                                     */
/****************************************************************************/
/* $Header: adcp.c,v 4.4 2001/06/19 12:12:01 oasisa Exp $			    */
/* Summary  : Driver Routines for ADCP on OASIS mooring			    */
/* Filename : adcp.c							    */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 4.4 $							    */
/* Created  : 02/15/95 from sensors.c					    */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* 15feb95 rah - created from sensors.c					    */
/* $Log:	adcp.c,v $
 * Revision 4.4  2001/06/19  12:12:01  12:12:01  oasisa (Oasis users)
 * New Repository; 6/19/2001 (klh)
 * 
 * Revision 1.1  2001/06/19  11:42:18  11:42:18  oasisa (Oasis users)
 * Initial revision
 * 
 * Revision 4.3  99/06/16  10:21:33  10:21:33  bobh (Bob Herlien)
 * Mar/May '99 Deployments of M3/M2
 * 
 * Revision 4.2  98/09/10  10:11:48  10:11:48  bobh (Bob Herlien)
 * Sept/Oct '98 deployments of M1, Eqpac 1 & 2
 * 
 * Revision 3.8  97/09/12  10:50:51  10:50:51  bobh (Bob Herlien)
 * Redeploy M1
 * 
 * Revision 3.7  97/07/23  11:18:08  11:18:08  bobh (Bob Herlien)
 * July '97 M1 deployment, new shutter code
 * 
 * Revision 3.4  96/06/18  15:24:22  15:24:22  bobh (Bob Herlien)
 * June '96 deployment of M1
 * 
 * Revision 3.1  95/03/09  19:30:57  19:30:57  hebo (Bob Herlien)
 * March '95 Deployment of M1A
 * 
 * Revision 3.0  95/02/21  18:42:44  18:42:44  hebo (Bob Herlien)
 * February '95 Deployment
 * 
*/
/****************************************************************************/

#include <types.h>			/* MBARI type definitions	    */
#include <const.h>			/* MBARI constants		    */
#include <oasis.h>			/* OASIS controller definitions	    */
#include <io.h>				/* OASIS I/O definitions	    */
#include <log.h>			/* Log record definitions	    */
#include <custom.h>			/* ADCP_TYPE definition		    */
#include <80C196.h>			/* 80196 Register mapping           */
#include <task.h>			/* OASIS Multitasking definitions   */
#include <stdio.h>			/* Standard I/O functions	    */

typedef struct				/************************************/
{					/* AdcpHdr - Header for ADCP record */
    Nat16	adcp_id;		/* Hdr and Data ID fields	    */
    Nat16	adcp_len;		/* Length of Record (not incl chksum*/
} AdcpHdr;				/************************************/

typedef Nat16	ChkSum;			/* ADCP checksum		    */

#define ADCP_ID	0x7f7f


/********************************/
/*	External Functions	*/
/********************************/

Extern Void	drvLog( Driver *dp, Byte *samplep, Int16 len );
Extern Void	drvLogError( Driver *dp, Errno err );
Extern Void	drv_ser_port( Driver *dp );
Extern Void	drv_ser_release( Driver *dp );
Extern Void	xputc( Int16 c );
Extern Void	xputs( const char *s );
Extern Int16	xgetn_tmout( char *s, Int16 len, Nat16 tmout );
Extern Void	bcopy( const Byte *src, Byte *dst, Nat16 len );
Extern char	*tmpMalloc( Nat16 size );
Extern Void	tmpFree( char *ptr );
Extern Int16	swap( Int16 w );


/********************************/
/*	External Data		*/
/********************************/

Extern Reg TimeOfDay	tod;		/* Current time in TimeOfDay format */
Extern Nat16		adcp_dstrsz;	/* Size of adcp_dstr		    */
Extern char		*adcp_dstr;	/* Discon string for adcp	    */


/************************************************************************/
/* Function    : adcp_wake						*/
/* Purpose     : ADCP serial wakeup function				*/ 
/* Inputs      : Driver ptr, Boolean (TRUE for on, FALSE for off)	*/
/* Outputs     : TRUE							*/
/* Comments    : At turnoff time, outputs adcp_dstr in binary mode	*/
/*		 Adcp_dstr typically is the string that starts sampling */
/************************************************************************/
	MBool
adcp_wake( Driver *dp, MBool on )
{
    Reg Byte	*p;
    WordByte	chksum;

    if ( !on )
    {
	chksum.wb_word = 0;

	for ( p = (Byte *)adcp_dstr; *p; p++ )
	    chksum.wb_word += (Nat16)(*p);

	xputs( adcp_dstr );
	xputc( chksum.wb_byte[1] );
	xputc( chksum.wb_byte[0] );
    }

    return( TRUE );

} /* adcp_wake() */


/************************************************************************/
/* Function    : adcp_drv						*/
/* Purpose     : ADCP driver						*/
/* Inputs      : Driver Pointer						*/
/* Outputs     : None							*/
/* Comments    : Assumes that ADCP is in binary data mode!		*/
/************************************************************************/
	Void
adcp_drv( Driver *dp )
{
    char	*bufp;
    Reg Nat16	tmout;
    Reg Nat16	len;
    Nat16	xmitLen;
    Errno	rtn;
    AdcpHdr	header;

    drv_ser_port( dp );			/* Setup serial port		*/
    tmout = (dp->drv_flags & DO_SYNC) ?	/* Get timeout			*/
		dp->drv_parms[PARM1] : dp->drv_parms[TIMEOUT];
    dp->drv_flags &= ~(DO_SYNC | FLAG_PERM);
    rtn = NO_DATA;

    if ( xgetn_tmout((char *)&header, sizeof(header), tmout) == sizeof(header) )
    {					/* Got size field. Save it.	*/
	xmitLen = (header.adcp_id == ADCP_ID) ?
		  header.adcp_len : swap(header.adcp_id);

	rtn = (xmitLen > MAXADCPSIZE) ? SYNC_ERR : SUCCESS;

	xmitLen += sizeof(ChkSum);
	
	if ( rtn == SUCCESS )
	    if ( (bufp = tmpMalloc(xmitLen)) == NULL )
		rtn = MALLOC_ERR;

	if ( rtn == SUCCESS )
	{					/* Sync wakeup to now	*/
	    dp->drv_wakeup = tod + dp->drv_parms[INTERVAL] -
			     (dp->drv_parms[TIMEOUT] >> 1);
	    bcopy( (Byte *)&header, (Byte *)bufp, sizeof(header) );
	    len = xgetn_tmout(bufp + sizeof(header), xmitLen - sizeof(header),
			      dp->drv_parms[PARM0]);
	    if ( len == (xmitLen - sizeof(header)) )
		drvLog( dp, (Byte *)bufp, xmitLen );
	    else
		rtn = TMOUT_ERR;
	    tmpFree( bufp );
	}
    }
    else if ( ++dp->drv_cnt >= dp->drv_parms[PARM2] )
    {				/* Didn't get data.  Try resync every 24 hrs*/
	dp->drv_flags |= (DO_SYNC | FLAG_PERM);	/* If resync, set sync flag */
	dp->drv_cnt = 0;			/* Reset sync counter	    */
    }

    if ( rtn != SUCCESS )			/* If error, 		    */
	drvLogError( dp, rtn );			/*   log the error	    */

    drv_ser_release( dp );			/* Release serial port	    */

} /* adcp_drv() */

