/****************************************************************************/
/* Copyright 1996 MBARI                                                     */
/****************************************************************************/
/* $Header: ac9.c,v 4.4 2001/06/19 12:10:58 oasisa Exp $			    */
/* Summary  : Driver Routines for AC-9 transmittance/absorbance meter	    */
/* Filename : ac9.c							    */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 4.4 $							    */
/* Created  : 04/22/96 from adcp.c					    */
/*									    */
/* MBARI provides this documentation and code "as is", with no warranty,    */
/* express or implied, of its quality or consistency. It is provided without*/
/* support and without obligation on the part of the Monterey Bay Aquarium  */
/* Research Institute to assist in its use, correction, modification, or    */
/* enhancement. This information should not be published or distributed to  */
/* third parties without specific written permission from MBARI.            */
/*									    */
/****************************************************************************/
/* Modification History:						    */
/* 22apr96 rah - created from adcp.c					    */
/* $Log:	ac9.c,v $
 * Revision 4.4  2001/06/19  12:10:58  12:10:58  oasisa (Oasis users)
 * New Repository; 6/19/2001 (klh)
 * 
 * Revision 1.1  2001/06/19  11:40:55  11:40:55  oasisa (Oasis users)
 * Initial revision
 * 
 * Revision 3.7  97/07/23  11:18:18  11:18:18  bobh (Bob Herlien)
 * July '97 M1 deployment, new shutter code
 * 
 * Revision 3.5  96/07/17  13:01:37  13:01:37  bobh (Bob Herlien)
 * July '96 deployment of M2 with ARGOS code
 * 
 * Revision 3.4  96/06/18  15:24:31  15:24:31  bobh (Bob Herlien)
 * June '96 deployment of M1
 * 
*/
/****************************************************************************/

#include <types.h>			/* MBARI type definitions	    */
#include <const.h>			/* MBARI constants		    */
#include <oasis.h>			/* OASIS controller definitions	    */
#include <io.h>				/* OASIS I/O definitions	    */
#include <log.h>			/* Log record definitions	    */
#include <task.h>			/* OASIS Multitasking definitions   */
#include <custom.h>			/* ARGOS definition		    */
#include <ac9.h>			/* AC-9 definitions		    */
#ifdef ARGOS
#include <argos.h>			/* ARGOS buffer definitions	    */
#endif


/********************************/
/*	External Functions	*/
/********************************/

Extern Void	drvLog( Driver *dp, Byte *samplep, Int16 len );
Extern Void	drvLogError( Driver *dp, Errno err );
Extern char	*drvSerPortAndMalloc( Driver *dp, Nat16 size );
Extern Void	drv_ser_release( Driver *dp );
Extern Void	drv_pwroff( Driver *dp );
Extern Int16	xgetc_tmout( Nat16 tmout  );
Extern Int16	xgetn_tmout( char *s, Int16 len, Nat16 tmout );
Extern Void	xputs( const char *s );
Extern Void	xflush_ser( Nat16 tmout  );
Extern Void	tmpFree( char *ptr );
Extern Void	bzero( void *s, int n );
Extern Nat16	ac9_chksum( Byte *p, Nat16 len );
Extern Void	process_ac9_pkt( Ac9Raw *bufp, Ac9Data *datap );

#ifdef ARGOS_AC9
Extern Void	argosAc9Sample( Ac9Result *ac9ResP );
Void		calcAc9Argos( Ac9Data *ac9p, Ac9Result *resP );
#endif
#ifdef PUMP_SYNC_AC9
Extern Void	drvSync( char *name );
#endif


/********************************/
/*	External Data		*/
/********************************/

Extern volatile Reg Nat16	tick;		/* 10 ms ticker		*/


/************************************************************************/
/* Function    : ac9_wake						*/
/* Purpose     : AC-9 serial wakeup function				*/ 
/* Inputs      : Driver ptr, Boolean (TRUE for on, FALSE for off)	*/
/* Outputs     : TRUE							*/
/************************************************************************/
	MBool
ac9_wake( Driver *dp, MBool on )
{
    if ( on )
	xputs( "!!!!!" );

    return( TRUE );

} /* ac9_wake() */


/************************************************************************/
/* Function    : syncAc9						*/
/* Purpose     : Get start of packet sync code for AC-9			*/
/* Inputs      : Timeout in seconds					*/
/* Outputs     : OK or ERROR						*/
/************************************************************************/
	Errno
syncAc9( Nat16 tmout )
{
    Nat16	startTick, syncBytes;
    Reg Int16	ch, curTmout;

    startTick = tick;

    for ( syncBytes = 0; syncBytes < 4; )
    {
	if ( (curTmout = tmout - ((tick - startTick)/TICKS_PER_SECOND)) <= 0 )
	    return( ERROR );

	if ( (ch = xgetc_tmout(curTmout)) == ERROR )
	    return( ERROR );

	if ( ch == ((syncBytes & 1) ? 0xff : 0) )
	    syncBytes++;
	else if ( ch == 0 )
	    syncBytes = 1;
	else
	    syncBytes = 0;
    }

    return( OK );

} /* syncAc9() */


/************************************************************************/
/* Function    : getAc9Packet						*/
/* Purpose     : Get one AC-9 packet					*/
/* Inputs      : Driver Pointer						*/
/* Outputs     : OK or Error code					*/
/************************************************************************/
	Errno
getAc9Packet( Driver *dp, Ac9Raw *pktp )
{
    if ( syncAc9(dp->drv_parms[TIMEOUT]) != OK )
	return( SYNC_ERR );

    if ( xgetn_tmout((char *)pktp, sizeof(Ac9Raw), dp->drv_parms[TIMEOUT])
	 != sizeof(Ac9Raw) )
	return( TMOUT_ERR );

    if ( pktp->ac9_chksum !=
	 ac9_chksum((Byte *)pktp, sizeof(Ac9Raw) - sizeof(Nat16)) )
	return( CKSUM_ERR );

    return( OK );

} /* getAc9Packet() */


/************************************************************************/
/* Function    : ac9_drv						*/
/* Purpose     : AC-9 driver						*/
/* Inputs      : Driver Pointer						*/
/* Outputs     : None							*/
/************************************************************************/
	Void
ac9_drv( Driver *dp )
{
    Ac9Raw	*ac9_bufp;
    Ac9Data	*ac9_datap;
    Reg Errno	rtn;
    Nat16	cnt, errors;

    if ( (ac9_datap = 
	  (Ac9Data *)drvSerPortAndMalloc(dp, AC9_BUFLEN + sizeof(Ac9Data))) 
	 == (Ac9Data *)NULL )
	return;

    bzero( (void *)ac9_datap, AC9_BUFLEN + sizeof(Ac9Data) );
    ac9_bufp = (Ac9Raw *)(ac9_datap + 1);

#ifdef PUMP_SYNC_AC9
    drvSync( PUMP_SYNC_AC9 );		/* Wait for water pump		*/
#endif
    xflush_ser( dp->drv_parms[PARM2] * TICKS_PER_SECOND );
					/* Wait for warmup, toss serial */
    for ( cnt = errors = 0;
	 (cnt < dp->drv_parms[PARM0]) && (errors < dp->drv_parms[PARM1]); )
    {
	if ( (rtn = (Byte)getAc9Packet(dp, ac9_bufp)) == OK )
	{
	    process_ac9_pkt( ac9_bufp, ac9_datap );
	    cnt++;
	}
	else if ( rtn == SYNC_ERR )
	    break;
    }
    
    drv_pwroff( dp );			/* Turn off ac9 power		*/
    drv_ser_release( dp );		/* Release serial port		*/

    ac9_datap->ac9_samples = cnt;	/* Save sample count		*/

    if ( cnt > 0 )
	drvLog( dp, (Byte *)ac9_datap, sizeof(Ac9Data) );
    else
	drvLogError( dp, rtn );

#ifdef ARGOS_AC9
    calcAc9Argos( ac9_datap, (Ac9Result *)ac9_bufp );
    argosAc9Sample( (Ac9Result *)ac9_bufp );
#endif

    tmpFree( (char *)ac9_bufp );	/* Return buffer		*/

} /* ac9_drv() */


#ifdef ARGOS_AC9

/************************************************************************/
/* Function    : calcAc9Argos						*/
/* Purpose     : Calculate AC-9 data for ARGOS				*/
/* Inputs      : Ptr to AC-9 data, Ptr to area for result		*/
/* Outputs     : None							*/
/* Comment     : resP must point to area big enough to hold Ac9Result	*/
/*		 In practice, we send ptr to the raw buffer, 640 bytes	*/
/************************************************************************/
	Void
calcAc9Argos( Ac9Data *ac9p, Ac9Result *resP )
{
    Reg Nat16	chan;
    Reg Nat32	value;

    for ( chan = 0; chan < AC9_CHANS; chan++ )
    {
	value = (ac9p->ac9_data[chan]) / (ac9p->ac9_ref[chan]/100);
	resP->ac9_result[chan] =
	    (value > NAT16_MAX) ? NAT16_MAX : (Nat16)value;
    }

    resP->ac9_temp = (Nat16)(ac9p->ac9_temp / ac9p->ac9_samples);

} /* calcAc9Argos() */

#endif /* ARGOS_AC9 */
