/****************************************************************************/
/* Copyright 1994 MBARI                                                     */
/****************************************************************************/
/* $Header: range.c,v 2.5 94/12/15 10:59:39 hebo Exp $			    */
/* Summary  : Routines to handle Log Record Ranges			    */
/* Filename : range.c							    */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 2.5 $							    */
/* Created  : 03/30/94							    */
/****************************************************************************/
/* Modification History:						    */
/* 30mar94 rah - created						    */
/* $Log
*/
/****************************************************************************/

#include <stdio.h>			/* Standard I/O			    */
#include <mbari/types.h>		/* MBARI type definitions	    */
#include <mbari/const.h>		/* MBARI constants		    */
#include <decode.h>			/* OASIS controller definitions	    */
#include <string.h>			/* for strcmp(), memmove	    */

#define LBUFSIZE	512		/* Size of line buffer		    */


/********************************/
/*	Global Data		*/
/********************************/

Global char	*rcdfile = "/oasis/raw/oasis.rcd";


/********************************/
/*	Module Local Data	*/
/********************************/

MLocal char	lbuf[LBUFSIZE];		/* Line buffer			    */


/************************************************************************/
/* Function    : rcdInit						*/
/* Purpose     : Initialize LogStruct					*/
/* Inputs      : LogStruct ptr, number of LogStructs			*/
/* Outputs     : None							*/
/************************************************************************/
	Void
rcdInit( LogStruct *lp, Nat32 nlogs )
{
    memset( (void *)lp, 0, nlogs * sizeof(LogStruct) );

} /*rcdInit() */


/************************************************************************/
/* Function    : isCanRcd						*/
/* Purpose     : Check string to see if it's a record for a given can	*/
/* Inputs      : String ptr, Can name					*/
/* Outputs     : Log number (0-n) if can string matches, else ERROR	*/
/************************************************************************/
	Int32
isCanRcd( char *s, char *can )
{
    Reg Nat32	len;
    Nat32	lognum;
    
    len = strlen(can);
    if ( strncmp(s, can, len) != 0 )
	return( ERROR );

    if ( sscanf(s + len, ".%d", &lognum) == 1 )
	return( lognum );
    else
	return( ERROR );

} /* isCanRcd() */


/************************************************************************/
/* Function    : readRcd						*/
/* Purpose     : Read Record file, fill in array of LogStructs		*/
/* Inputs      : Can name, Ptr to array of LogStructs, num structs in array*/
/* Outputs     : TRUE if read file OK					*/
/************************************************************************/
	MBool
readRcd( char *can, LogStruct *lp, Nat32 nlogs )
{
    Reg FILE	*fp;
    Reg char	*p;
    Reg Nat32	range, i;
    Reg Int32	lognum;
    Nat32	min, max;
    
    if ( (fp = fopen(rcdfile, "r")) == (FILE *)NULL )
    {
	fprintf(stderr, "Cannot open record file %s\n", rcdfile);
	return( FALSE );
    }    

    while( fgets(lbuf, sizeof(lbuf), fp) != NULL )
    {
	if ( ((p = strtok(lbuf, ":")) == NULL) ||
	     ((lognum = isCanRcd(p, can)) == ERROR) || (lognum >= nlogs) )

	    continue;

	for ( range = 0; (p = strtok(NULL, ",")) != NULL; range < NRANGES )
	{
	    i = sscanf( p, "%d - %d", &min, &max );
	    if ( i == 1 )
		max = min;
	    if ( i >= 1 )
	    {
		lp[lognum].range[range].min = min;
		lp[lognum].range[range].max = max;
		range++;
	    }
	}
	lp[lognum].nRanges = range;
    }

    close( fp );
    return( TRUE );

} /* readRcd() */


/***************************************************************************/
/* Function    : writeRcd						   */
/* Purpose     : Write Record file, replace can strings with LogStruct info*/
/* Inputs      : Can name, Ptr to array of LogStructs, num structs in array*/
/* Outputs     : None							   */
/***************************************************************************/
	Void
writeRcd( char *can, LogStruct *lp, Nat32 nlogs )
{
    Reg FILE		*fp, *nfp;
    Reg char		*p, *name;
    Reg Int32		i, lognum;
    Reg LogStruct	*clp;

    if ( (fp = fopen(rcdfile, "r")) == (FILE *)NULL )
	fprintf(stderr, "Cannot open record file %s\n", rcdfile);

    strcpy( lbuf, rcdfile );
    if ( (p = strrchr(lbuf, '/')) != NULL )
	*p = '\0';
    else
	strcpy( lbuf, "." );

    if ( (name = tempnam(lbuf, NULL)) == NULL )
    {	
	fprintf(stderr, "Cannot get temp file name\n");
	return;
    }

    if ( (nfp = fopen(name, "w")) == (FILE *)NULL )
    {
	fprintf(stderr, "Cannot open temp file %s\n", name);
	return;
    }    
					/* Copy all rcd file that isn't	*/
    if ( fp != (FILE *)NULL )		/*  covered by current can	*/
	while( fgets(lbuf, sizeof(lbuf), fp) != NULL )
	    if (((lognum = isCanRcd(lbuf, can)) == ERROR) || (lognum >= nlogs))
		fwrite( lbuf, 1, strlen(lbuf), nfp );

				/* Now print current can records to rcd file*/
    for ( lognum = 0, clp = lp; lognum < nlogs; lognum++, clp++ )
	if ( clp->nRanges > 0 )
	{
	    fprintf( nfp, "%s.%d: ", can, lognum );
	    for( i = 0; i < clp->nRanges; i++ )
	    {
		if ( i > 0 )
		    fprintf( nfp, "," );
		fprintf( nfp, "%d", clp->range[i].min );

		if ( clp->range[i].min < clp->range[i].max )
		    fprintf( nfp, "-%d", clp->range[i].max );
	    }
	    fprintf( nfp, "\n" );
	}

    if ( fclose(nfp) == ERROR )
	fprintf( stderr, "Cannot close record file %s\n", name );
    else
    {
	unlink( rcdfile );
	rename( name, rcdfile );
    }

    free( name );

} /* writeRcd() */


/************************************************************************/
/* Function    : gotRcd							*/
/* Purpose     : Note received record number in LogStruct		*/
/* Inputs      : LogStruct, record number				*/
/* Outputs     : None							*/
/************************************************************************/
	Void
gotRcd( LogStruct *lp, Nat32 rcd )
{
    Reg Nat32	i;
    
    for ( i = 0; i < lp->nRanges; i++ )
    {
	if ( (rcd >= lp->range[i].min) && (rcd <= lp->range[i].max) )
	    return;				/* If already logged, rtn*/

	if ( rcd == (lp->range[i].min - 1) )
	{					/* If at low end of range*/
	    lp->range[i].min = rcd;		/* just note and return  */
	    return;
	}

	if ( rcd < lp->range[i].min )
	{					/* If between existing ranges*/
	    if ( lp->nRanges >= NRANGES )	/* create new range	  */
		return;
	    memmove( (void *)&lp->range[i+1], (void *)&lp->range[i],
		     (lp->nRanges - i) * sizeof(RcdRange) );
	    lp->nRanges++;
	    lp->range[i].min = lp->range[i].max = rcd;
	    return;
	}

	if ( rcd == (lp->range[i].max + 1) )
	{					/* If at hi end of range   */
	    lp->range[i].max = rcd;		/* note and return	   */
	    if (((i + 1) < lp->nRanges) && (rcd >= (lp->range[i+1].min - 1)))
	    {					/* Collapse two ranges	   */
		lp->range[i].max = lp->range[i+1].max;
		lp->nRanges--;
		memmove( (void *)&lp->range[i+1], (void *)&lp->range[i+2],
			 (lp->nRanges - i) * sizeof(RcdRange) );
		memset( (void *)&lp->range[lp->nRanges], 0, sizeof(RcdRange) );
	    }
	    return;
	}
    }
				/* Rcd num is bigger than existing ranges */
    if ( lp->nRanges < NRANGES )
    {
	lp->range[lp->nRanges].min = lp->range[lp->nRanges].max = rcd;
	lp->nRanges++;
    }

} /* gotRcd() */


/************************************************************************/
/* Function    : getMissingRcd						*/
/* Purpose     : Find next set of missing records			*/
/* Inputs      : LogStruct, record number to search from, place to put	*/
/*		 missing record range					*/
/* Outputs     : TRUE if more missing records, FALSE if not		*/
/* Side Effect : If return == TRUE, *rcd is filled in with range of	*/
/*		 missing records					*/
/************************************************************************/
	MBool
getMissingRcd( LogStruct *lp, Nat32 last, RcdRange *rcd )
{
    Reg Int32	i;
    
    for ( i = 0; i < (lp->nRanges - 1); i++ )
    {
	if ( last < lp->range[i].min )
	{
	    rcd->min = last;
	    rcd->max = lp->range[i].min - 1;
	    return( TRUE );
	}

	if ( last <= lp->range[i].max )
	{
	    rcd->min = lp->range[i].max + 1;
	    rcd->max = lp->range[i+1].min - 1;
	    return( TRUE );
	}
    }

    return( FALSE );

} /* getMissingRcd() */
