/****************************************************************************/
/* Copyright 1995 MBARI                                                     */
/****************************************************************************/
/* $Header: no3.c,v 2.5 94/12/15 10:59:40 hebo Exp $			    */
/* Summary  : NO3 decode routines for decode.c, extract.c		    */
/* Filename : no3.c							    */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 2.5 $							    */
/* Created  : 04/18/95							    */
/****************************************************************************/
/* Modification History:						    */
/* 18apr95 rah - created						    */
/* $Log:	no3.c,v $
*/
/****************************************************************************/

#include <stdio.h>			/* Standard I/O			    */
#include <mbari/types.h>		/* MBARI type definitions	    */
#include <mbari/const.h>		/* MBARI constants		    */
#include <decode.h>			/* OASIS controller definitions	    */
#include <time.h>			/* Time				    */
#include <math.h>

/********************************/
/*	External Function	*/
/********************************/

Extern double	ctd_salinity( double c, double t, double p );


/********************************/
/*	Module Local Data	*/
/********************************/

MLocal char	buff[256];		/* Scratch buffer		    */


/************************************************************************/
/* Function    : init_one_cal						*/
/* Purpose     : Initialize one CTD calibration value			*/
/* Inputs      : Ptr to Cal to init					*/
/* Outputs     : None							*/
/************************************************************************/
	Void
init_no3_cal( No3Cal *p )
{
    p->nc_samplechar = 'S';
    p->nc_samplenum = 1;
    p->nc_maxCounts = 3000;
    p->nc_stdConc = 10;
    p->nc_Co = (double)(-0.1);
    p->nc_Cf = (double)72.;
    p->nc_To = (double)(-2.2);
    p->nc_Tf = (double)34.;
    p->nc_pressure = (double)(0.5);

} /* init_no3_cal() */


/************************************************************************/
/* Function    : read_no3_cal						*/
/* Purpose     : Read NO3 calibration file				*/
/* Inputs      : Name of file, place to put calibrations		*/
/* Outputs     : OK or ERROR						*/
/************************************************************************/
	Status
read_no3_cal( char *name, No3Cal *no3cp )
{
    FILE	*fp;

    init_no3_cal( no3cp );

    if ( (fp = fopen(name, "rb")) == (FILE *)NULL )
	return( ERROR );

    while ( fscanf(fp, " %s", buff) != EOF )
    {
	if ( strcmp(buff, "samplechar") == 0 )
	    fscanf( fp, " %c", &no3cp->nc_samplechar );

	else if ( strcmp(buff, "samplenum") == 0 )
	    fscanf( fp, " %d", &no3cp->nc_samplechar );

	else if ( strcmp(buff, "maxCounts") == 0 )
	    fscanf( fp, " %d", &no3cp->nc_maxCounts );

	else if ( strcmp(buff, "stdConc") == 0 )
	    fscanf( fp, " %d", &no3cp->nc_stdConc );

	else if ( strcmp(buff, "pressure") == 0 )
	    fscanf( fp, " %d", &no3cp->nc_pressure );

	else if ( strcmp(buff, "Co") == 0 )
	    fscanf( fp, " %lg", &no3cp->nc_Co );

	else if ( strcmp(buff, "Cf") == 0 )
	    fscanf( fp, " %lg", &no3cp->nc_Cf );

	else if ( strcmp(buff, "To") == 0 )
	    fscanf( fp, " %lg", &no3cp->nc_To );

	else if ( strcmp(buff, "Tf") == 0 )
	    fscanf( fp, " %lg", &no3cp->nc_Tf );

    }

    fclose( fp );
    return( OK );

} /* read_no3_cal() */


/************************************************************************/
/* Function    : decode_no3						*/
/* Purpose     : Decode NO3 information					*/
/* Inputs      : Pointer to CTD data, length, ptr to return struct	*/
/* Outputs     : OK or SIZE_ERR						*/
/************************************************************************/
	Status
decode_no3( Byte *no3data, Int len, No3Decode *no3dp, No3Cal *no3cp )
{
    Int32	nValues;
    Int32	no3Count;
    Int32	Vf, Vo, Vt, Vc;
    Int32	indicator;
    Int32	unusedInt;
    double	unusedFloat;
    double	Ts, Tz, Cs, Cz;
    char	unused[64];

    nValues = sscanf((char *)no3data, " %c %d, %s %s,%d,%d,%lg,%d,%d,%d,%d",
		     &indicator, &no3dp->no3_sample, unused, unused,
		     &no3Count, &unusedInt, &unusedFloat, &Vf, &Vo, &Vt, &Vc);

    if ( nValues < 5 )
	return( SIZE_ERR );

    if ( indicator != no3cp->nc_samplechar )
	return( FMT_ERR );

    no3dp->no3_conc = 0.0;
    no3dp->no3_temp = 0.0;
    no3dp->no3_salinity = 0.0;
    no3dp->no3_conductivity = 0.0;

    no3dp->no3_conc = (no3cp->nc_stdConc/0.345) *
			(-log10(no3Count/no3cp->nc_maxCounts));

    if ( nValues >= 10 )
    {
	Ts = (no3cp->nc_Tf - no3cp->nc_To) / (Vf - Vo);
	Tz = no3cp->nc_Tf - (Ts * Vf);
	no3dp->no3_temp = (Ts * Vf) + Tz;
    }

    if ( nValues >= 11 )
    {
	Cs = (no3cp->nc_Cf - no3cp->nc_Co) / (Vf - Vo);
	Cz = no3cp->nc_Cf - (Cs * Vf);
	no3dp->no3_conductivity = (Cs * Vc) + Cz;
	no3dp->no3_salinity = 
	    ctd_salinity( no3dp->no3_conductivity, no3dp->no3_temp,
			  no3cp->nc_pressure );
    }

    return( OK );

} /* decode_no3() */
