/****************************************************************************/
/* Copyright 1992 MBARI                                                     */
/****************************************************************************/
/* $Header: m2.c,v 1.1 94/01/17 11:09:55 hebo Exp $			    */
/* Summary  : Program to print meteorological readings from OASIS M2 mooring*/
/* Filename : m2.c							    */
/* Author   : Robert Herlien (rah)					    */
/* Project  : OASIS Mooring						    */
/* $Revision: 1.1 $							    */
/* Created  : 09/24/92							    */
/****************************************************************************/
/* Modification History:						    */
/* 10feb92 rah - created						    */
/* $Log:	m2.c,v $
 * Revision 1.1  94/01/17  11:09:55  11:09:55  hebo (Bob Herlien)
 * Misc changes
 * 
 * Revision 1.0  92/09/24  10:29:53  10:29:53  hebo (Bob Herlien)
 * Initial revision
 * 
*/
/****************************************************************************/

#include <stdio.h>			/* Standard I/O			    */
#include <stdlib.h>			/* For exit()			    */
#include <mbari/types.h>		/* MBARI type definitions	    */
#include <mbari/const.h>		/* MBARI constants		    */
#include <decode.h>			/* OASIS controller definitions	    */
#include <time.h>			/* Time				    */
#include <math.h>			/* For fmod()			    */
#include <ctype.h>

#define NAMESIZE	256		/* Space allocated for file names   */
#define ALINELEN	170		/* Approx length of one ATLAS record*/
#define CLINELEN	 60		/* Approx length of one CTD record  */
#define GLINELEN	 50		/* Approx length of one GPS record  */
#define LINEBUFLEN	512		/* Size of line buffer		    */
#define NRECS		150		/* Max # ATLAS records displayed    */
#define DFLT_RECS	20		/* Default # ATLAS records displayed*/
#define NFIELDS		21		/* # fields in ATLAS record	    */
#define MS_TO_KTS	1.9438		/* Conversion of m/s to knots	    */
#define YEAR0		70		/* Unix Epoch year (1970)	    */
#define SECS_PER_DAY	86400		/* Seconds per day		    */
#define START_WINDOW	0.007		/* 10 minute window at start	    */

typedef struct
{
    double	ctd_time;
    CTDDecode	ctdd;
} CTDDec;

typedef struct
{
    double	gps_time;
    double	gps_lat;
    double	gps_long;
    double	gps_std;
    int		gps_samples;
    int		gps_status;
} GPSDec;


/********************************/
/*	Global Data		*/
/********************************/

Global AtlasDecode atlrecs[NRECS];	/* ATLAS records		    */
Global CTDDec	ctdrecs[NRECS];		/* CTD records			    */
Global GPSDec	gpsrecs[NRECS];		/* GPS records			    */
Global char	atlas_file[NAMESIZE];	/* Name of ATLAS data file	    */
Global char	ctd_file[NAMESIZE];	/* Name of CTD data file	    */
Global char	gps_file[NAMESIZE];	/* Name of GPS data file	    */
Global char	*atlas_dir = "/oasis/m2/atlas";
Global char	*ctd_dir = "/oasis/m2/ctd";
Global char	*gps_dir = "/oasis/m2/gps";
Global char	buffer[LINEBUFLEN];

Global char	*months[12] =		/* Month names			    */
{"Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec"};

Global Nat16	jdays[12] =		/* Days since Jan. 1 in non-leap yr */
{ 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334 };


/************************************************************************/
/* Function    : julday							*/
/* Purpose     : Calculate Julian Day given year, month, day		*/
/* Inputs      : year (00 - 99), month (0 - 11), day (1 - 31)		*/
/* Outputs     : Julian day						*/
/* Comment     : Returns Julian day in range 1:366			*/
/************************************************************************/
	Int32
julday( Int32 yr, Int32 mo, Int32 day )
{
    Int32		leap;

    leap = (((yr % 4) == 0) && (mo > 1)) ? 1 : 0;
    return( jdays[mo] + day + leap );

} /* julday() */


/************************************************************************/
/* Function    : ep_days						*/
/* Purpose     : Calculate number days since Unix epoch			*/
/* Inputs      : years since 1970, Unix-type julian day (0:365)		*/
/* Outputs     : Days since 1/1/1970					*/
/* Comment     : The (year + 1)/4 term accounts for leap years, the	*/
/*		 first of which was 1972 & should be added starting '73	*/
/************************************************************************/
	Int32
ep_days( year, yday )
     int	year, yday;
{
    return( (365 * year) + ((year + 1)/4) + yday );

} /* ep_days() */


/************************************************************************/
/* function    : main							*/
/* purpose     : Main routine						*/
/* inputs      : argc, argv						*/
/* outputs     : none							*/
/************************************************************************/
	Void
main( int argc, char **argv )
{
    time_t	curtime, basetime, atm, ctm, gtm;
    struct tm	*tp;
    Int		i, j, k, itime, arecs, crecs, grecs, n, year, mo, yday, mday;
    FILE	*afp, *cfp, *gfp;
    double	otime, starttime;
    Int		lat, lng;

    time( &curtime );
    tp = gmtime( &curtime );
    year = tp->tm_year;
    itime = (1000 * year) + tp->tm_yday + 1;
    arecs = DFLT_RECS;
    afp = cfp = NULLF;

    for ( i = 1; i < argc; i++ )
    {
	if ( *argv[i] == '-' )
	    switch( argv[i][1] )
	    {
	      case 'd':
		if ( isdigit(argv[i][2]) )
		    itime = atoi( &argv[i][2] );
		else
		    itime = atoi( argv[++i] );
		if ( itime < 1000 )
		    itime += year * 1000;
		break;
	    
	      case 'f':
		i++;
		atlas_dir = argv[i];
		break;

	      case 'n':
		if ( isdigit(argv[i][2]) )
		    arecs = atoi( &argv[i][2] );
		else
		    arecs = atoi( argv[++i] );
		if ( arecs > NRECS )
		    arecs = NRECS;
		break;

	      default:
		printf("Unknown command line switch \"%s\"\n", argv[i]);
	    }
    }

    for ( i = itime; itime > i - 10; itime-- )
    {
	sprintf( atlas_file, "%s/%05d", atlas_dir, itime );
	sprintf( ctd_file, "%s/%05d", ctd_dir, itime );
	sprintf( gps_file, "%s/%05d", gps_dir, itime );
	afp = fopen( atlas_file, "r" );
	cfp = fopen( ctd_file, "r" );
	gfp = fopen( gps_file, "r" );
	if ( afp != NULLF )
	    break;
    }

    crecs = arecs;
    grecs = arecs/3 + 1;
    starttime = 0.0;

    if ( afp == NULLF )
    {
	printf( "Could not open ATLAS file %s/%05d\n", atlas_dir, i );
	arecs = 0;
    }

    if ( cfp == NULLF )
    {
	printf( "Could not open CTD file %s/%05d\n", ctd_dir, i );
	crecs = 0;
    }

    if ( gfp == NULLF )
    {
	printf( "Could not open GPS file %s/%05d\n", gps_dir, i );
	grecs = 0;
    }

    if ( afp != NULLF )
    {
	fseek( afp, (long)arecs * ALINELEN * (-1), SEEK_END );
	fgets(buffer, LINEBUFLEN, afp);
	for ( i = 0; fgets(buffer, LINEBUFLEN, afp) != NULL; )
	{
	    if ( buffer[0] == '#' )
		continue;
	    if ( sscanf(buffer, " %lg %lg %lg %lg %lg %lg %lg %lg %lg %lg %lg \
			%lg %lg %lg %lg %lg %lg %lg %lg %lg %lg", 
		   &otime, &atlrecs[i].atd_time, &atlrecs[i].atd_air,
		   &atlrecs[i].atd_sst, &atlrecs[i].atd_temp[0],
		   &atlrecs[i].atd_temp[1], &atlrecs[i].atd_temp[2],
		   &atlrecs[i].atd_temp[3], &atlrecs[i].atd_temp[4],
		   &atlrecs[i].atd_temp[5], &atlrecs[i].atd_temp[6],
		   &atlrecs[i].atd_temp[7], &atlrecs[i].atd_temp[8],
		   &atlrecs[i].atd_temp[9], &atlrecs[i].atd_press[0],
		   &atlrecs[i].atd_press[1], &atlrecs[i].atd_rh,
		   &atlrecs[i].atd_windspd, &atlrecs[i].atd_winddir,
		   &atlrecs[i].atd_windu,   &atlrecs[i].atd_windv)
		>= NFIELDS )
		atlrecs[i++].atd_time = otime;
	}
	fclose( afp );
	arecs = i;
    }

    if ( arecs <= 0 )
	printf( "No ATLAS data in file %s\n", atlas_file );
    else
	starttime = atlrecs[0].atd_time;

    if ( cfp != NULLF )
    {
	for ( i = 0; fgets(buffer, LINEBUFLEN, cfp) != NULL; )
	{
	    if ( buffer[0] == '#' )
		continue;
	    if ( sscanf(buffer, " %lg %d %lg %lg %lg %lg %lg %lg",
			&ctdrecs[i].ctd_time, &ctdrecs[i].ctdd.ctd_sample,
			&ctdrecs[i].ctdd.ctd_press, &ctdrecs[i].ctdd.ctd_temp,
			&ctdrecs[i].ctdd.ctd_cond, &ctdrecs[i].ctdd.ctd_sal,
			&ctdrecs[i].ctdd.ctd_trans, &ctdrecs[i].ctdd.ctd_fluor)
		>= 8 )
		if ( ctdrecs[i].ctd_time > starttime - START_WINDOW )
		    i++;
	}
	fclose( cfp );
	crecs = i;
    }

    if ( crecs <= 0 )
	printf( "No CTD data in file %s\n", ctd_file );

    if ( gfp != NULLF )
    {
	for ( i = 0; fgets(buffer, LINEBUFLEN, gfp) != NULL; )
	{
	    if ( buffer[0] == '#' )
		continue;
	    if ( sscanf(buffer, " %lg %lg %lg %lg %d %d", 
			&gpsrecs[i].gps_time, &gpsrecs[i].gps_lat, 
			&gpsrecs[i].gps_long, &gpsrecs[i].gps_std,
			&gpsrecs[i].gps_samples, &gpsrecs[i].gps_status) == 6 )
		if ( gpsrecs[i].gps_time > starttime - START_WINDOW )
		    i++;
	}
	fclose( gfp );
	grecs = i;
    }

    if ( grecs <= 0 )
	printf( "No GPS data in file %s\n", gps_file );

    yday = itime % 1000;
    for ( mo = 0; (yday > julday(year,mo+1,0)) && (mo < 11); mo++ )
	;				/* Loop to find month		*/

    mday = yday - julday(year,mo,0);	/* Find day of month		*/
		/* Get time_t for 00:00.  The +30 is for rounding minutes*/
    basetime = ep_days(year - YEAR0, yday) * SECS_PER_DAY + 30;

    printf("\nLatest M2 ATLAS and CTD data from %s %d, %d (Julian day %d)\n\n", 
	   months[mo], mday, year + 1900, yday );
    printf("Local  Air   ATLAS   ATLAS   Rel    Wind    CTD  Salinity      Position\n");
    printf("Time   Temp   SST     20m   Humid kts from Temp\n");

    for ( i = j = k = 0; (i < arecs); )
    {
	atm = basetime + (SECS_PER_DAY * fmod(atlrecs[i].atd_time, 1.0));
	if ( j < crecs )
	    ctm = basetime + (SECS_PER_DAY * fmod(ctdrecs[j].ctd_time, 1.0));
	else
	    ctm = (time_t)0x7fffffff;
	if ( k < grecs )
	    gtm = basetime + (SECS_PER_DAY * fmod(gpsrecs[k].gps_time, 1.0));
	else
	    gtm = (time_t)0x7fffffff;
	tp = localtime(&atm);
	n = (Int)atlrecs[i].atd_winddir + 180;
	if ( n >= 360 )
	    n -= 360;

	if ( abs(ctm - atm) <= 300 )
	{
	    printf("%2d:%02d ", tp->tm_hour, tp->tm_min);
	    printf("%5.1f %7.3f %7.3f %3.0f  %4.1f %3d", 
		   atlrecs[i].atd_air, atlrecs[i].atd_sst, 
		   atlrecs[i].atd_temp[1], atlrecs[i].atd_rh,
		   MS_TO_KTS * atlrecs[i].atd_windspd, n);
	    i++;
	    printf(" %6.3f %7.4f", 
		   ctdrecs[j].ctdd.ctd_temp, ctdrecs[j].ctdd.ctd_sal );
	    j++;
	}
	else if ( atm <= ctm )
	{
	    printf("%2d:%02d ", tp->tm_hour, tp->tm_min);
	    printf("%5.1f %7.3f %7.3f %3.0f  %4.1f %3d               ", 
		   atlrecs[i].atd_air, atlrecs[i].atd_sst, 
		   atlrecs[i].atd_temp[1], atlrecs[i].atd_rh,
		   MS_TO_KTS * atlrecs[i].atd_windspd, n);
	    i++;
	}
	else
	{
	    tp = localtime(&ctm);
	    printf("%2d:%02d                                     ", 
		   tp->tm_hour, tp->tm_min);
	    printf("%6.3f %7.4f", 
		   ctdrecs[j].ctdd.ctd_temp, ctdrecs[j].ctdd.ctd_sal );
	    j++;
	}

	if ( gtm <= atm + 300 )
	{
	    lat = labs((Int)gpsrecs[k].gps_lat);
	    lng = labs((Int)gpsrecs[k].gps_long);
	    printf( " %d:%06.3fN %d:%06.3fW", 
		   lat, (fabs(gpsrecs[k].gps_lat) - (double)lat) * 60.0,
		   lng, (fabs(gpsrecs[k].gps_long) - (double)lng) * 60.0 );
	    k++;
	}
	printf("\n");
    }

    exit( 0 );

} /* main() */
